package ch.sahits.game.openpatrician.engine.time;

import ch.sahits.game.event.data.ClockTick;
import ch.sahits.game.event.data.ClockTickIntervalChange;
import ch.sahits.game.event.data.PeriodicalTimeDayUpdate;
import ch.sahits.game.event.data.PeriodicalTimeMonthEndUpdate;
import ch.sahits.game.event.data.PeriodicalTimeMonthUpdate;
import ch.sahits.game.event.data.PeriodicalTimeWeekEndUpdate;
import ch.sahits.game.event.data.PeriodicalTimeWeekUpdate;
import ch.sahits.game.event.data.PeriodicalTimeYearEndUpdate;
import ch.sahits.game.event.data.PeriodicalTimeYearUpdate;
import ch.sahits.game.openpatrician.annotation.ClassCategory;
import ch.sahits.game.openpatrician.annotation.EClassCategory;
import ch.sahits.game.openpatrician.model.Date;
import com.google.common.base.Preconditions;
import com.google.common.eventbus.AsyncEventBus;
import com.google.common.eventbus.Subscribe;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.joda.time.DateTime;
import org.joda.time.DateTimeConstants;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;

import javax.annotation.PostConstruct;
import javax.annotation.PreDestroy;

/**
 * The periodical time updater encapsules clock ticks of instances that are interested in
 * notification but only on a periodical level. The period may vary but the implementation is
 * always the same. The periods as defined in {@link EUpdateIntervalRegistration} are possible,
 * where as a year is composed by 365 days and a month has always 30 days. So these periods are
 * not really exact.
 * @author Andi Hotz, (c) Sahits GmbH, 2012
 * Created on Jul 22, 2012
 *
 */
@ClassCategory(EClassCategory.SINGLETON_BEAN)
public class PeriodicalTimeUpdaterV2 {
    private final Logger logger = LogManager.getLogger(getClass());
	/** Period of the clock ticking */
	private int tickInterval;
    @Autowired
	private Date date;
	/** Period that the listener should be notified */
	private int updateInterval;
    private final EUpdateIntervalRegistration intervall;
    /** Time passed since the start of the period */
    private int timePassed = 0;
    @Autowired
    @Qualifier("serverClientEventBus")
    private AsyncEventBus clientServerEventBus;
    @Autowired
    @Qualifier("timerEventBus")
    private AsyncEventBus timerEventBus;

    public PeriodicalTimeUpdaterV2(EUpdateIntervalRegistration intervall){
        Preconditions.checkArgument(intervall != EUpdateIntervalRegistration.DAILY, "This updater does not handle daily events");
        this.intervall = intervall;
    }
    @PostConstruct
    public void initializeEventBus() {
        tickInterval = date.getTickUpdate();
        final DateTime cal = date.getCurrentDate();
        switch (intervall) {
            case YEAR:
                updateInterval = 24*60*365;
                timePassed = (cal.getYear()-1)*24*60;
                timePassed += cal.getHourOfDay()*60;
                timePassed += cal.getMinuteOfDay();
                break;
            case MONTH:
                updateInterval = 24*60*30;
                timePassed = (cal.getDayOfMonth()-1)*24*60;
                timePassed += cal.getHourOfDay()*60;
                timePassed += cal.getMinuteOfDay();
                break;
            case WEEK:
                updateInterval = 7*24*60;
                timePassed = cal.getDayOfWeek()*24*60;
                timePassed += cal.getHourOfDay()*60;
                timePassed += cal.getMinuteOfDay();
                break;
            case END_OF_DAY:
                updateInterval = 24*60;
                timePassed += cal.getHourOfDay()*60;
                timePassed += cal.getMinuteOfDay();
                break;
            case END_OF_MONTH:
                updateInterval = 24*60*cal.dayOfMonth().getMaximumValue();
                timePassed = (cal.getDayOfMonth()-1)*24*60;
                timePassed += cal.getHourOfDay()*60;
                timePassed += cal.getMinuteOfDay();
                break;
            case END_OF_YEAR:
                updateInterval = 24*60*cal.dayOfYear().getMaximumValue();
                timePassed = (cal.getDayOfYear()-1)*24*60;
                timePassed += cal.getHourOfDay()*60;
                timePassed += cal.getMinuteOfDay();
                break;
            case END_OF_WEEK:
                updateInterval = 24*60*cal.dayOfWeek().getMaximumValue();
                timePassed = (cal.getDayOfWeek()-1)*24*60;
                timePassed += cal.getHourOfDay()*60;
                timePassed += cal.getMinuteOfDay();
                break;
            default:
                throw new IllegalArgumentException("Unknown update interval: "+intervall);
        }
        clientServerEventBus.register(this);
        timerEventBus.register(this);
    }


	@Subscribe
	public void updateTick(ClockTickIntervalChange event ) {
		tickInterval = event.getInterval();
	}


    @Subscribe
    public void handleClockTick(ClockTick event) {
        timePassed += tickInterval;
        if (timePassed>updateInterval){
            switch (intervall) {
                case YEAR:
                    clientServerEventBus.post(new PeriodicalTimeYearUpdate());
                    break;
                case MONTH:
                    clientServerEventBus.post(new PeriodicalTimeMonthUpdate());
                    break;
                case WEEK:
                    clientServerEventBus.post(new PeriodicalTimeWeekUpdate());
                    break;
                case END_OF_DAY:
                    clientServerEventBus.post(new PeriodicalTimeDayUpdate());
                    int dayOfWeek = date.getCurrentDate().getDayOfWeek();
                    if (dayOfWeek == DateTimeConstants.SUNDAY) {
                        clientServerEventBus.post(new PeriodicalTimeWeekEndUpdate());
                    }
                    int monthOfYear = date.getCurrentDate().getMonthOfYear();
                    int dayOfMonth = date.getCurrentDate().getDayOfMonth();
                    if (monthOfYear == DateTimeConstants.DECEMBER && dayOfMonth == 31) {
                        clientServerEventBus.post(new PeriodicalTimeYearEndUpdate());
                    }
                    DateTime last = date.getCurrentDate().dayOfMonth().withMaximumValue();
                    if (dayOfMonth == last.getDayOfMonth()) {
                        clientServerEventBus.post(new PeriodicalTimeMonthEndUpdate());
                    }
                    break;
                case END_OF_YEAR:
                case END_OF_MONTH:
                case END_OF_WEEK:
                    break;
                default:
                    throw new IllegalArgumentException("Unknown update interval: "+intervall);                }
            timePassed=0;
        }

    }
    @PreDestroy
    public void unregisterFromEventBus() {
        clientServerEventBus.unregister(this);
        timerEventBus.unregister(this);
    }

}
