package ch.sahits.game.openpatrician.engine;

import ch.sahits.game.event.EGameStatusChange;
import ch.sahits.game.event.GameStateChange;
import ch.sahits.game.event.data.BuildingAuctionFinished;
import ch.sahits.game.event.data.BuildingNotAuctioned;
import ch.sahits.game.event.data.GraphInitialisationComplete;
import ch.sahits.game.event.data.PeriodicalTimeWeekEndUpdate;
import ch.sahits.game.event.data.ShipAuctionFinished;
import ch.sahits.game.event.data.ShipEntersPortEvent;
import ch.sahits.game.event.data.ShipNotAuctioned;
import ch.sahits.game.event.data.ShipyardOrderBuild;
import ch.sahits.game.event.data.ai.BlockadeShipRequestEvent;
import ch.sahits.game.event.data.ai.SpecialMissionFinishedEvent;
import ch.sahits.game.openpatrician.annotation.*;
import ch.sahits.game.openpatrician.clientserverinterface.model.event.MarriageBrokerAnnouncementState;
import ch.sahits.game.openpatrician.clientserverinterface.service.MapService;
import ch.sahits.game.openpatrician.clientserverinterface.service.StrategyHolderService;
import ch.sahits.game.openpatrician.engine.event.AIPlayerEngineState;
import ch.sahits.game.openpatrician.engine.event.EEventState;
import ch.sahits.game.openpatrician.engine.event.EventEngineState;
import ch.sahits.game.openpatrician.engine.event.task.MarriageOfferTask;
import ch.sahits.game.openpatrician.engine.event.task.ServerSideTaskFactory;
import ch.sahits.game.openpatrician.engine.land.city.ShipyardEngine;
import ch.sahits.game.openpatrician.engine.player.AITradeChecker;
import ch.sahits.game.openpatrician.engine.player.CollectConstructionWaresMissionData;
import ch.sahits.game.openpatrician.engine.player.IAIEventHandler;
import ch.sahits.game.openpatrician.engine.player.strategy.BlockadeStrategy;
import ch.sahits.game.openpatrician.engine.sea.SeafaringService;
import ch.sahits.game.openpatrician.model.AIPlayerList;
import ch.sahits.game.openpatrician.model.Date;
import ch.sahits.game.openpatrician.model.IAIPlayer;
import ch.sahits.game.openpatrician.model.IPlayer;
import ch.sahits.game.openpatrician.model.city.ICity;
import ch.sahits.game.openpatrician.model.city.IShipyard;
import ch.sahits.game.openpatrician.model.event.TimedTask;
import ch.sahits.game.openpatrician.model.event.TimedUpdatableTaskList;
import ch.sahits.game.openpatrician.model.people.IShipOwner;
import ch.sahits.game.openpatrician.model.player.IAIConstructionSelectionStrategy;
import ch.sahits.game.openpatrician.model.player.IAIEventDecisionStrategy;
import ch.sahits.game.openpatrician.model.player.IAITradeStrategy;
import ch.sahits.game.openpatrician.model.player.IAITradeStrategyType;
import ch.sahits.game.openpatrician.model.player.IProductionConsumptionKnowledge;
import ch.sahits.game.openpatrician.model.product.IBlockadeMission;
import ch.sahits.game.openpatrician.model.product.ISpecialMission;
import ch.sahits.game.openpatrician.model.product.ITradeMissionData;
import ch.sahits.game.openpatrician.model.sea.BlockadeState;
import ch.sahits.game.openpatrician.model.sea.IBlockade;
import ch.sahits.game.openpatrician.model.sea.TravellingVessels;
import ch.sahits.game.openpatrician.model.ship.EShipType;
import ch.sahits.game.openpatrician.model.ship.INavigableVessel;
import ch.sahits.game.openpatrician.model.ship.IShip;
import ch.sahits.game.openpatrician.util.service.DisableProperties;
import com.google.common.eventbus.AsyncEventBus;
import com.google.common.eventbus.Subscribe;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.joda.time.DateTime;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;

import javax.annotation.PostConstruct;
import java.util.*;
import java.util.Map.Entry;
import java.util.concurrent.ConcurrentLinkedQueue;

import static java.util.Collections.emptyList;

/**
 * This implements the engine for an AI Player
 * @author Andi Hotz, (c) Sahits GmbH, 2014
 *         Created on Jan 23, 2014
 */
@LazySingleton
@ClassCategory(EClassCategory.SINGLETON_BEAN)
public class AIPlayerEngine extends AbstractEngine implements IAIEventHandler {
    private final Logger logger = LogManager.getLogger(getClass());
    @Autowired
    private AIPlayerList players;
    @Autowired
    @Qualifier("serverClientEventBus")
    private AsyncEventBus clientServerEventBus;
    @Autowired
    @Qualifier("timerEventBus")
    private AsyncEventBus timerEventBus;
    @Autowired
    private BlockadeState blockadeState;
    @Autowired
    private BlockadeStrategy blockadeStrategy;
    @Autowired
    private SeafaringService seafaringService;
    @Autowired
    private Date date;
    @Autowired
    private MapService mapService;
    @Autowired
    private TravellingVessels vessels;
    @Autowired
    private AITradeChecker tradeChecker;
    @Autowired
    private ShipyardEngine shipyardEngine;
    @Autowired
    private TimedUpdatableTaskList taskList;
    @Autowired
    private ServerSideTaskFactory taskFactory;
    @Autowired
    private Random rnd;
    @Autowired
    private StrategyHolderService strategyHolderService;
    @Autowired
    private DisableProperties disableService;
    @Autowired
    private EventEngineState eventState;
    @Autowired
    private AIPlayerEngineState engineState;
    /** Queue for ship enters port events. If the list is null, this means the events can be handled directly. */
    @ListType(ShipEntersPortEvent.class)
    private Queue<ShipEntersPortEvent> shipEntersPortEventQueue = new ConcurrentLinkedQueue<>();

    @Override
    public List<AbstractEngine> getChildren() {
        return emptyList();
    }
    @PostConstruct
    private void init() {
        clientServerEventBus.register(this);
        timerEventBus.register(this);
    }

    private void initializeStrategyForVessel(IAIPlayer player, INavigableVessel ship) {
        IAITradeStrategy tradeStrategy = player.getTradeStrategyType(ship).getStrategy();
        tradeStrategy.initialzeTradeCycle(player, ship);
        player.updateTradeWaitingStatus(ship, false);
    }

    @Subscribe
    public void handleBlockadeStart(BlockadeShipRequestEvent event) {
        IAIPlayer player = event.getPlayer();
        int requiredShips = event.getNumberOfShips();
        for (int i = 0; i < requiredShips; i++) {
            Optional<IShip> optShip = blockadeStrategy.selectShip(player);
            if (optShip.isPresent()) {
                IShip vessel = optShip.get();
                player.setTradeMission(vessel, new IBlockadeMission() {});
                seafaringService.travelTo(vessel, event.getAssemblyCity().getCoordinates());
            } else {
                break;
            }
        }
    }
    @Subscribe
    public void handleShipArrivesInCity(ShipEntersPortEvent event) {
        if (shipEntersPortEventQueue != null && !shipEntersPortEventQueue.isEmpty()) {
            shipEntersPortEventQueue.add(event);
            logger.info("Add ShipEntersPortEvent to queue.");
        } else {
            handleShipArrival(event);
        }
    }

    /**
     * Internal handling of the ship arrival.
     * @param event
     */
    private void handleShipArrival(ShipEntersPortEvent event) {
        final INavigableVessel vessel = event.getShip();
        final ICity city = event.getCity();
        if (!vessel.getLocation().equals(city.getCoordinates())) {
            logger.warn("The vessel is not in the city, but is moving, this can happen when a new destination is chosen after the ShipPositionUpdate task determined the ship entered the port, but the event was not yet handled.");
            return;
        }
        IShipOwner owner = vessel.getOwner();
        logger.debug("Current position of vessel ({}): {}", vessel.getName(), vessel.getLocation());
        vessels.remove(vessel);
        if (owner instanceof IAIPlayer) {
            IAIPlayer player = (IAIPlayer) owner;
            IProductionConsumptionKnowledge knowlege = player.getProductionAndConsumptionKnowledge();
            knowlege.updateKnowledge(event.getCity());
            ITradeMissionData missionData = player.getTradeMission(vessel);
            if (missionData instanceof ISpecialMission) {
                if (missionData instanceof IBlockadeMission) {
                    handleBlockadeMission(vessel, player, event.getCity());
                } else {
                    throw new IllegalStateException("The special mission "+missionData+" is not implemented");
                }
            } else {
                player.updateTradeWaitingStatus(vessel, false);
                try {
                    IAITradeStrategy strategy = player.getTradeStrategyType(vessel).getStrategy();
                    engineState.activateVesselForTrade(vessel);
                    strategy.handleShipArrivesInPort(event);
                } catch (RuntimeException e) {
                    logger.error("Failed to handle ship arrival of "+vessel.getName()+" of "+player.getName()+" "+player.getLastName(), e);
                } finally {
                    engineState.deactivateVesselForTrade(vessel);
                }
            }
        }
    }

    /**
     * Listen to the start of the game loading to continue with the handling of ship arrival events.
     * @param gameStateChange
     */
    @Subscribe
    public void handleGameLoad(GameStateChange gameStateChange) {
        if (gameStateChange.getStatusChange() == EGameStatusChange.LOAD_GAME_INIT) {
            shipEntersPortEventQueue = new ConcurrentLinkedQueue<>();
            logger.info("Loading of a game was initialized. Queuing further ShipEntersPortEvent.");
        }
    }
    @Subscribe
    public void handleTradeStrategyStart(GraphInitialisationComplete event) {
        if (shipEntersPortEventQueue != null) {
            logger.info("Handle {} queued ShipEntersPortEvent after graph initialisation", shipEntersPortEventQueue.size());
            while (!shipEntersPortEventQueue.isEmpty()) {
                ShipEntersPortEvent shipEntersPortEvent = shipEntersPortEventQueue.poll();
                handleShipArrival(shipEntersPortEvent);
            }
            shipEntersPortEventQueue = null;
        }
        // When loading a game from in game flag must be set to false, if the map is a different one.
        if (disableService.aiTradingEnabled()) {
            for (IAIPlayer player : players) {
                ICity hometown = player.getHometown();
                IProductionConsumptionKnowledge knowlege = player.getProductionAndConsumptionKnowledge();
                knowlege.updateKnowledge(hometown);
                List<IShip> shipList = player.getFleet();
                for (IShip ship : shipList) {
                    try {
                        // Only initialize if there are no trade steps (not after loading)
                        if (!player.isInitialized(ship)) {
                            initializeStrategyForVessel(player, ship);
                        }
                    } catch (RuntimeException e) {
                        logger.error("Failed to initialize strategy for ship  " + ship.getName() + " of " + player.getName() + " " + player.getLastName(), e);
                    }
                }

                List<INavigableVessel> vesselList = player.getSelectableVessels();
                for (INavigableVessel ship : vesselList) {
                    startTradeStrategySteps(player, ship);
                }
            }
        }
    }

    private void startTradeStrategySteps(IAIPlayer player, INavigableVessel ship) {
        IAITradeStrategy tradeStrategy = player.getTradeStrategyType(ship).getStrategy();
        // If the vessel is traveling do not execute
        if (!player.waitingForTradeStepToFinish(ship)) {
            tradeStrategy.executeTradeSteps(player, ship);
        }  else {
            logger.warn("Skip starting of trade execution as waitingForTradeStepToFinish for {} of {} {} is {}",ship.getName(), player.getName(), player.getLastName(), player.waitingForTradeStepToFinish(ship));
        }
    }

    private void handleBlockadeMission(INavigableVessel vessel, IAIPlayer player, ICity city) {
        DateTime now = date.getCurrentDate();
        ICity aldermanCity = mapService.getAldermanCity();
        if (city.equals(aldermanCity)) {
            if (blockadeState.entrySet().isEmpty()) {
                initializeStrategyForVessel(player, vessel);
            } else {
                for (Entry<ICity, IBlockade> entry : blockadeState.entrySet()) {
                    IBlockade blockade = entry.getValue();
                    if (blockade.getAssemblyDate().isBefore(now)) {
                        // Missed assemby date
                        initializeStrategyForVessel(player, vessel);
                    }
                }
            }
        } else {
            initializeStrategyForVessel(player, vessel);
        }
    }

    /**
     * Check weather the current vessel is engaged in executing trade steps.
     * @param vessel
     * @return
     */
    public boolean executeCurrentTrade(INavigableVessel vessel) {
        return engineState.isActive(vessel);
    }
    @Subscribe
    public void handleWeeklyTasks(PeriodicalTimeWeekEndUpdate event) {
        if (disableService.aiTradingEnabled()) {
            for (IAIPlayer player : players) {
                List<INavigableVessel> shipList = player.getSelectableVessels();
                for (INavigableVessel vessel : shipList) {
                    if (!tradeChecker.isEngagedInTrading(vessel) && tradeChecker.shouldBeTrading(vessel)) {
                        IAIPlayer owner = (IAIPlayer) vessel.getOwner();
                        logger.warn("The vessel {} of {} {} is currently not trading (position: {}), strategy={}: reinitialize", vessel.getName(), owner.getName(), owner.getLastName(), vessel.getLocation(), owner.getTradeStrategyType(vessel));
                        clearRemainingTradeSteps(vessel, player);
                        initializeStrategyForVessel(player, vessel);
                        // Execute
                        IAITradeStrategy tradeStrategy = player.getTradeStrategyType(vessel).getStrategy();
                        tradeStrategy.executeTradeSteps(player, vessel);
                    }
                }
                // Check Ship construction
                IAIConstructionSelectionStrategy constructionStategy = player.getConstructionSelectionType().getStrategy();
                if (constructionStategy.shouldOrderNewConstruction(player)) {
                    EShipType type = constructionStategy.shouldBuildShipType(player);
                    ICity buildLocation = constructionStategy.getBuildLocation(player);
                    constructionStategy.initShipConstruction(player, buildLocation, type);
                }
            }
        }
    }
    /**
     * Remove any remaining trade steps.
     * @param vessel for which the trade steps should be checked
     * @param player for which the trade steps should be checked.
     */
    private void clearRemainingTradeSteps(INavigableVessel vessel, IAIPlayer player) {
        while(player.hasMoreTradeSteps(vessel)) {
            player.getNextTradeStep(vessel);
        }
        player.setTradeMission(vessel, null);
    }

    /**
     * Handle the reinitialisation after finishing a special mission.
     * @param event
     */
    @Subscribe
    public void handleSpecialMissionFinished(SpecialMissionFinishedEvent event) {
        INavigableVessel vessel = event.getVessel();
        IAIPlayer player = (IAIPlayer) vessel.getOwner();
        ICity city = event.getCity();
        logger.debug("Special mission for ship {} of {} {} ended in {}",vessel.getName(), player.getName(), player.getLastName(), city.getName());
        ITradeMissionData missionData = event.getMissionData();
        if (missionData instanceof CollectConstructionWaresMissionData) {
            IShipyard shipyard = city.getCityState().getShipyardState();
            ShipyardOrderBuild shipConstructionOrder = new ShipyardOrderBuild(shipyard, ((CollectConstructionWaresMissionData) missionData).getShipType(), player);
            shipyardEngine.handleOrderBuildEvent(shipConstructionOrder);
        }
        initializeRandomTradeStrategy(vessel, player);
        initializeStrategyForVessel(player, vessel);
        // Execute
        IAITradeStrategy tradeStrategy = player.getTradeStrategyType(vessel).getStrategy();
        tradeStrategy.executeTradeSteps(player, vessel);
    }

    private void initializeRandomTradeStrategy(INavigableVessel vessel, IAIPlayer player) {
        List<IAITradeStrategyType> tradeStrategies = strategyHolderService.getTradeStrategies();
        player.setTradeStrategyType(vessel, tradeStrategies.get(rnd.nextInt(tradeStrategies.size())));
    }

    @Subscribe
    public void handleShipAuctionFinished(ShipAuctionFinished event) {
        IPlayer owner = event.getNewOwner();
        if (owner instanceof IAIPlayer) {
            IShip ship = event.getShip();
            ship.setOwner(owner);
            owner.addShip(ship);
            initializeRandomTradeStrategy(ship, (IAIPlayer) owner);
            initializeStrategyForVessel((IAIPlayer) owner, ship);
            startTradeStrategySteps((IAIPlayer) owner, ship);
        }
    }
    @Subscribe
    public void handleShipNotAuction(ShipNotAuctioned event) {
        IShip ship = event.getShip();
        IPlayer owner = (IPlayer) ship.getOwner();
        if (owner instanceof IAIPlayer) {
            owner.addShip(ship);
            initializeRandomTradeStrategy(ship, (IAIPlayer) owner);
            initializeStrategyForVessel((IAIPlayer) owner, ship);
            startTradeStrategySteps((IAIPlayer) owner, ship);
        }
    }
    @Subscribe
    public void handleBuildingAuctionFinished(BuildingAuctionFinished event) {
        if (event.getNewOwner() instanceof IAIPlayer) {
            event.getBuilding().setOwner(event.getNewOwner());
        }
    }
    @Subscribe
    public void handleBuildingNotAuction(BuildingNotAuctioned event) {
        // no action
    }

    @Override
    public void handleMarriageEvent(IAIPlayer player, MarriageBrokerAnnouncementState state) {
        IAIEventDecisionStrategy strategy = player.getEventDecitionStrategyType().getStrategy();
        boolean meetBroker = rnd.nextInt(12) < 2;
        if (!player.getSpouseData().isPresent()  && meetBroker && strategy.acceptMarriagBrokereOffer(player)) {
            int delay = rnd.nextInt(50) + 24;
            DateTime deadLine = state.getDate().plusDays(delay);
            MarriageOfferTask task = taskFactory.getMarriageOfferTask(player, state, deadLine);
            logger.debug("Add marriage offer task for {} {}", player.getName(), player.getLastName());
            taskList.add(task);
            eventState.setMarriageState(player, EEventState.MARRIAGE_UNDER_CONSIDERATION);
        }
    }


    /**
     * Method to handle the finishing of a ship construction.
     * @param ship that is newly constructed
     * @param player owner of the ship
     */
    public void handleShipConstructionFinished(IShip ship, IAIPlayer player) {
        player.addShip(ship);
        initializeNewShip(ship, player);
    }

    /**
     * Method to handle the initializing of a trade strategy for a new ship.
     * @param ship
     * @param player
     */
    @Override
    public void initializeNewShip(IShip ship, IAIPlayer player) {
        initializeRandomTradeStrategy(ship, player);
        initializeStrategyForVessel(player, ship);
        startTradeStrategySteps(player, ship);
    }

    // TODO ahotz 21.05.2016: When ship leaves convoy add trade steps for it
    // TODO ahotz 21.05.2016: When ship joins convoy remove trade steps for the ship
    // TODO: andi 5/22/16 When ship is captured or destroyed 

}
