package ch.sahits.game.openpatrician.engine.land.city;

import ch.sahits.game.event.data.ClockTick;
import ch.sahits.game.event.data.ClockTickIntervalChange;
import ch.sahits.game.openpatrician.annotation.ClassCategory;
import ch.sahits.game.openpatrician.annotation.DependentInitialisation;
import ch.sahits.game.openpatrician.annotation.EClassCategory;
import ch.sahits.game.openpatrician.clientserverinterface.model.factory.GameFactory;
import ch.sahits.game.openpatrician.clientserverinterface.model.factory.PlayerInteractionFactory;
import ch.sahits.game.openpatrician.clientserverinterface.service.CityProductionAndConsumptionService;
import ch.sahits.game.openpatrician.engine.AbstractEngine;
import ch.sahits.game.openpatrician.engine.event.task.ServerSideTaskFactory;
import ch.sahits.game.openpatrician.engine.player.PlayerEngine;
import ch.sahits.game.openpatrician.model.Date;
import ch.sahits.game.openpatrician.model.IBalanceSheet;
import ch.sahits.game.openpatrician.model.IGame;
import ch.sahits.game.openpatrician.model.building.ITradingOffice;
import ch.sahits.game.openpatrician.model.city.CityProductionStorage;
import ch.sahits.game.openpatrician.model.city.ICity;
import ch.sahits.game.openpatrician.model.city.impl.CitiesState;
import ch.sahits.game.openpatrician.model.city.impl.CityState;
import ch.sahits.game.openpatrician.model.event.FoundingCityBecomesAccessible;
import ch.sahits.game.openpatrician.model.event.TimedTask;
import ch.sahits.game.openpatrician.model.event.TimedUpdatableTaskList;
import ch.sahits.game.openpatrician.model.impl.BalanceSheet;
import ch.sahits.game.openpatrician.spring.DependentAnnotationConfigApplicationContext;
import ch.sahits.game.openpatrician.util.MapInitializedBean;
import ch.sahits.game.openpatrician.util.javafx.IJavaFXApplicationThreadExecution;
import com.google.common.annotations.VisibleForTesting;
import com.google.common.eventbus.AsyncEventBus;
import com.google.common.eventbus.Subscribe;
import com.thoughtworks.xstream.annotations.XStreamOmitField;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.joda.time.DateTime;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.context.ApplicationContext;
import org.springframework.context.annotation.Lazy;
import org.springframework.stereotype.Component;

import javax.annotation.PostConstruct;
import javax.annotation.PreDestroy;
import java.util.ArrayList;
import java.util.List;
import java.util.Optional;
import java.util.Random;

import static com.google.common.collect.Lists.newArrayList;

/**
 * The CityEngine drives/generates the events that are based in the city.
 * Mainly these are:
 * <ul>
 * <li>Consume of wares</li>
 * <li>Production of wares from workshops owned by the city</li>
 * </ul>
 * @author Andi Hotz, (c) Sahits GmbH, 2011
 * Created on Nov 28, 2011
 *
 */
@Component
@Lazy
@DependentInitialisation(MapInitializedBean.class)
@ClassCategory(EClassCategory.SINGLETON_BEAN)
public class CityEngine extends AbstractEngine {
    @XStreamOmitField
    private  final Logger logger = LogManager.getLogger(getClass());
	@Autowired
	private Random rnd;
	@Autowired
	private ApplicationContext context;
    @Autowired
    private ShipyardEngine shipyardEngine;
	@Autowired
	private TimedUpdatableTaskList taskList;

	/** Update the ware consumed and produced every 12h */
	private final static int WARE_UPDATES_MINUTES = 12*60;
	/** Number of ticks that need to waited untill {@link #WARE_UPDATES_MINUTES} */
	private int numberOfTicks;
	/** Counter that counts the ticks */
	private int tickCounter = 0;
	/** State indicating that the engine is not ready */
	private final static byte STOPPED = 0x00;
	/** State indicating that the engine is properly initialized */
	private final static byte STARTED = 0x01;
	/** State of the engine */
	private byte state = STOPPED;
    @Autowired
    private Date date;
	@Autowired
	private CitiesState citiesState;
    @Autowired
    @Qualifier("serverClientEventBus")
    private AsyncEventBus clientServerEventBus;
    @Autowired
    @Qualifier("serverEventBus")
    private AsyncEventBus serverEventBus;
	@Autowired
	@Qualifier("timerEventBus")
	private AsyncEventBus timerEventBus;
	@Autowired
	@Qualifier("javaFXApplicationThreadExecution")
	private IJavaFXApplicationThreadExecution threadExecutor;
	@Autowired
	private CityProductionAndConsumptionService productionAndConsumtionService;

    @Autowired
	private AutomaticTradingEngine automaticTradingEngine;
	@Autowired
	private LoanerEngine loanerEngine;
    @Autowired
    private CityHallEngine cityHallEngine;
	@Autowired
	private PlayerEngine playerEngine;
	@Autowired
	private ChurchEngine churchEngine;
	@Autowired
	private TavernEngine tavernEngine;
	@Autowired
	private GuildEngine guildEngine;

    @Autowired
    private GameFactory gameFactory;
    @Autowired
    private CityProductionStorage cityProductionStorage;

    @Autowired
    private ServerSideTaskFactory taskFactory;


    @PostConstruct
	public void initialize() {
		numberOfTicks = WARE_UPDATES_MINUTES/date.getTickUpdate();
        clientServerEventBus.register(this);
		timerEventBus.register(this);
        taskList.add(taskFactory.getWeeklyCityCheck());
	}


	@PreDestroy
	private void unregister() {
		clientServerEventBus.unregister(this);
		timerEventBus.unregister(this);
	}

	@Subscribe
	public void handleClockTickIntervallChange(ClockTickIntervalChange event) {
		if (event.getInterval() == 0) {
			numberOfTicks = -1;
		} else {
			numberOfTicks = WARE_UPDATES_MINUTES / event.getInterval();
		}
	}
	// for Test purposes
	int getNumberOfTicks() {
		return numberOfTicks;
	}

	@Subscribe
	public void handleClockTicked(ClockTick event) {
		tickCounter++;
		if (tickCounter==numberOfTicks && state==STARTED){
			tickCounter=0;
			threadExecutor.execute(this::consumeWares);
		}

	}
	/**
	 * Compute the consume of wares of each city
	 */
	private void consumeWares() {
		for (CityState state : citiesState.getCityEngineStates()) {
			productionAndConsumtionService.consumeWares(state);
		}
	}
	/**
	 * Add a new city to the engine
	 * @param city
	 */
	private void addCity(ICity city){
		CityState cityState = city.getCityState();
		tavernEngine.addCity(cityState.getTavernState(), city);
		loanerEngine.addNewLoaner(city);
        cityHallEngine.establishCityHall(city);
		churchEngine.establishChurch(city);
        guildEngine.establishGuild(city);
	}
	/**
	 * Start the engine. This should be called after all cities were added
	 */
	public void start(IGame game){
        for (ICity city : game.getMap().getCities()) {
            CityState cityState = gameFactory.getCityState(city);
            city.setCityState(cityState);
            addCity(city);
        }
        citiesState.init();
        state=STARTED;

	}


    @Override
    public List<AbstractEngine> getChildren() {
        ArrayList<AbstractEngine> engines = newArrayList();
        engines.add(tavernEngine);
        engines.add(shipyardEngine);
        engines.add(automaticTradingEngine);
		engines.add(loanerEngine);
        engines.add(cityHallEngine);
        engines.add(playerEngine);
        engines.add(churchEngine);
		engines.add(guildEngine);
        return engines;
    }

    public ShipyardEngine getShipyardEngine() {
        return shipyardEngine;
    }

    /**
     * Find the tafern engine for the matching city.
     * @return
     */
    public TavernEngine findTavernEngine() {
        return tavernEngine;
    }

	/**
	 * Handle the event where a new city is added to the map.
	 * @param event
     */
	@Subscribe
	public void handleCityAdd(FoundingCityBecomesAccessible event) {
		ICity city = event.getCity();
		addCity(city);
        CityState cityState = gameFactory.getCityState(city);
		citiesState.addCity(city, cityState);
        cityProductionStorage.add(city);
	}

}
