package ch.sahits.game.openpatrician.engine.land.city;

import ch.sahits.game.event.data.PeriodicalDailyUpdate;
import ch.sahits.game.openpatrician.annotation.ClassCategory;
import ch.sahits.game.openpatrician.annotation.EClassCategory;
import ch.sahits.game.openpatrician.annotation.LazySingleton;
import ch.sahits.game.openpatrician.clientserverinterface.model.factory.StateFactory;
import ch.sahits.game.openpatrician.clientserverinterface.service.GuildService;
import ch.sahits.game.openpatrician.dialog.ButtonTemplate;
import ch.sahits.game.openpatrician.dialog.DialogTemplate;
import ch.sahits.game.openpatrician.engine.AbstractEngine;
import ch.sahits.game.openpatrician.engine.event.task.ServerSideTaskFactory;
import ch.sahits.game.openpatrician.model.Date;
import ch.sahits.game.openpatrician.model.DisplayMessage;
import ch.sahits.game.openpatrician.model.DisplayTemplateMessage;
import ch.sahits.game.openpatrician.model.ICitizen;
import ch.sahits.game.openpatrician.model.ICompany;
import ch.sahits.game.openpatrician.model.IHumanPlayer;
import ch.sahits.game.openpatrician.model.IPlayer;
import ch.sahits.game.openpatrician.model.ModelFactory;
import ch.sahits.game.openpatrician.model.city.ICity;
import ch.sahits.game.openpatrician.model.city.ICreditor;
import ch.sahits.game.openpatrician.model.city.ILoaner;
import ch.sahits.game.openpatrician.model.city.LoanerList;
import ch.sahits.game.openpatrician.model.city.guild.GuildList;
import ch.sahits.game.openpatrician.model.city.guild.IGuild;
import ch.sahits.game.openpatrician.model.city.guild.IShipAuction;
import ch.sahits.game.openpatrician.model.city.impl.IDebt;
import ch.sahits.game.openpatrician.model.event.TargetedEvent;
import ch.sahits.game.openpatrician.model.event.TimedTask;
import ch.sahits.game.openpatrician.model.event.TimedUpdatableTaskList;
import ch.sahits.game.openpatrician.model.personal.ESocialRank;
import ch.sahits.game.openpatrician.model.personal.IReputation;
import ch.sahits.game.openpatrician.model.ship.IShip;
import com.google.common.annotations.VisibleForTesting;
import com.google.common.eventbus.AsyncEventBus;
import com.google.common.eventbus.Subscribe;
import org.joda.time.DateTime;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;

import javax.annotation.PostConstruct;
import javax.annotation.PreDestroy;
import java.util.ArrayList;
import java.util.Iterator;
import java.util.List;
import java.util.Random;

/**
 * Engine for controlling all the loaners.
 * @author Andi Hotz, (c) Sahits GmbH, 2014
 *         Created on Dec 30, 2014
 */
@LazySingleton
@ClassCategory(EClassCategory.SINGLETON_BEAN)
public class LoanerEngine extends AbstractEngine {
    @Autowired
    @Qualifier("serverClientEventBus")
    private AsyncEventBus clientServerEventBus;
    @Autowired
    private StateFactory factory;
    @Autowired
    private Date date;
    @Autowired
    private Random rnd;
    @Autowired
    private LoanerList loaners;
    @Autowired
    private StateFactory stateFactory;
    @Autowired
    private ModelFactory modelFactory;
    @Autowired
    private GuildList guildList;
    @Autowired
    private GuildService guildService;
    @Autowired
    private TimedUpdatableTaskList taskList;
    @Autowired
    private ServerSideTaskFactory taskFactory;

    @PostConstruct
    private void init() {
        clientServerEventBus.register(this);
        taskList.add(taskFactory.getWeeklyLoanerCheck());
    }
    @PreDestroy
    private void unregister() {
        clientServerEventBus.unregister(this);
    }


    @Subscribe
    public void handleDailyUpdate(PeriodicalDailyUpdate event) {
        DateTime now = date.getCurrentDate();
        for (ILoaner loaner : loaners) {
            List<IDebt> debts = loaner.getDebts();
            for (Iterator<IDebt> iterator = debts.iterator(); iterator.hasNext(); ) {
                IDebt debt = iterator.next();
                if (debt.getDueDate().isBefore(now)) {
                    ICitizen debitor = debt.getDebitor();
                    int cash = (int) (debt.getAmount() * debt.getInterest());
                    if (debitor instanceof IPlayer) {
                        IPlayer player = (IPlayer) debitor;
                        ICompany company = player.getCompany();
                        if (company.getCash() > cash) {
                            company.updateCash(-cash);
                            if (debt.getCreditor() instanceof IPlayer) {
                                ((IPlayer)debt.getCreditor()).getCompany().updateCash(cash);
                            }
                        } else {
                            postMessageDebtNotFulfillable(debt, loaner.getCity());
                        }
                    } else {  // Debitor is not a player
                        if (debitor.getRank().equals(ESocialRank.BARGAINER) ||
                                debitor.getRank().equals(ESocialRank.CHANDLER) ||
                                debitor.getRank().equals(ESocialRank.MERCHANT) ||
                                debitor.getRank().equals(ESocialRank.TRADESMAN) ) {
                            int payBack = rnd.nextInt(4);
                            if (payBack == 0) {
                                postMessageDebtNotFulfillable(debt, loaner.getCity());
                            } else {
                                if (debt.getCreditor() instanceof IPlayer) {
                                    ((IPlayer)debt.getCreditor()).getCompany().updateCash(cash);
                                }
                            }
                        } else {
                            // Can always pay back
                            if (debt.getCreditor() instanceof IPlayer) {
                                ((IPlayer)debt.getCreditor()).getCompany().updateCash(cash);
                            }
                        }
                    }
                    iterator.remove();
                }
            }
        }
    }
    @VisibleForTesting
    void postMessageDebtNotFulfillable(final IDebt debt, final ICity city) {
        String titleTemplate = "ch.sahits.game.openpatrician.engine.land.city.LoanerEngine.paybackTitle";
        String messageTemplate = "ch.sahits.game.openpatrician.engine.land.city.LoanerEngine.paybackMessage";
        String remitBtnLbl = "ch.sahits.game.openpatrician.engine.land.city.LoanerEngine.remitBtn";
        String pawnBtnLbl = "ch.sahits.game.openpatrician.engine.land.city.LoanerEngine.pawnBtn";
        int sum = (int) (debt.getAmount() * debt.getInterest());
        Object[] messageArgs = new Object[] {debt.getDebitor().getName(), debt.getDebitor().getLastName(), sum};
        Runnable remit = createRemitRunnable(debt);
        Runnable pawn = createPawnRunnable(debt, city);
        List<ButtonTemplate> buttons = new ArrayList<>();
        ButtonTemplate remitBtn = ButtonTemplate.builder()
                .action(remit)
                .labelKey(remitBtnLbl)
                .largeButton(true)
                .build();
        ButtonTemplate pawnBtn = ButtonTemplate.builder()
                .action(pawn)
                .labelKey(pawnBtnLbl)
                .largeButton(true)
                .build();
        buttons.add(remitBtn);
        buttons.add(pawnBtn);
        DialogTemplate dialogTemplate = DialogTemplate.builder()
                .titleKey(titleTemplate)
                .messageKey(messageTemplate)
                .messageArgs(messageArgs)
                .buttons(buttons)
                .closable(false)
                .build();
        DisplayTemplateMessage displayMessage = new DisplayTemplateMessage(titleTemplate, dialogTemplate);
        if (debt.getDebitor() instanceof IHumanPlayer) {
            TargetedEvent message = new TargetedEvent((IHumanPlayer) debt.getDebitor(), displayMessage);
            clientServerEventBus.post(message);
        }
    }
    @VisibleForTesting
    Runnable createRemitRunnable(final IDebt debt) {
        return new Runnable() {
            @Override
            public void run() {
                ICreditor creditor = debt.getCreditor();
                if (creditor instanceof IPlayer) {
                    ICity city = debt.getDebitor().getHometown();
                    IReputation reputation = city.getReputation((IPlayer) creditor);
                    reputation.update(500);
                }
            }
        };
    }
    @VisibleForTesting
    Runnable createPawnRunnable(final IDebt debt, final ICity city) {
        return () -> {
            if (debt.getDebitor() instanceof IHumanPlayer) {
                pawnFromHumanDebitor(debt, city);

                // Add task for the following day to collect the money
            } else if (debt.getCreditor() instanceof IHumanPlayer){
                int value = rnd.nextInt(debt.getAmount()/2) + debt.getAmount();
                value = (int) Math.min(value, debt.getAmount()*debt.getInterest());
                ((IPlayer)debt.getCreditor()).getCompany().updateCash(value);
                ICitizen debitor = debt.getDebitor();
                DisplayMessage msg = new DisplayMessage("ch.sahits.game.openpatrician.engine.land.city.LoanerEngine.pawnMsg1", new Object[]{debitor.getName(), debitor.getLastName(), value});
                TargetedEvent message = new TargetedEvent((IHumanPlayer) debt.getCreditor(), msg);
                clientServerEventBus.post(message);
            }
        };
    }
    @VisibleForTesting
    void pawnFromHumanDebitor(final IDebt debt, final ICity city) {
        // pawn a ship and auction it
        int sum = (int) (debt.getAmount() * debt.getInterest());
        final IPlayer debitor = (IPlayer) debt.getDebitor();
        IShip shipMatch = null;
        int maxDelta = Integer.MAX_VALUE;
        for (IShip ship : debitor.getFleet()) {
            int shipValue = ship.getValue();
            if  (Math.abs(shipValue - sum) < maxDelta) {
                maxDelta = Math.abs(shipValue - sum);
                shipMatch = ship;
            }
        }
        if (shipMatch != null) {
            DateTime auctionDate = date.getCurrentDate().plusDays(10);
            // Add auction to guild
            IGuild guild = guildList.findGuild(city).get();
            auctionDate = guildService.cleanUpAuctionAndDetermineAuctionDate(auctionDate, guild);
            IShipAuction auction = modelFactory.createShipAuction(auctionDate, shipMatch.getValue(), debitor, shipMatch);
            guild.getAuctions().add(auction);

            final DateTime dateCollectMoney = auctionDate.plusDays(1);
            TimedTask collectMoney = taskFactory.getLoanerCollectMoneyAfterAuctionTask(debt, dateCollectMoney);
            taskList.add(collectMoney);
        } else {
            final DateTime dateCollectMoney = date.getCurrentDate();
            TimedTask collectMoney = taskFactory.getLoanerCollectMoneyTask(debt, debitor, dateCollectMoney);
            taskList.add(collectMoney);
        }
    }

    void addNewLoaner(ICity city) {
        factory.createLoaner(city);
    }

    @Override
    public List<AbstractEngine> getChildren() {
        return new ArrayList<>();
    }

}
