package ch.sahits.game.openpatrician.engine.player.strategy;

import ch.sahits.game.event.data.RepairFinishedEvent;
import ch.sahits.game.event.data.ai.HireSailorEvent;
import ch.sahits.game.openpatrician.annotation.ClassCategory;
import ch.sahits.game.openpatrician.annotation.EClassCategory;
import ch.sahits.game.openpatrician.annotation.LazySingleton;
import ch.sahits.game.openpatrician.clientserverinterface.model.PathInterpolatorMap;
import ch.sahits.game.openpatrician.clientserverinterface.service.MapProxy;
import ch.sahits.game.openpatrician.clientserverinterface.service.MapService;
import ch.sahits.game.openpatrician.engine.player.tradesteps.AggregatedBuyTradeStep;
import ch.sahits.game.openpatrician.engine.player.tradesteps.AggregatedCheckedBuyTradeStep;
import ch.sahits.game.openpatrician.engine.player.tradesteps.AggregatesDumpTradeStep;
import ch.sahits.game.openpatrician.engine.player.tradesteps.AggregatesSellTradeStep;
import ch.sahits.game.openpatrician.engine.player.tradesteps.AmountBasedAggregatedDumpTradeStep;
import ch.sahits.game.openpatrician.engine.player.tradesteps.CheckForRepairTradeStep;
import ch.sahits.game.openpatrician.engine.player.tradesteps.CheckHireCaptainTradeStep;
import ch.sahits.game.openpatrician.engine.player.tradesteps.GuildJoinTradeStep;
import ch.sahits.game.openpatrician.engine.player.tradesteps.HireSailorsStep;
import ch.sahits.game.openpatrician.engine.player.tradesteps.PayBackLoanTradeStep;
import ch.sahits.game.openpatrician.engine.player.tradesteps.TakeLoanTradeStep;
import ch.sahits.game.openpatrician.engine.player.tradesteps.TravelToTradeStep;
import ch.sahits.game.openpatrician.model.IAIPlayer;
import ch.sahits.game.openpatrician.model.city.ICity;
import ch.sahits.game.openpatrician.model.player.IAITradeStrategy;
import ch.sahits.game.openpatrician.model.player.IAITradeStrategyType;
import ch.sahits.game.openpatrician.model.player.ICityProductionConsumptionKnowledge;
import ch.sahits.game.openpatrician.model.player.IProductionConsumptionKnowledge;
import ch.sahits.game.openpatrician.model.product.EWare;
import ch.sahits.game.openpatrician.model.product.ITradeStep;
import ch.sahits.game.openpatrician.model.product.IWare;
import ch.sahits.game.openpatrician.model.ship.INavigableVessel;
import com.google.common.annotations.VisibleForTesting;
import com.google.common.base.Preconditions;
import com.google.common.eventbus.Subscribe;
import com.thoughtworks.xstream.annotations.XStreamOmitField;
import javafx.util.Pair;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.context.ApplicationContext;

import java.util.ArrayList;
import java.util.Collections;
import java.util.Comparator;
import java.util.LinkedList;
import java.util.List;
import java.util.ListIterator;
import java.util.Optional;
import java.util.Set;
import java.util.stream.Collectors;

/**
 * Base implementation of the trading strategy. The base strategy maintains a list of TradeSteps, that
 * are executed in order. Once the list is empty the next steps need to be defined. It might be that the
 * last executed step complets a trade cycle and the next step would be the repetition of the first.
 */
@LazySingleton
@ClassCategory(EClassCategory.SINGLETON_BEAN)
public abstract class BasePlayerTradeStrategy implements IAITradeStrategy {
    private final Logger logger = LogManager.getLogger(getClass());
    @Autowired
    @XStreamOmitField
    private MapProxy mapProxy;
    @Autowired
    private ApplicationContext context;
    @Autowired
    @XStreamOmitField
    private MapService mapService;

    @Autowired
    private PathInterpolatorMap interpolators;

    protected IAITradeStrategyType tradeStrategyType;

    protected boolean isMatchingTradeStrategy(IAIPlayer player, INavigableVessel vessel) {
        IAITradeStrategyType tradeStrategyType = player.getTradeStrategyType(vessel);
        return tradeStrategyType.equals(this.tradeStrategyType);
    }

    /**
     * Retrieve the next trade step that should be executed and remove it from the list of an AI player.
     * @param player on witch to retrieve the next task
     * @return
     */
    public ITradeStep getNextStep(IAIPlayer player, INavigableVessel vessel) {
        return player.getNextTradeStep(vessel);
    }

    /**
     * Check whether the AI player has more trade steps
     * @param player on which to check
     * @return
     */
    public boolean hasMoreTradeSteps(IAIPlayer player, INavigableVessel vessel) {
        return player.hasMoreTradeSteps(vessel);
    }

    /**
     * Add a new trade step for a player.
     * @param player for which to add the trade step
     * @param step to be added.
     */
    public void append(IAIPlayer player, INavigableVessel vessel, ITradeStep step) {
        player.addTradeStep(step, vessel);
    }

    /**
     * Add a new trade step to the begining of the list.
     * @param player for which to add the trade step
     * @param step to be added.
     * @param vessel
     */
    public void inject(IAIPlayer player, INavigableVessel vessel, ITradeStep step) {
        player.injectTradeStep(step, vessel);
    }

    /**
     * Figure out the ware that are most needed in the city. The need is calculated upon the knowledge
     * of the production, consumption and sotred amount. If a ware is not consumed at all it is not needed at
     * all, compared to wares that have more consumption than stored + produced.
     * @param knowledge base knowledge
     * @return sorted list Pair of wares and their importants. The lower the number the more important.
     */
    public List<Pair<IWare, Number>> getMostNeededWares(ICityProductionConsumptionKnowledge knowledge) {
        Preconditions.checkNotNull(knowledge, "The city production knowledge may not be null");
        List<Pair<IWare, Number>> need = new ArrayList<>();
        for (IWare ware : EWare.values()) {
            int stored = knowledge.getStoredAmount(ware);
            int produced = knowledge.getProductionAmount(ware);
            int consumed = knowledge.getConsumptionAmount(ware);
            int value;
            if (consumed == 0) {
                value = Integer.MAX_VALUE;
            } else {
                value = stored + produced - consumed;
            }
            need.add(new Pair<>(ware, value));
        }
        return need.stream().sorted(new WareNeedComparator()).collect(Collectors.toList());
    }

    /**
     * Check if the <code>ware</code> is needed in a city.
     * @param knowledge of the city that should be checked.
     * @param ware for which should be checked.
     * @return true if there is not much of the ware around.
     */
    public boolean isNeeded(ICityProductionConsumptionKnowledge knowledge, IWare ware) {
        List<Pair<IWare, Number>> needs = getMostNeededWares(knowledge);
        for (Pair<IWare, Number> need : needs) {
            if (need.getKey().equals(ware) && need.getValue().intValue() < 10) {
                return true;
            } else if (need.getKey().equals(ware) || need.getValue().intValue() > 10) {
                return false;
            }

        }
        return false;
    }

    /**
     * Figure out the ware that are most needed in the city. The need is calculated upon the knowledge
     * of the production, consumption and sotred amount. If a ware is not consumed at all it is not needed at
     * all, compared to wares that have more consumption than stored + produced. Also consider the amount of that ware
     * that is loaded on the <code>vessel</code>. That ware is treated the same way as if it stored in the city, with the
     * effect that on the first call the most needed ware might be BEER, but once an amount is bought the most needed ware might
     * change.
     * @param knowledge base knowledge
     * @param vessel on which the wares are loaded.
     * @return sorted list Pair of wares and their importants. The lower the number the more important.
     */
    public List<Pair<IWare, Number>> getMostNeededWares(ICityProductionConsumptionKnowledge knowledge, INavigableVessel vessel) {
        List<Pair<IWare, Number>> sortedNeeds = getMostNeededWares(knowledge);
        for (ListIterator<Pair<IWare, Number>> iterator = sortedNeeds.listIterator(); iterator.hasNext(); ) {
            Pair<IWare, Number> need = iterator.next();
            if (vessel.getWare(need.getKey()).getAmount() > 0) {
                int value = need.getValue().intValue() + vessel.getWare(need.getKey()).getAmount();
                iterator.remove();
                iterator.add(new Pair<>(need.getKey(), value));
            }
        }
        return sortedNeeds.stream().sorted(new WareNeedComparator()).collect(Collectors.toList());
    }

    /**
     * Figure out what wares are needed in <code>nextStop</code>, that can be provided in the current town.
     * @param knowledge base knowledge
     * @param knowledgeCurrentTown knowledge of the current town
     * @param nextStop stop of the next city.
     * @return list of wares that can be delivered from the current city to the <code>nextStop</code>
     */
    protected List<IWare> getWaresNeedIn(IProductionConsumptionKnowledge knowledge, ICityProductionConsumptionKnowledge knowledgeCurrentTown, ICity nextStop) {
        List<Pair<IWare, Number>> sortedNeeds;
        ICityProductionConsumptionKnowledge knowledgeFirstTown = knowledge.getKnowlege(nextStop);
        sortedNeeds = getMostNeededWares(knowledgeFirstTown);
        List<IWare> deliverWare = new ArrayList<>(); // wares to be bought in hometown and sold in nextStop
        for (Pair<IWare, Number> need : sortedNeeds) {
            if (knowledgeCurrentTown.getProductionAmount(need.getKey()) > 0) {
                deliverWare.add(need.getKey());
            }
        }
        return deliverWare;
    }

    /**
     * Figure out what the next stop should be to supply the ware in the list <code>waresOfInterest</code>.
     * This list is ordered so that wares in the front are more important.
     * @param baseTown city from which the travel starts
     * @param knowledge data
     * @param waresOfInterest ordered list of needed wares
     * @param exclude city to which should not be traveled.
     * @return City that can provide at least some of the wares and is not to far away.
     */
    protected ICity findNextStop(ICity baseTown, IProductionConsumptionKnowledge knowledge, List<IWare> waresOfInterest, INavigableVessel vessel, ICity exclude) {
        List<ICity> cities = knowledge.findListWithProductionsMinimalDistance(baseTown, waresOfInterest.get(0), vessel); // TODO ahotz 08.06.2016: do not only consider the first ware
        if (cities.get(0).equals(exclude) && cities.size() > 1) {
            return cities.get(1);
        } else {
            return cities.get(0);
        }
    }

    /**
     * Create the travel step to a different city.
     * @param vessel that is travelling
     * @param travelToCity destination city
     * @return TravelToTradeStep
     */
    protected TravelToTradeStep createTravelToStep(INavigableVessel vessel, ICity travelToCity) {
        TravelToTradeStep travelTo = context.getBean(TravelToTradeStep.class);
        travelTo.setDestinationCity(travelToCity);
        travelTo.setVessel(vessel);
        return travelTo;
    }

    /**
     * Create the trade step to check for repairs.
     * @param vessel that should be checked.
     * @param city in which should be checked.
     * @return CheckForRepairTradeStep
     */
    protected CheckForRepairTradeStep createCheckRepairStep(INavigableVessel vessel, ICity city) {
        CheckForRepairTradeStep repairStep = context.getBean(CheckForRepairTradeStep.class);
        repairStep.setCity(city);
        repairStep.setVessel(vessel);
        return repairStep;
    }



    /**
     * Create a new instance forthe hire sailor step.
     * @param vessel
     * @param city
     * @return
     */
    protected HireSailorsStep createHireSailorStep(INavigableVessel vessel, ICity city) {
        HireSailorsStep hireSailors = context.getBean(HireSailorsStep.class);
        hireSailors.setShip(vessel);
        hireSailors.setCity(city);
        return hireSailors;
    }

    /**
     * Create a new instance of the TakeLoan trade step.
     * @param player that takes a loan
     * @param city in which the loan is taken
     * @return
     */
    protected TakeLoanTradeStep createCheckAndTakeLoanStep(IAIPlayer player, ICity city) {
        TakeLoanTradeStep takeLoan = context.getBean(TakeLoanTradeStep.class);
        takeLoan.setCity(city);
        takeLoan.setPlayer(player);
        return takeLoan;
    }
    /**
     * Create a new instance of the TakeLoan trade step.
     * @param player that takes a loan
     * @param city in which the loan is taken
     * @return
     */
    protected PayBackLoanTradeStep createPaybackLoanStep(IAIPlayer player, ICity city) {
        PayBackLoanTradeStep takeLoan = context.getBean(PayBackLoanTradeStep.class);
        takeLoan.setCity(city);
        takeLoan.setPlayer(player);
        return takeLoan;
    }

    protected AggregatedBuyTradeStep createAggregatedBuyTradeStep(INavigableVessel vessel, ICity city, List<IWare> waresToBuy) {
        AggregatedBuyTradeStep buyStepHometown = context.getBean(AggregatedBuyTradeStep.class);
        buyStepHometown.setCity(city);
        buyStepHometown.setVessel(vessel);
        buyStepHometown.setExecuteNext(true);
        for (IWare ware : waresToBuy) {
            int maxPrice = ware.getMaxBuyPriceOffensive();
            buyStepHometown.addBuyStep(ware, maxPrice);
        }
        return buyStepHometown;
    }

    protected AggregatedCheckedBuyTradeStep createAggregatedCheckedBuyTradeStep(INavigableVessel vessel, ICity city, List<IWare> waresToBuy) {
        AggregatedCheckedBuyTradeStep buyStepHometown = context.getBean(AggregatedCheckedBuyTradeStep.class);
        buyStepHometown.setCity(city);
        buyStepHometown.setVessel(vessel);
        buyStepHometown.setExecuteNext(true);
        for (IWare ware : waresToBuy) {
            int maxPrice = ware.getMaxBuyPriceOffensive();
            buyStepHometown.addBuyStep(ware, maxPrice);
        }
        return buyStepHometown;
    }

    protected AggregatesSellTradeStep createAggregatedSellStep(INavigableVessel vessel, ICity city, List<IWare> waresToSell) {
        AggregatesSellTradeStep sellStep = context.getBean(AggregatesSellTradeStep.class);
        sellStep.setVessel(vessel);
        sellStep.setCity(city);
        sellStep.setExecuteNext(true);
        for (IWare ware : waresToSell) {
            sellStep.addSellStep(ware);
        }
        return sellStep;
    }

    protected AggregatesDumpTradeStep createAggregatedDumpStep(INavigableVessel vessel, ICity city, List<IWare> waresToSell) {
        AggregatesDumpTradeStep sellStep = context.getBean(AggregatesDumpTradeStep.class);
        sellStep.setVessel(vessel);
        sellStep.setCity(city);
        sellStep.setExecuteNext(true);
        for (IWare ware : waresToSell) {
            sellStep.addSellStep(ware);
        }
        return sellStep;
    }

    /**
     * Create a conditional dump trade step. All wares are sold if the ships load is greater than the <code>maxAmount</code>.
     * @param vessel in the trade step
     * @param city where the trade step happens
     * @param waresToSell list of wares that should be sold
     * @param maxAmount max amount of wares that should be on the ship
     * @return dump sell trade step
     */
    protected AmountBasedAggregatedDumpTradeStep createConditionalAggregatedDumpStep(INavigableVessel vessel, ICity city, List<IWare> waresToSell, int maxAmount) {
        AmountBasedAggregatedDumpTradeStep sellStep = context.getBean(AmountBasedAggregatedDumpTradeStep.class);
        sellStep.setVessel(vessel);
        sellStep.setCity(city);
        sellStep.setExecuteNext(true);
        sellStep.setMaxAmount(maxAmount);
        for (IWare ware : waresToSell) {
            sellStep.addSellStep(ware);
        }
        return sellStep;
    }

    /**
     * Create the trade step to check and hire a captain.
     * @param vessel
     * @param city
     * @param player
     * @return
     */
    protected CheckHireCaptainTradeStep createHireCaptain(INavigableVessel vessel, ICity city, IAIPlayer player) {
        CheckHireCaptainTradeStep hireCaptain = context.getBean(CheckHireCaptainTradeStep.class);
        hireCaptain.setVessel(vessel);
        hireCaptain.setCity(city);
        hireCaptain.setPlayer(player);
        return hireCaptain;
    }

    protected GuildJoinTradeStep createJoinGuildTradeStep(INavigableVessel vessel, ICity city, IAIPlayer player) {
        GuildJoinTradeStep tradeStep = context.getBean(GuildJoinTradeStep.class);
        tradeStep.setVessel(vessel);
        tradeStep.setCity(city);
        tradeStep.setPlayer(player);
        return tradeStep;

    }

    /**
     * The implementation of the execute trade steps makes the following assumptions:
     * <ul>
     *     <li>When called there is at least one trade step defined</li>
     * </ul>
     * Typically the trade steps are defined up to the travel step. When the ship
     * then reaches the port the next steps need to be defined.
     * @param player
     * @param vessel
     * @return
     */
    @Override
    public boolean executeTradeSteps(IAIPlayer player, INavigableVessel vessel) {
        boolean proceedWithExecutionOfNextStep = true;
        if (hasMoreTradeSteps(player, vessel)) {
            try {
                logger.debug("Executed more steps for {} of {} {}", vessel.getName(), player.getName(), player.getLastName());
                ITradeStep nextStep = getNextStep(player, vessel);
                logger.debug("Next step for {} of {} {}: {}", vessel.getName(), player.getName(), player.getLastName(), nextStep);
                if (nextStep.execute()) {
                    logger.debug("Execute recursive after " + nextStep);
                    proceedWithExecutionOfNextStep = executeTradeSteps(player, vessel);
                } else {
                    proceedWithExecutionOfNextStep = false;
                    logger.debug("Do not execute any further steps after: {}", nextStep);
                }
            } catch (IllegalStateException e) {
                throw e;
            } catch (RuntimeException e) {
                logger.error("Failed in execution of trade step for strategy: "+player.getTradeStrategyType(vessel)+" player "+player.getName()+" "+player.getLastName()+", vessel: "+vessel.getName(), e);
            }
        } else {
            throw new IllegalStateException(String.format("There are no further steps defined for %s of %s %s for strategy %s", vessel.getName(), player.getName(), player.getLastName(), getClass().getName()));
        }
        player.updateTradeWaitingStatus(vessel, !proceedWithExecutionOfNextStep);
        return proceedWithExecutionOfNextStep;
    }



    /**
     * Handle the event of the ship coming back from repair and the trade steps have to
     * be picked up again.
     * @param event
     */
    @Subscribe
    public void handleRepairFinished(RepairFinishedEvent event) {
        INavigableVessel vessel = event.getShip();
        IAIPlayer player = (IAIPlayer) vessel.getOwner();
        if (isMatchingTradeStrategy(player, vessel)) {
            Preconditions.checkArgument(hasMoreTradeSteps(player, vessel), "There most be steps defined after repair for "+vessel.getName()+" of "+player.getName()+" "+player.getLastName());
            player.updateTradeWaitingStatus(vessel, false);
            executeTradeSteps(player, vessel);
        }
    }

    /**
     * Andling the hiring of sailors. This event is fired when the last try to hire sailors failed to insufficient
     * amount of sailors available.
     * @param event
     */
    @Subscribe
    public void handleHireSailors(HireSailorEvent event) {
        INavigableVessel ship = event.getShip();
        ICity city = event.getCity();
        IAIPlayer player = (IAIPlayer) ship.getOwner();
        if (isMatchingTradeStrategy(player, ship)) {
            HireSailorsStep hireSailor = createHireSailorStep(ship, city);
            inject(player, ship, hireSailor);
            executeTradeSteps(player, ship); // thre might not be any tradesteps defined for the ship if it belongs to a group.
        }
    }

    /**
     * Find all wares that are loaded on the ship.
     * @param vessel
     * @return
     */
    protected List<IWare> getLoadedWares(INavigableVessel vessel) {
        return vessel.getLoadedWares()
                .stream()
                .filter(ware -> vessel.getWare(ware).getAmount() > 0)
                .collect(Collectors.toList());
    }

    protected ArrayList<IWare> findWaresOfInterest(List<Pair<IWare, Number>> sortedNeeds) {
        ArrayList<IWare> waresOfInterest = new ArrayList<>(); // wares to be delivered to hometown
        for (Pair<IWare, Number> need : sortedNeeds) {
            if (need.getValue().doubleValue() < 0 || waresOfInterest.isEmpty()) {
                waresOfInterest.add(need.getKey());
            } else if (waresOfInterest.size() < 2) { // Have at least 2 entries in the list
                waresOfInterest.add(need.getKey());
            } else {
                break; // list is sorted by value increasing order
            }
        }
        return waresOfInterest;
    }

    /**
     * Determine the city to travel to based on the required wares. If there is
     * a city that provides all the wares it is chosen otherwise a city that provides
     * part of the wares is returned.
     * @param player
     * @param requiredWares list of required wares
     * @param excludeCity origin city to which should not be traveled.
     * @return
     */
    protected ICity findDestinationToBuyRequiredProductionWares(IAIPlayer player, List<IWare> requiredWares, INavigableVessel vessel, Optional<ICity> excludeCity) {
        ICity destination;
        Optional<ICity> optDest = findCityWithAllRequiredWares(requiredWares, player, vessel, excludeCity);
        if (optDest.isPresent()) {
            destination = optDest.get();
        } else {
            destination = findCitySupplyingWare(player, requiredWares.get(0), vessel, excludeCity);
        }
        return destination;
    }

    /**
     * Find a destination city that requires all the wares.
     * @param requiredWares
     * @param player
     * @param excludeCity
     * @return
     */
    @VisibleForTesting
    Optional<ICity> findCityWithAllRequiredWares(List<IWare> requiredWares, IAIPlayer player, INavigableVessel vessel, Optional<ICity> excludeCity) {
         IProductionConsumptionKnowledge knowledge = player.getProductionAndConsumptionKnowledge();
        List<ICity> cities = new ArrayList<>();
        List<ICity> mapCities = new ArrayList<>(mapProxy.getAllReachableNonBlockadedCities(vessel));
        if (excludeCity.isPresent()) {
            mapCities.remove(excludeCity.get());
        }
        for (ICity city : mapCities) {
            ICityProductionConsumptionKnowledge cityKnowledge = knowledge.getKnowlege(city);
            boolean producesAll = true;
            for (IWare ware : requiredWares) {
                int productionAmount = cityKnowledge.getProductionAmount(ware);
                if (productionAmount < 1) {
                    producesAll = false;
                    break;
                }
            }
            if (producesAll) {
                cities.add(city);
            }
        }
        if (cities.isEmpty()) {
            return Optional.empty();
        } else {
            Collections.shuffle(cities);
            return Optional.of(cities.get(0));
        }
    }

    /**
     * Finding a destination that can supply a ware.
     * @param player
     * @param ware
     * @param excludeCity
     * @return
     */
    @VisibleForTesting
    ICity findCitySupplyingWare(IAIPlayer player, IWare ware, INavigableVessel vessel, Optional<ICity> excludeCity) {
        IProductionConsumptionKnowledge knowledge = player.getProductionAndConsumptionKnowledge();
        List<ICity> mapCities = new ArrayList<>(mapProxy.getAllReachableNonBlockadedCities(vessel));
        if (excludeCity.isPresent()) {
            mapCities.remove(excludeCity.get());
        }
        for (ICity city : mapCities) {
            ICityProductionConsumptionKnowledge cityKnowledge = knowledge.getKnowlege(city);
            if (cityKnowledge.getProductionAmount(ware) > 0) {
                return city;
            }
        }
        logger.warn("Did not find city supplying ware: {}, excluding {}. Try finding city that should produce it", ware, excludeCity.get().getName());
        for (ICity city : mapCities) {
            for (IWare produced : city.getEffectiveProduction()) {
                if (produced.equals(ware)) {
                    return city;
                }
            }
        }
        for (ICity city : mapCities) {
            for (IWare produced : city.getIneffectiveProduction()) {
                if (produced.equals(ware)) {
                    return city;
                }
            }
        }
        return null;
    }

    /**
     * Add default trading steps:
     * <ol>
     *     <li>AggregatesSellTradeStep</li>
     *     <li>PayBackLoanTradeStep</li>
     *     <li>PayBackLoanTradeStep</li>
     *     <li>AggregatedCheckedBuyTradeStep</li>
     *     <li>CheckForRepairTradeStep</li>
     *     <li>HireSailorsStep</li>
     *     <li>TravelToTradeStep</li>
     * </ol>
     * @param vessel doing the trade
     * @param player owner of the vessel
     * @param city current city
     * @param nextStop next stop
     * @param nonSellableWares wares that cannot be sold
     * @param buyWares wares that need to be bought
     * @param skipRepair skip the task for repairing ships.
     */
    protected void addDefaultTradeSteps(INavigableVessel vessel, IAIPlayer player, ICity city, ICity nextStop, Set<IWare> nonSellableWares, Set<IWare> buyWares, boolean skipRepair) {
        List<IWare> sellWares = new LinkedList<>();
        sellWares.addAll(getLoadedWares(vessel));
        sellWares.removeAll(nonSellableWares); // Retain the wares needed for the production chain
        IProductionConsumptionKnowledge knowledge = player.getProductionAndConsumptionKnowledge();
        buyWares.addAll(getWaresNeedIn(knowledge, knowledge.getKnowlege(nextStop), nextStop));


        AggregatesSellTradeStep sellStep = createAggregatedSellStep(vessel, city, sellWares);
        append(player, vessel, sellStep);
        PayBackLoanTradeStep payBackLoan = createPaybackLoanStep(player, city);
        append(player, vessel, payBackLoan);
        CheckHireCaptainTradeStep hireCaptain = createHireCaptain(vessel, city, player);
        append(player, vessel, hireCaptain);
        GuildJoinTradeStep joinTradeStep = createJoinGuildTradeStep(vessel, city, player);
        append(player, vessel, joinTradeStep);
        TakeLoanTradeStep takeLoanStep = createCheckAndTakeLoanStep(player, city);
        append(player, vessel, takeLoanStep);
        AggregatedCheckedBuyTradeStep buyStepHometown = createAggregatedCheckedBuyTradeStep(vessel, city, new ArrayList<>(buyWares));
        append(player, vessel, buyStepHometown);
        if (!skipRepair) {
            CheckForRepairTradeStep repairStep = createCheckRepairStep(vessel, city);
            append(player, vessel, repairStep);
        }
        HireSailorsStep hireSailors = createHireSailorStep(vessel, city);
        append(player, vessel, hireSailors);
        TravelToTradeStep travelTo = createTravelToStep(vessel, nextStop);
        append(player, vessel, travelTo);

    }



    private static class WareNeedComparator implements Comparator<Pair<IWare, Number>> {

        @Override
        public int compare(Pair<IWare, Number> firstPair, Pair<IWare, Number> secondPair) {
            return (int)Math.rint(firstPair.getValue().doubleValue() - secondPair.getValue().doubleValue());
        }
    }
}
