package ch.sahits.game.openpatrician.engine.player.strategy;

import ch.sahits.game.event.data.ai.SpecialMissionFinishedEvent;
import ch.sahits.game.openpatrician.annotation.ClassCategory;
import ch.sahits.game.openpatrician.annotation.EClassCategory;
import ch.sahits.game.openpatrician.annotation.LazySingleton;
import ch.sahits.game.openpatrician.clientserverinterface.service.MapService;
import ch.sahits.game.openpatrician.clientserverinterface.service.ShipService;
import ch.sahits.game.openpatrician.clientserverinterface.service.TradeService;
import ch.sahits.game.openpatrician.engine.player.CollectWaresMissionData;
import ch.sahits.game.openpatrician.engine.player.tradesteps.AggregatedBuyTradeStep;
import ch.sahits.game.openpatrician.engine.player.tradesteps.AggregatesDumpTradeStep;
import ch.sahits.game.openpatrician.engine.sea.SeafaringService;
import ch.sahits.game.openpatrician.model.IAIPlayer;
import ch.sahits.game.openpatrician.model.building.ITradingOffice;
import ch.sahits.game.openpatrician.model.city.ICity;
import ch.sahits.game.openpatrician.model.event.IShipEntersPortEvent;
import ch.sahits.game.openpatrician.model.player.IAITradeStrategyType;
import ch.sahits.game.openpatrician.model.product.IWare;
import ch.sahits.game.openpatrician.model.ship.INavigableVessel;
import com.google.common.eventbus.AsyncEventBus;
import com.thoughtworks.xstream.annotations.XStreamOmitField;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;

import javax.annotation.PostConstruct;
import javax.annotation.PreDestroy;
import java.util.ArrayList;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Optional;

/**
 * Strategy to collect wares. This strategy is not employed for normal trading.
 * The trade mission data created by {@link #createMissionData(IAIPlayer)} will be
 * set on the player for the vessel.
 * @author Andi Hotz, (c) Sahits GmbH, 2016
 *         Created on Jul 25, 2016
 */
@LazySingleton
@ClassCategory(EClassCategory.SINGLETON_BEAN)
public abstract class CollectionWaresStrategy extends BasePlayerTradeStrategy {
    @XStreamOmitField
    private final Logger logger = LogManager.getLogger(getClass());
    @Autowired
    @Qualifier("serverClientEventBus")
    private AsyncEventBus clientServerEventBus;
    @Autowired
    private TradeService tradeService;
    @Autowired
    private SeafaringService seafaringService;
    @Autowired
    private ShipService shipService;
    @Autowired
    private MapService mapService;
    public CollectionWaresStrategy() {
        this.tradeStrategyType = EAITradeStrategyType.COLLECT_WARES;
    }

    @PostConstruct
    private void init() {
        clientServerEventBus.register(this);
    }
    @PreDestroy
    private void destroy() {
        clientServerEventBus.unregister(this);
    }

    @Override
    public void initialzeTradeCycle(IAIPlayer player, INavigableVessel vessel) {
        CollectWaresMissionData missionData = createMissionData(player);
        player.setTradeMission(vessel, missionData);
        // Assume the ship is not in a port => redirect
        List<IWare> requiredWares = new ArrayList<>(missionData.getRequiredWareAmounts().keySet());
        Optional<ICity> currentCity = shipService.findCity(vessel);
        ICity firstStop = findDestinationToBuyRequiredProductionWares(player, requiredWares, vessel, currentCity);
        IAITradeStrategyType strategyType = getStrategyType();
        player.setTradeStrategyType(vessel, strategyType);
        logger.debug("Reroute vessel "+vessel.getName()+" to "+firstStop.getName()+" current location: "+vessel.getLocation());
        if (player.hasMoreTradeSteps(vessel)) {
            logger.debug("There are tradesteps for vessel "+vessel.getName()+" that may cause problems");
        }
        seafaringService.travelToCity(vessel, firstStop);
    }

    /**
     * Provide the mission data for collecting wares.
     * @param player
     * @return
     */
    protected abstract CollectWaresMissionData createMissionData(IAIPlayer player);

    /**
     * Retrieve the strategy type for this strategy.
      * @return
     */
    protected abstract IAITradeStrategyType getStrategyType();

    @Override
    public void handleShipArrivesInPort(IShipEntersPortEvent event) {
        final ICity city = event.getCity();
        final INavigableVessel vessel = event.getShip();
        final IAIPlayer player = (IAIPlayer) vessel.getOwner();
        final CollectWaresMissionData tradeMission = (CollectWaresMissionData) player.getTradeMission(vessel);
        final ICity destination = tradeMission.getCity();
        boolean haveEverything = false;
        Map<IWare, Integer> requiredWareAmounts = tradeMission.getRequiredWareAmounts();
        if (city.equals(destination)) {
            // Check if we have everything
            Optional<ITradingOffice> optOffice = player.findTradingOffice(city);
            haveEverything = checkAllWaresCollected(city, vessel, tradeMission, optOffice);
            if (optOffice.isPresent()) {
                ITradingOffice tradingOffice = optOffice.get();
                tradeService.transferFromVesselToStorage(vessel, player, city, requiredWareAmounts);
                for (IWare ware : requiredWareAmounts.keySet()) {
                    int amountStored = tradingOffice.getWare(ware).getAmount();
                    if (amountStored > requiredWareAmounts.get(ware)) {
                        requiredWareAmounts.put(ware, 0);
                    } else {
                        int newRequiredAmount = requiredWareAmounts.get(ware) - amountStored;
                        requiredWareAmounts.put(ware, newRequiredAmount);
                    }
                }
            }
            if (haveEverything) {
                List<IWare> loadedWares = getLoadedWares(vessel);
                loadedWares.removeAll(requiredWareAmounts.keySet());
                AggregatesDumpTradeStep dumpStep = createAggregatedDumpStep(vessel, city, loadedWares);
                dumpStep.execute();
                SpecialMissionFinishedEvent newEvent = new SpecialMissionFinishedEvent(vessel, city, tradeMission);
                player.setTradeMission(vessel, null);
                clientServerEventBus.post(newEvent);
            }
        } // Destination city
        if (!haveEverything) {
            // Noremal trade
            List<IWare> requiredWares = getRequiredWares(vessel, requiredWareAmounts);
            AggregatedBuyTradeStep buyStep = createAggregatedBuyTradeStep(vessel, city, requiredWares);
            buyStep.execute();
            requiredWares = getRequiredWares(vessel, requiredWareAmounts);
            ICity nextStop = destination;
            if (!requiredWares.isEmpty()) {
                nextStop = findDestinationToBuyRequiredProductionWares(player, requiredWares, vessel, Optional.of(city));
            }
            if (nextStop == null) {
                nextStop = destination;
            }
            if (nextStop.equals(city)) {
                   nextStop = mapService.findNearestCityExclusive(city.getCoordinates());
            }
            addDefaultTradeSteps(vessel, player, city, nextStop, requiredWareAmounts.keySet(), new HashSet<>(requiredWareAmounts.keySet()), true);
            executeTradeSteps(player, vessel);
        }
    }

    private List<IWare> getRequiredWares(INavigableVessel vessel, Map<IWare, Integer> requiredWareAmounts) {
        List<IWare> requiredWares = new ArrayList<>(requiredWareAmounts.keySet());
        for (Iterator<IWare> iterator = requiredWares.iterator(); iterator.hasNext(); ) {
            IWare ware = iterator.next();
            int loadedAmount = vessel.getWare(ware).getAmount();
            if (loadedAmount >= requiredWareAmounts.get(ware)) {
                iterator.remove();
            }
        }
        return requiredWares;
    }

    /**
     * Check if all the required wares are actually collected.
     * @param city
     * @param vessel
     * @param tradeMission
     * @param tradingOffice
     * @return
     */
    protected abstract boolean checkAllWaresCollected(ICity city, INavigableVessel vessel, CollectWaresMissionData tradeMission, Optional<ITradingOffice> tradingOffice);
}
