package ch.sahits.game.openpatrician.engine.player.tradesteps;

import ch.sahits.game.openpatrician.annotation.ClassCategory;
import ch.sahits.game.openpatrician.annotation.EClassCategory;
import ch.sahits.game.openpatrician.annotation.ListType;
import ch.sahits.game.openpatrician.annotation.Prototype;
import ch.sahits.game.openpatrician.model.building.ITradingOffice;
import ch.sahits.game.openpatrician.model.city.ICity;
import ch.sahits.game.openpatrician.model.product.ITradeStep;
import ch.sahits.game.openpatrician.model.product.IWare;
import com.thoughtworks.xstream.annotations.XStreamOmitField;
import lombok.Setter;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.context.ApplicationContext;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

/**
 * Aggregate multiple BuySteps for the trading office
 * @author Andi Hotz, (c) Sahits GmbH, 2016
 *         Created on Dec 21, 2016
 */
@ClassCategory({EClassCategory.SERIALIZABLE_BEAN, EClassCategory.PROTOTYPE_BEAN})
@Prototype
public class AggregatedBuyTradingOfficeTradeStep implements ITradeStep {
    @Setter
    private ICity city;
    @Setter
    private ITradingOffice tradingOffice;
    @Autowired
    @XStreamOmitField
    private ApplicationContext context;
    @ListType(BuyTradingOfficeTradeStep.class)
    private ArrayList<BuyTradingOfficeTradeStep> tradeSteps = new ArrayList<>();

    public void addBuyStep(IWare ware, int maxPrice) {
        BuyTradingOfficeTradeStep buyStep = context.getBean(BuyTradingOfficeTradeStep.class);
        buyStep.setMaxBuyPrice(maxPrice);
        buyStep.setCity(city);
        buyStep.setTradingOffice(tradingOffice);
        buyStep.setWare(ware);
        tradeSteps.add(buyStep);
    }

    /**
     * Sort the list of wares that should be bought so that the
     * wares with the least amount are at the begining of the list.
     * @return priority sorted ware list.
     */
    private List<BuyTradingOfficeTradeStep> getPriorityList() {
        ArrayList<BuyTradingOfficeTradeStep> list = new ArrayList<>();
        Map<BuyTradingOfficeTradeStep,Integer> loadedWares = new HashMap<>();
        for (BuyTradingOfficeTradeStep step : tradeSteps) {
            IWare ware = step.getWare();
            int amount = tradingOffice.getWare(ware).getAmount();
            loadedWares.put(step, amount);
        }
        loadedWares.entrySet()
                .stream()
                .sorted(Map.Entry.<BuyTradingOfficeTradeStep, Integer>comparingByValue())
                .forEach(entry -> list.add(entry.getKey()));
        return list;
    }

    @Override
    public boolean execute() {
        List<BuyTradingOfficeTradeStep> priorityList = getPriorityList();
        for (BuyTradingOfficeTradeStep buyTradeStep : priorityList) {
            buyTradeStep.execute();
            if (tradingOffice.capacityProperty().get() - tradingOffice.getOccupiedSpace() == 0) {
                break;
            }
        }
        return true;
    }

    @Override
    public String toString() {
        StringBuilder sb = new StringBuilder();
        tradeSteps.stream().forEach(step -> sb.append(step.getWare().name()).append(" "));
        return "AggregatedBuyTradingOfficeTradeStep in "+city.getName()+" buying: "+sb.toString().toString();
    }
}
