package ch.sahits.game.openpatrician.engine.time;

import ch.sahits.game.event.data.ClockTick;
import ch.sahits.game.event.data.PeriodicalDailyUpdate;
import ch.sahits.game.openpatrician.annotation.ClassCategory;
import ch.sahits.game.openpatrician.annotation.EClassCategory;
import ch.sahits.game.openpatrician.annotation.LazySingleton;
import ch.sahits.game.openpatrician.annotation.MapType;
import ch.sahits.game.openpatrician.model.Date;
import com.google.common.base.Preconditions;
import com.google.common.eventbus.AsyncEventBus;
import com.google.common.eventbus.Subscribe;
import lombok.EqualsAndHashCode;
import lombok.Getter;
import lombok.RequiredArgsConstructor;
import org.joda.time.DateTime;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;

import javax.annotation.PostConstruct;
import javax.annotation.PreDestroy;
import java.util.HashMap;
import java.util.Iterator;
import java.util.Map;
import java.util.Map.Entry;

/**
 * @author Andi Hotz, (c) Sahits GmbH, 2014
 *         Created on Dec 23, 2014
 */
@LazySingleton
@ClassCategory(EClassCategory.SINGLETON_BEAN)
public class IndividualPeriodicalUpdater implements IIndividualPeriodicalUpdater {
    @Autowired
    private Date date;
    @Autowired
    @Qualifier("serverClientEventBus")
    private AsyncEventBus clientServerEventBus;
    @Autowired
    @Qualifier("timerEventBus")
    private AsyncEventBus timerEventBus;


    /**
     * Map storing the registered objects together with their next updatable time.
     */
    @MapType(key = UpdateableObject.class, value = DateTime.class)
    private Map<UpdateableObject, DateTime> registeredUpdatableObjects = new HashMap<>();
    @PostConstruct
    private void register() {
        timerEventBus.register(this);
    }
    @PreDestroy
    private void unregister() {
        timerEventBus.unregister(this);
    }

    @Override
    public void register(Object addresse, Object target) {
        DateTime now = date.getCurrentDate();
        DateTime nextUpdate;
        UpdateableObject o = new UpdateableObject(target, addresse);
        nextUpdate = now.plusDays(1);
        registeredUpdatableObjects.put(o, nextUpdate);
    }

    @Override
    public void unregister(Object target) {
        for (Iterator<Entry<UpdateableObject, DateTime>> iterator = registeredUpdatableObjects.entrySet().iterator(); iterator.hasNext(); ) {
            Entry<UpdateableObject, DateTime> entry = iterator.next();
            if (entry.getKey().getTarget().equals(target)) {
                iterator.remove();
                break;
            }
        }
    }

    @Override
    @Subscribe
    public void handleClockTick(ClockTick event) {
        DateTime now = date.getCurrentDate();
        for (Entry<UpdateableObject, DateTime> entry : registeredUpdatableObjects.entrySet()) {
             if (now.isAfter(entry.getValue())) {
                 // send notification
                 entry.setValue(now.plus(1));
                 PeriodicalDailyUpdate evt = new PeriodicalDailyUpdate(entry.getKey().getAddressee(), entry.getKey().getTarget());
                 clientServerEventBus.post(evt);
             }
        }
    }
    // For testing
    Map<UpdateableObject, DateTime> getRegisteredUpdatableObjects() {
        return registeredUpdatableObjects;
    }
    @RequiredArgsConstructor
    @EqualsAndHashCode
    static class UpdateableObject {
        @Getter
        private final Object target;
        @Getter
        private final Object addressee;
    }

}
