package ch.sahits.game.openpatrician.engine.event.task;

import ch.sahits.game.openpatrician.annotation.ClassCategory;
import ch.sahits.game.openpatrician.annotation.EClassCategory;
import ch.sahits.game.openpatrician.annotation.Prototype;
import ch.sahits.game.openpatrician.model.DisplayMessage;
import ch.sahits.game.openpatrician.model.ICompany;
import ch.sahits.game.openpatrician.model.IHumanPlayer;
import ch.sahits.game.openpatrician.model.IPlayer;
import ch.sahits.game.openpatrician.model.city.impl.IDebt;
import ch.sahits.game.openpatrician.model.event.TargetedEvent;
import ch.sahits.game.openpatrician.model.event.TimedTask;
import com.google.common.eventbus.AsyncEventBus;
import com.thoughtworks.xstream.annotations.XStreamOmitField;
import org.joda.time.DateTime;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;

/**
 * Timed task for the loaner to collect money without a prior auction.
 * Created by andi on 26.03.17.
 */
@Prototype
@ClassCategory({EClassCategory.SERIALIZABLE_BEAN, EClassCategory.PROTOTYPE_BEAN})
public class LoanerCollectMoneyWithoutAuctionTimedTask extends TimedTask {
    @Autowired
    @Qualifier("serverClientEventBus")
    @XStreamOmitField
    private AsyncEventBus clientServerEventBus;

    private IDebt debt;
    private IPlayer debitor;

    public LoanerCollectMoneyWithoutAuctionTimedTask(IDebt debt, IPlayer debitor, DateTime deadline) {
        this.debt = debt;
        this.debitor = debitor;
        setExecutionTime(deadline);
    }

    @Override
    public void run() {
        ICompany company = debitor.getCompany();
        int requiredAmount = debt.getAmount();
        int amount = (int) Math.min(requiredAmount, company.getCash());
        company.updateCash(-amount);
        debitor.getCompany().updateCash(amount);
        if (debt.getCreditor() instanceof IPlayer) {
            ((IPlayer) debt.getCreditor()).getCompany().updateCash(amount);
            if (debt.getCreditor() instanceof IHumanPlayer) {
                DisplayMessage msg = new DisplayMessage("ch.sahits.game.openpatrician.engine.land.city.LoanerEngine.paybackAfterWithoutAuction", new Object[]{debitor.getName(), debitor.getLastName(), amount});
                TargetedEvent message = new TargetedEvent((IHumanPlayer) debt.getCreditor(), msg);
                clientServerEventBus.post(message);
            }
        }
    }
}
