package ch.sahits.game.openpatrician.engine.event.task;

import ch.sahits.game.openpatrician.annotation.ClassCategory;
import ch.sahits.game.openpatrician.annotation.DependentInitialisation;
import ch.sahits.game.openpatrician.annotation.EClassCategory;
import ch.sahits.game.openpatrician.client.ICityPlayerProxyJFX;
import ch.sahits.game.openpatrician.clientserverinterface.model.event.MarriageBrokerAnnouncementState;
import ch.sahits.game.openpatrician.clientserverinterface.service.ClientServerTaskFactory;
import ch.sahits.game.openpatrician.model.IAIPlayer;
import ch.sahits.game.openpatrician.model.IPlayer;
import ch.sahits.game.openpatrician.model.city.IChurch;
import ch.sahits.game.openpatrician.model.city.ICity;
import ch.sahits.game.openpatrician.model.city.IShipyard;
import ch.sahits.game.openpatrician.model.city.impl.IDebt;
import ch.sahits.game.openpatrician.model.city.impl.IShipBuildTask;
import ch.sahits.game.openpatrician.model.event.TimedTask;
import ch.sahits.game.openpatrician.model.people.ISeaPirate;
import ch.sahits.game.openpatrician.model.ship.EShipType;
import ch.sahits.game.openpatrician.model.ship.EShipUpgrade;
import ch.sahits.game.openpatrician.model.ship.INavigableVessel;
import ch.sahits.game.openpatrician.model.ship.IShip;
import ch.sahits.game.openpatrician.util.StartNewGameBean;
import javafx.geometry.Point2D;
import org.joda.time.DateTime;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.context.ApplicationContext;
import org.springframework.context.annotation.Lazy;
import org.springframework.stereotype.Service;

/**
 * @author Andi Hotz, (c) Sahits GmbH, 2014
 *         Created on Mar 01, 2014
 */
@Service
@Lazy
@DependentInitialisation(StartNewGameBean.class)
@ClassCategory(EClassCategory.SINGLETON_BEAN)
public class ServerSideTaskFactory extends ClientServerTaskFactory {
    @Autowired
    private ApplicationContext context;
    /**
     * Retrieve the ship repair task.
     * @return
     */
    public TimedTask getRepairTask(DateTime executionTime, IShip ship, ICityPlayerProxyJFX proxy, IShipyard shipyard) {
        Object[] args = new Object[]{executionTime, ship, proxy, shipyard};
        return (RepairTask)  context.getBean("repairTask", args);
    }

    /**
     * Retrieve the ship build task.
     * @return
     */
    public IShipBuildTask getShipBuildTask(DateTime executionTime, EShipType type, IPlayer owner, Point2D location, IShipyard shipyard) {
        Object[] args = new Object[]{executionTime, type, owner, location, shipyard};
        return (ShipBuildTask)  context.getBean("shipBuildTask", args);
    }
    /**
     * Retrieve the ship refit task.
     * @return
     */
    public TimedTask getRefitShipTask(DateTime executionTime, IShip ship, EShipUpgrade upgrade, ICityPlayerProxyJFX proxy, IShipyard shipyard) {
        Object[] args = new Object[]{executionTime, ship, upgrade, proxy, shipyard};
        return (RefitShipTask)  context.getBean("refitShipTask", args);
    }

    /**
     * Retrieve the repair task for a pirate vessel
     * @param dueDate execution time
     * @param vessel vessel to be repaired
     * @param owner pirate owner
     * @return special repair task
     */
    public TimedTask getPirateRepairTask(DateTime dueDate, INavigableVessel vessel, ISeaPirate owner, IShipyard shipyard) {
        Object[] args = new Object[]{dueDate, vessel, owner, shipyard};
        return (RepairPirateShipTask) context.getBean("repairPirateShipTask", args);
    }

    /**
     * Retrieve the repair task for an AI ship.
     * @param dueDate execution time
     * @param ship to be repaired
     * @return repair task for AI ships
     */
    public TimedTask getAIShipRepairTask(DateTime dueDate, IShip ship, IShipyard shipyard) {
        Object[] args = new Object[]{dueDate, ship, shipyard};
        return (AIRepairTask) context.getBean("aIRepairTask", args);
    }

    /**
     * Retrieve a weekly loaner check task.
     * @return
     */
    public TimedTask getWeeklyLoanerCheck() {
        return context.getBean(WeeklyLoanerCheckTask.class);
    }

    /**
     * Retrieve a weekly city check task.
     * @return
     */
    public TimedTask getWeeklyCityCheck() {
        return context.getBean(WeeklyCityCheckTask.class);
    }

    /**
     * Retrieve a timed task for the marriage offer.
     * @param player that is addressed
     * @param state of the marraiage offer announcement
     * @param deadLine when the offer will happen.
     * @return timed task to be submitted to the timed task list.
     */
    public MarriageOfferTask getMarriageOfferTask(IAIPlayer player, MarriageBrokerAnnouncementState state, DateTime deadLine) {
        return (MarriageOfferTask) context.getBean("marriageOfferTask", new Object[]{player, state, deadLine});
    }
    /**
     * Create the timed event for the plage info message
     * @param popultation of the city
     * @param duration of the plague
     * @param city where the plague is wasting
     * @param plagueStart date of the plage start
     * @param plagueEnd date of the plague end and the display date
     * @return timed task to be supplied to the timed task list.
     */
    public PlageTimedTask getPlagueTask(int popultation, int duration, ICity city, DateTime plagueStart, DateTime plagueEnd) {
        return (PlageTimedTask) context.getBean("plageTimedTask", popultation, duration, city,plagueStart, plagueEnd);
    }

    /**
     * Create the timed task for the church upgrade.
     * @param church to be upgreaded
     * @param deadline when it will happen
     * @return timed task to be supplied to the timed task list.
     */
    public ChurchUpgradeTimedTask getChurchUpgradeTask(IChurch church, DateTime deadline) {
        return (ChurchUpgradeTimedTask) context.getBean("churchUpgradeTimedTask", church, deadline);
    }

    /**
     * Create the timed task for the loaner to collect money from the debitor after an auction.
     * @param debt debt contract
     * @param deadline when this will happen.
     * @return timed task to be supplied to the timed task list.
     */
    public LoanerCollectMoneyAfterAuctionTimedTask getLoanerCollectMoneyAfterAuctionTask(IDebt debt, DateTime deadline) {
        return (LoanerCollectMoneyAfterAuctionTimedTask) context.getBean("loanerCollectMoneyAfterAuctionTimedTask", debt, deadline);
    }

    /**
     * Create a timed task for the loaner to collect money.
     * @param debt debt contract
     * @param debitor from whom to collect the money.
     * @param deadline when this will happen.
     * @return timed task to be supplied to the timed task list.
     */
    public LoanerCollectMoneyWithoutAuctionTimedTask getLoanerCollectMoneyTask(IDebt debt, IPlayer debitor, DateTime deadline) {
        return (LoanerCollectMoneyWithoutAuctionTimedTask) context.getBean("loanerCollectMoneyWithoutAuctionTimedTask", debt, debitor, deadline);
    }

    /**
     * Retrieve the timed task for the declined marriage offer.
     * @param player who declined the offer.
     * @return timed task to be supplied to the timed task list.
     */
    public MarriageOfferDeclinedTimedTask getMarriageOfferDeclinedTask(IPlayer player) {
        return (MarriageOfferDeclinedTimedTask) context.getBean("marriageOfferDeclinedTimedTask", player);
    }

}
