package ch.sahits.game.openpatrician.engine.land.city;

import ch.sahits.datastructure.GenericPair;
import ch.sahits.game.event.data.ClockTickDayChange;
import ch.sahits.game.openpatrician.annotation.ClassCategory;
import ch.sahits.game.openpatrician.annotation.DependentInitialisation;
import ch.sahits.game.openpatrician.annotation.EClassCategory;
import ch.sahits.game.openpatrician.clientserverinterface.service.MapService;
import ch.sahits.game.openpatrician.engine.AbstractEngine;
import ch.sahits.game.openpatrician.engine.land.CaptainEngine;
import ch.sahits.game.openpatrician.javafx.bindings.StaticIntegerBinding;
import ch.sahits.game.openpatrician.model.AIPlayerList;
import ch.sahits.game.openpatrician.model.AUpdatable;
import ch.sahits.game.openpatrician.model.AUpdatableCityRelated;
import ch.sahits.game.openpatrician.model.AmountableProvider;
import ch.sahits.game.openpatrician.model.Date;
import ch.sahits.game.openpatrician.model.IAIPlayer;
import ch.sahits.game.openpatrician.model.map.IMap;
import ch.sahits.game.openpatrician.model.IPlayer;
import ch.sahits.game.openpatrician.model.city.EPopulationClass;
import ch.sahits.game.openpatrician.model.city.ICity;
import ch.sahits.game.openpatrician.model.city.impl.CitiesState;
import ch.sahits.game.openpatrician.model.city.impl.TavernState;
import ch.sahits.game.openpatrician.model.collection.CityTavernRegistry;
import ch.sahits.game.openpatrician.model.people.IBaseTraveler;
import ch.sahits.game.openpatrician.model.people.IBuyer;
import ch.sahits.game.openpatrician.model.people.IConcurrent;
import ch.sahits.game.openpatrician.model.people.IContractBroker;
import ch.sahits.game.openpatrician.model.people.ICourier;
import ch.sahits.game.openpatrician.model.people.IInformant;
import ch.sahits.game.openpatrician.model.people.IPatrol;
import ch.sahits.game.openpatrician.model.people.IPerson;
import ch.sahits.game.openpatrician.model.people.IPirate;
import ch.sahits.game.openpatrician.model.people.IPirateHunter;
import ch.sahits.game.openpatrician.model.people.ISailorState;
import ch.sahits.game.openpatrician.model.people.ISeaPirate;
import ch.sahits.game.openpatrician.model.people.ISmuggler;
import ch.sahits.game.openpatrician.model.people.ITavernPerson;
import ch.sahits.game.openpatrician.model.people.IThieve;
import ch.sahits.game.openpatrician.model.people.ITrader;
import ch.sahits.game.openpatrician.model.people.ITransportTrader;
import ch.sahits.game.openpatrician.model.people.ITreasureMapOwner;
import ch.sahits.game.openpatrician.model.people.IWarehouseTenant;
import ch.sahits.game.openpatrician.model.people.impl.SeaPiratesState;
import ch.sahits.game.openpatrician.model.product.AmountablePrice;
import ch.sahits.game.openpatrician.model.product.ComputablePriceV2;
import ch.sahits.game.openpatrician.model.product.EWare;
import ch.sahits.game.openpatrician.model.product.IWare;
import ch.sahits.game.openpatrician.util.RandomNameLoader;
import ch.sahits.game.openpatrician.util.StartNewGameBean;
import com.google.common.annotations.VisibleForTesting;
import com.google.common.base.Preconditions;
import com.google.common.eventbus.AsyncEventBus;
import com.google.common.eventbus.Subscribe;
import javafx.beans.property.SimpleIntegerProperty;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.context.annotation.Lazy;
import org.springframework.stereotype.Component;

import javax.annotation.PostConstruct;
import javax.annotation.PreDestroy;
import java.util.ArrayList;
import java.util.List;
import java.util.Map.Entry;
import java.util.Optional;
import java.util.Random;

import static com.google.common.collect.Lists.newArrayList;

/**
 * Every city has a corresponding tavern engine. The guests in
 * a tavern may change on a daily basis
 * @author Andi Hotz, (c) Sahits GmbH, 2013
 * Created on Jan 19, 2013
 *
 */
@Component
@Lazy
@DependentInitialisation(StartNewGameBean.class)
@ClassCategory(EClassCategory.SINGLETON_BEAN)
public class TavernEngine  extends AbstractEngine {
	@Autowired
	private Random rnd;
	@Value("${beggar.salior.ratio}")
	private int beggarSailorRation;
	@Autowired
	private CaptainEngine captainEngine;
    @Autowired
    @Qualifier("timerEventBus")
    private AsyncEventBus timerEventBus;

	@Autowired
	private CityTavernRegistry cities;
	@Autowired
	private MapService cityUtils;
    @Autowired
    private IMap map;
    @Autowired
    private Date date;
    @Autowired
    private SeaPiratesState pirateState;
    @Autowired
    private CitiesState cityState;
    @Autowired
    private AIPlayerList aiPlayers;
    @Autowired
    private ComputablePriceV2 computablePrice;
    @Autowired
    private AmountableProvider amountableProvider;
	/*
 * Use custom names
 */
	private static RandomNameLoader firstNameLoader = new RandomNameLoader("firstnames.properties");
	private static RandomNameLoader lastNameLoader = new RandomNameLoader("lastnames.properties");
    private static RandomNameLoader pirateFirstNameLoader = new RandomNameLoader("pirate_firstnames.properties");
    private static RandomNameLoader pirateLastNameLoader = new RandomNameLoader("pirate_lastnames.properties");

	public void addCity(TavernState state, ICity city) {
        cities.put(city, state);
        int nbSailors = computeNumbersOfSailors(city);
        final ISailorState sailors = state.getSailors();
        sailors.setNumberOfSailors(nbSailors);
        handlePersonsBecomingAbsent(state);
        handlePersonsBecomingPresent(city, state);
    }

    @Override
    public List<AbstractEngine> getChildren() {
        return newArrayList(new AbstractEngine[]{captainEngine});
    }

    @PostConstruct
	private void init() {

        timerEventBus.register(this);
	}
	@PreDestroy
	private void unregister() {
        timerEventBus.unregister(this);
	}

	private int computeNumbersOfSailors(ICity city) {
		int nbBeggars = city.getPopulation(EPopulationClass.BEGGAR);
		int nbSailors = (int)(nbBeggars / beggarSailorRation*rnd.nextDouble()*2);
		return nbSailors;
	}

    @Subscribe
    public void handleDayChange(ClockTickDayChange event) {
        for (Entry<ICity, TavernState> entry : cities.entrySet()) {
            ICity city = entry.getKey();
            TavernState state = entry.getValue();
            int nbSailors = computeNumbersOfSailors(city);
            state.getSailors().setNumberOfSailors(nbSailors);
            handlePersonsBecomingAbsent(state);
            handlePersonsBecomingPresent(city, state);
        }

    }
	/**
	 * Randomized state update for persons that are not present.
	 */
	private void handlePersonsBecomingAbsent(TavernState state) {
		List<IPerson> present = state.presentPersonsProperty();
        List<ITavernPerson> leavingPersons = newArrayList();
		for (IPerson p : present) {
			if (p instanceof ITavernPerson) {
				ITavernPerson person = (ITavernPerson) p;
				double maxPresence = Math.abs(person.getMaxDaysPresent());
				double presenceTime = Math.abs(person.getNumberOfDaysSinceArrival());
				double absence = rnd.nextDouble()*presenceTime/maxPresence;
				if (absence > 0.75) {
                    leavingPersons.add(person);
                }
			}
			// non tavern persons cannot become absent on their own
		} // end for present
		synchronized (state) {
			for (ITavernPerson person : leavingPersons) {
				person.leave();
			}
		}
    }
	/**
	 * Randomized state update for persons that are present.
	 */
	private void handlePersonsBecomingPresent(ICity city, TavernState state) {
		List<ITavernPerson> absent = state.absentPersonProperty();
        List<ITavernPerson> arrivingPersons = newArrayList();
		for (ITavernPerson person : absent) {
			double maxPresence = Math.abs(person.getMaxDaysPresent());
			double maxNbAbsence = Math.abs(person.getMaxDaysAbsent());
			double absentTime = person.getNumberOfDaysSinceArrival()-maxPresence;
			double presence = rnd.nextDouble()*2*(absentTime/maxNbAbsence);
			if (presence > 0.5) {
				updateCityRelatedPersons(person, city);
                updatePersons(person);
                arrivingPersons.add(person);
            }
		} // end for absent
		synchronized (state) {
			for (ITavernPerson person : arrivingPersons) {
				person.arrive();
			}
		}
    }
    @VisibleForTesting
    void updatePersons(ITavernPerson person) {
        if (isAnnotatedType(person, AUpdatable.class)) {
            if (person instanceof IBuyer) {
                IBuyer buyer = (IBuyer) person;
                buyer.setName(firstNameLoader.getRandomName()+" "+lastNameLoader.getRandomName());
                int days = rnd.nextInt(5);
                buyer.setDeadline(date.getCurrentDate().plusDays(5+days));
                int wareIndex = rnd.nextInt(EWare.values().length);
                IWare ware = EWare.values()[wareIndex];
                buyer.setWare(ware);
                int amount = rnd.nextInt(21)+3; // TODO aho Jan 27, 2013: differentiate between bale and barrel
                int price = computablePrice.buyPrice(ware, new SimpleIntegerProperty(5), new StaticIntegerBinding(amount));
                AmountablePrice<IWare> amt = amountableProvider.createWareAmountable();
                amt.add(amount, price);
                buyer.setAmountablePrice(amt);
            } else if (person instanceof IWarehouseTenant) {
                IWarehouseTenant tenant = (IWarehouseTenant) person;
                tenant.setName(firstNameLoader.getRandomName()+" "+lastNameLoader.getRandomName());
                tenant.setAmount(rnd.nextInt(800)+100);
            } else if (person instanceof IInformant) {
                IInformant informant = (IInformant) person;
                int val = rnd.nextInt(2);
                informant.setShowMapIfAvailable(rnd.nextInt(3) == 0);

                informant.setPrice(rnd.nextInt(1000)+120); // consider age of the game
                updateWareNeedOrSurplus(val, informant);
                int nextVal = rnd.nextInt(2);
                if (val != nextVal) {
                    updateWareNeedOrSurplus(nextVal, informant);
                }

                boolean noInformationYet = !informant.getWareNeeded().isPresent() && !informant.getWareSurplus().isPresent();
                if (rnd.nextBoolean() || noInformationYet) {
                    ISeaPirate pirate = pirateState.findSuccessfulPirate();
                    ICity nearest = cityUtils.findNearestCity(pirate.getShip().getLocation());
                    informant.setPirateLocation(pirate, nearest);
                }
            } else if (person instanceof IConcurrent) {
                IConcurrent concurrent = (IConcurrent) person;
                List<IAIPlayer> players = aiPlayers.getAll();
                final int index = rnd.nextInt(players.size());

                IPlayer p1 = players.get(index);
                int other = rnd.nextInt(players.size());
                while (other == index) {
                    other = rnd.nextInt(players.size());
                }
                IPlayer p2 = players.get(other);
                concurrent.setClientPlayer(p1);
                concurrent.setConcurrentPlayer(p2);
                concurrent.setPremium(rnd.nextInt(8500)+1500);
            } else if (person instanceof IContractBroker) {
                IContractBroker broker = (IContractBroker) person;
                broker.setName(firstNameLoader.getRandomName()+" "+lastNameLoader.getRandomName()); // TODO use brooker names
            } else if (person instanceof ITreasureMapOwner) {
                ITreasureMapOwner owner = (ITreasureMapOwner) person;
                owner.setName(firstNameLoader.getRandomName()+" "+lastNameLoader.getRandomName());
                owner.setPrice(rnd.nextInt(1500)+1000);
            } else if (person instanceof IThieve) {
                IThieve thieve = (IThieve) person;
                thieve.setPrice(rnd.nextInt(400)+100);
            } else if (person instanceof IPirate) {
                IPirate pirate = (IPirate) person;
                pirate.setName(pirateFirstNameLoader.getRandomName()+" "+pirateLastNameLoader.getRandomName());
                pirate.setFollowers(rnd.nextInt(15)+12);
                pirate.setProfitShare((rnd.nextInt(7)*5+35)/100.0);
            } else if (person instanceof IPirateHunter) {
                IPirateHunter hunter = (IPirateHunter) person;
                hunter.setClientName(firstNameLoader.getRandomName()+" "+lastNameLoader.getRandomName());
                hunter.setPirateName(firstNameLoader.getRandomName()+" "+lastNameLoader.getRandomName());
                hunter.setPremium(rnd.nextInt(5000)+1434);
            }
        }
    }
    private void updateWareNeedOrSurplus(int val, IInformant informant) {
        switch (val) {
            case 0: // need
                Optional<GenericPair<ICity, IWare>> missing = cityState.findCityWithMostNeededWare();
                if (missing.isPresent()) {
                    GenericPair<ICity, IWare> pair = missing.get();
                    informant.setNeededWare(pair.getFirst(), pair.getSecond());
                }
                break;
            case 1: // surplus
                Optional<GenericPair<ICity, IWare>> surplus = cityState.findCityWithSurplusWare();
                if (surplus.isPresent()) {
                    GenericPair<ICity, IWare> pair = surplus.get();
                    informant.setWareSurplus(pair.getFirst(), pair.getSecond());
                }
                break;
            default:
                throw new IllegalStateException("Invalid random number: "+val);
        }
    }

    @VisibleForTesting
	void updateCityRelatedPersons(ITavernPerson person, ICity city) {
		if (isAnnotatedType(person, AUpdatableCityRelated.class)) {
			  if (person instanceof ICourier) {
                  ICourier courier = (ICourier) person;
                  courier.setName(firstNameLoader.getRandomName() + " " + lastNameLoader.getRandomName());
                  courier.setPremium(rnd.nextInt(2000) + 4300);
                  int nbDestinations = rnd.nextInt(3) + 3;
                  ICity[] destinations = new ICity[nbDestinations];
                  courier.setDestinations(destinations);
                  fillDestinations(city, destinations);
                  courier.setCurrentDestIndex(0);
              } else if (person instanceof ITransportTrader) {
                  ITransportTrader transportTrader = (ITransportTrader) person;
                  transportTrader.setName(firstNameLoader.getRandomName() + " " + lastNameLoader.getRandomName());
                  transportTrader.setPremium(rnd.nextInt(700) + 300);
                  transportTrader.setNeededCapacity(rnd.nextInt(40) + 4);
                  List<ICity> visited = new ArrayList<>();
                  visited.add(city);
                  ICity next = cityUtils.findNearbyCityRepeated(city, 1, 1024, visited);
                  transportTrader.setDestination(next);
              } else if (person instanceof ITrader) {
                  ITrader trader = (ITrader) person;
                  EWare ware = EWare.values()[rnd.nextInt(EWare.values().length)];
                  final AmountablePrice<IWare> amountablePrice = city.getWare(ware);
                  int availableAmount = amountablePrice.getAmount();
                  while (availableAmount == 0) {
                      ware = EWare.values()[rnd.nextInt(EWare.values().length)];
                      availableAmount = city.getWare(ware).getAmount();
                  }
                  int amount;
                  if (ware.getSizeAsBarrels() == 1) {
                      amount = rnd.nextInt(100);
                  } else {
                      amount = rnd.nextInt(20);
                  }
                  amount = Math.max(1, amount);
                  trader.setAmount(amount);
                  double variance = rnd.nextDouble() * 0.25 - 0.125;
                  int newPrice = (int) Math.abs(Math.rint(variance * computablePrice.sellPrice(ware, new SimpleIntegerProperty(availableAmount), new StaticIntegerBinding(amount))));
                  trader.setAvgPricePerItem(newPrice);
                  trader.setWare(ware);
              } else if (person instanceof ISmuggler) {
                  ISmuggler smuggler = (ISmuggler) person;
                  smuggler.setAmount(rnd.nextInt(200) + 16);
                  smuggler.setPremium(rnd.nextInt(1000) + 180);
                  List<ICity> cities = map.getCities();
                  ICity nextCity = cities.get(rnd.nextInt(cities.size()));
                  while (nextCity.equals(city)) {
                      nextCity = cities.get(rnd.nextInt(cities.size()));
                  }
                  smuggler.setDestination(nextCity);
              } else if (person instanceof IPatrol) {
                  IPatrol patrol = (IPatrol) person;
                  patrol.setName(firstNameLoader.getRandomName()+" "+lastNameLoader.getRandomName());
                  patrol.setPremium(rnd.nextInt(2000)+4300);
                  patrol.setBonus(rnd.nextInt(2000)+2000);
                  ICity[] destinations = new ICity[3];
                  patrol.setDestinations(destinations);
                  fillDestinations(city, destinations);
                  patrol.setCurrentDestIndex(0);
              } else if (person instanceof IBaseTraveler) {
                  int premium = rnd.nextInt(1543)+324;
                  ((IBaseTraveler)person).setPremium(premium);
                  final ICity destination = cityUtils.findRandomCity(city);
                  Preconditions.checkNotNull(destination);
                  ((IBaseTraveler)person).setDestination(destination);
              }
		}
	}

    private void fillDestinations(ICity city, ICity[] destinations) {
        List<ICity> visited = new ArrayList<>();
        visited.add(city);
        ICity curCity = city;
        for (int i = 0; i < destinations.length; i++) {
            ICity next = cityUtils.findNearbyCityRepeated(curCity, .2, 1024, visited);
            destinations[i] = next;
            visited.add(next);
            curCity = next;
        }
    }

    private boolean isAnnotatedType(ITavernPerson person, Class annotation) {
		return person.getClass().getAnnotation(annotation) != null;
	}

}
