package ch.sahits.game.openpatrician.engine.player.strategy;

import ch.sahits.game.event.data.ShipEntersPortEvent;
import ch.sahits.game.openpatrician.annotation.ClassCategory;
import ch.sahits.game.openpatrician.annotation.EClassCategory;
import ch.sahits.game.openpatrician.engine.land.city.ShipyardEngine;
import ch.sahits.game.openpatrician.model.DisplayMessage;
import ch.sahits.game.openpatrician.model.IHumanPlayer;
import ch.sahits.game.openpatrician.model.IPlayer;
import ch.sahits.game.openpatrician.model.city.ICity;
import ch.sahits.game.openpatrician.model.city.IShipyard;
import ch.sahits.game.openpatrician.model.event.TargetedEvent;
import ch.sahits.game.openpatrician.model.people.IShipOwner;
import ch.sahits.game.openpatrician.model.player.IAIShipRepairStrategy;
import ch.sahits.game.openpatrician.model.ship.IConvoy;
import ch.sahits.game.openpatrician.model.ship.INavigableVessel;
import ch.sahits.game.openpatrician.model.ship.IShip;
import com.google.common.base.Preconditions;
import com.google.common.eventbus.AsyncEventBus;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;

import java.util.ArrayList;
import java.util.List;

/**
 * Base implementation for the repair strategy. The repairing is the same for all strategies,
 * only the decission when it should happen differs.
 */
@ClassCategory(EClassCategory.SINGLETON_BEAN)
public abstract class BaseRepairStrategy implements IAIShipRepairStrategy {
    private final Logger logger = LogManager.getLogger(getClass());
    @Autowired
    private ShipyardEngine shipyardEngine;
    @Autowired
    @Qualifier("serverClientEventBus")
    private AsyncEventBus clientServerEventBus;
    @Override
    public void repair(INavigableVessel vessel, ICity city) {
        Preconditions.checkState(vessel.getLocation().equals(city.getCoordinates()), "The vessel "+vessel.getName()+" is not in the city "+city.getCoordinates()+" but at "+vessel.getLocation());
        IShipyard shipyard = city.getCityState().getShipyardState();
        if (vessel instanceof IConvoy) {
            List<IShip> ships = dissolveConvoy((IConvoy) vessel, city);
            for (IShip ship : ships) {
                shipyardEngine.repairAIShip(shipyard, ship);
            }
        } else {
            shipyardEngine.repairAIShip(shipyard, (IShip)vessel);
        }
        logger.debug("Repair vessel {} in {}", vessel.getName(), city.getName());
    }

    /**
     * Dissolve the convoy and return the list of ships of the owner.
     * @param convoy to be disolved
     * @param city where the convoy is disolved
     * @return list of ships that were in the convoy that belong to the convoys owner.
     */
    List<IShip> dissolveConvoy(IConvoy convoy, ICity city) {
        ArrayList<IShip> shipList = new ArrayList<>();
        for (IShip ship : convoy.getShips()) {
            ship.setAvailable(true);
            IShipOwner owner = ship.getOwner();
            convoy.removeShip(ship);
            if (owner instanceof IHumanPlayer) {
                DisplayMessage msg = new DisplayMessage("ch.sahits.game.graphic.display.dialog.sea.ConvoyDialog.convoyDissolveNotification", new Object[]{convoy.getName(), ship.getName(), city.getName()});
                TargetedEvent targetMsg = new TargetedEvent((IHumanPlayer) owner, msg);
                clientServerEventBus.post(targetMsg);
            }
            if (owner instanceof IPlayer) {
                IPlayer player = (IPlayer) owner;
                player.addSelectableVessel(ship);
            }
            if (owner.equals(convoy.getOwner())) {
                shipList.add(ship);
            }
            clientServerEventBus.post(new ShipEntersPortEvent(ship, city));
        }
        return shipList;
    }
}
