package ch.sahits.game.openpatrician.engine.player.strategy;

import ch.sahits.game.openpatrician.annotation.ClassCategory;
import ch.sahits.game.openpatrician.annotation.EClassCategory;
import ch.sahits.game.openpatrician.annotation.LazySingleton;
import ch.sahits.game.openpatrician.model.IAIPlayer;
import ch.sahits.game.openpatrician.model.city.ICity;
import ch.sahits.game.openpatrician.model.ship.EShipType;
import ch.sahits.game.openpatrician.model.ship.INavigableVessel;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.context.ApplicationContext;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;
import java.util.Optional;

/**
 * This strategy builds the biggest possible ship type in the hometown.
 * @author Andi Hotz, (c) Sahits GmbH, 2016
 *         Created on Jul 30, 2016
 */
@LazySingleton
@ClassCategory(EClassCategory.SINGLETON_BEAN)
public class HometownBiggestShipConstructionSelectionStrategy extends BaseShipConstructionSelectionStrategy {
    @Autowired
    private ApplicationContext context;

    public HometownBiggestShipConstructionSelectionStrategy() {
        super(100000, 50, 15000);
    }



    @Override
    public EShipType shouldBuildShipType(IAIPlayer player) {
        // Which ships can be built in hometown
        ICity where = getBuildLocation(player);
        List<EShipType> shipTypes = new ArrayList<>(Arrays.asList(where.getCityState().getShipyardState().getBuildableShipTypes()));
        int fleetCapacity = getFleetCapacity(player.getFleet());
        int maxTotalCargoCapacity = getMaxTotalCargoCapacity();
        double capacityOfMaxPercent = fleetCapacity *1.0 / maxTotalCargoCapacity;
        // if total capacity is below 70% of the max capacity => Holk
        if (capacityOfMaxPercent >= 0.7) {
            return biggestShipType(shipTypes);
        }
        // if total capacity is below 30% of the max capacity => Cog
        if (capacityOfMaxPercent <= 0.3) {
            shipTypes.remove(EShipType.HOLK);
        }
        // if total capacity is below 10% of the max capacity => Crayer
        if (capacityOfMaxPercent <= 0.1) {
            shipTypes.remove(EShipType.COG);
        }
        // if total capacity is below 5% of the max capacity => Snaikka
        if (capacityOfMaxPercent <= 0.05) {
            shipTypes.remove(EShipType.CRAYER);
        }
        return biggestShipType(shipTypes);
    }

    private EShipType biggestShipType(List<EShipType> shipTypes) {
        EShipType[] sizedTypes = {EShipType.HOLK, EShipType.COG, EShipType.CRAYER, EShipType.SNAIKKA};
        for (EShipType type : sizedTypes) {
            if (shipTypes.contains(type)) {
                return type;
            }
        }
        return null;
    }

    /**
     * {@inheritDoc}
     * The build location is always the players hometown.
     * @param player
     * @return
     */
    @Override
    public ICity getBuildLocation(IAIPlayer player) {
        return player.getHometown();
    }

    @Override
    public void initShipConstruction(IAIPlayer player, ICity buildLocation, EShipType shipType) {
        CollectConstructionWareStrategy strategy = null;
        switch (shipType) {
            case HOLK:
                strategy = context.getBean(CollectConstructionHometownHolkWareStrategy.class);
                break;
            case COG:
                strategy = context.getBean(CollectConstructionHometownCogWareStrategy.class);
                break;
            case CRAYER:
                strategy = context.getBean(CollectConstructionHometownCrayerWareStrategy.class);
                break;
            case SNAIKKA:
                strategy = context.getBean(CollectConstructionHometownSnaikkaWareStrategy.class);
                break;
        }
        Optional<INavigableVessel> vessel = selectCollectingVessel(player, shipType);
        if (vessel.isPresent()) {
            strategy.initialzeTradeCycle(player, vessel.get());
        }
    }


}
