package ch.sahits.game.openpatrician.engine.player.strategy;

import ch.sahits.game.openpatrician.annotation.ClassCategory;
import ch.sahits.game.openpatrician.annotation.EClassCategory;
import ch.sahits.game.openpatrician.annotation.LazySingleton;
import ch.sahits.game.openpatrician.clientserverinterface.service.MapProxy;
import ch.sahits.game.openpatrician.clientserverinterface.service.MapService;
import ch.sahits.game.openpatrician.clientserverinterface.service.ShipService;
import ch.sahits.game.openpatrician.engine.player.ProductionChainMissionData;
import ch.sahits.game.openpatrician.engine.player.tradesteps.AggregatedCheckedBuyTradeStep;
import ch.sahits.game.openpatrician.engine.player.tradesteps.AggregatesDumpTradeStep;
import ch.sahits.game.openpatrician.engine.player.tradesteps.AggregatesSellTradeStep;
import ch.sahits.game.openpatrician.engine.player.tradesteps.CheckForRepairTradeStep;
import ch.sahits.game.openpatrician.engine.player.tradesteps.CheckHireCaptainTradeStep;
import ch.sahits.game.openpatrician.engine.player.tradesteps.GuildJoinTradeStep;
import ch.sahits.game.openpatrician.engine.player.tradesteps.HireSailorsStep;
import ch.sahits.game.openpatrician.engine.player.tradesteps.PayBackLoanTradeStep;
import ch.sahits.game.openpatrician.engine.player.tradesteps.TakeLoanTradeStep;
import ch.sahits.game.openpatrician.engine.player.tradesteps.TravelToTradeStep;
import ch.sahits.game.openpatrician.model.IAIPlayer;
import ch.sahits.game.openpatrician.model.city.ICity;
import ch.sahits.game.openpatrician.model.event.IShipEntersPortEvent;
import ch.sahits.game.openpatrician.model.player.ICityProductionConsumptionKnowledge;
import ch.sahits.game.openpatrician.model.player.IProductionConsumptionKnowledge;
import ch.sahits.game.openpatrician.model.product.EWare;
import ch.sahits.game.openpatrician.model.product.IWare;
import ch.sahits.game.openpatrician.model.product.ProductionChain;
import ch.sahits.game.openpatrician.model.ship.INavigableVessel;
import com.google.common.eventbus.AsyncEventBus;
import com.thoughtworks.xstream.annotations.XStreamOmitField;
import javafx.util.Pair;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;

import javax.annotation.PostConstruct;
import javax.annotation.PreDestroy;
import java.util.ArrayList;
import java.util.Collections;
import java.util.LinkedList;
import java.util.List;
import java.util.Optional;

/**
 * This trade strategy focuses on a production chain with supplying wares for the
 * production and selling the produced wares to a different city.
 * @author Andi Hotz, (c) Sahits GmbH, 2016
 *         Created on Jun 19, 2016
 */
@LazySingleton
@ClassCategory(EClassCategory.SINGLETON_BEAN)
public class ProductionChainTradeStrategy extends BasePlayerTradeStrategy {
    private final Logger logger = LogManager.getLogger(getClass());
    @Autowired
    @XStreamOmitField
    private ProductionChain productionChain;
    @Autowired
    @XStreamOmitField
    private ShipService shipService;
    @Autowired
    @Qualifier("serverClientEventBus")
    private AsyncEventBus clientServerEventBus;
    @Autowired
    @XStreamOmitField
    private MapService mapService;
    @Autowired
    @XStreamOmitField
    private MapProxy mapProxy;

    public ProductionChainTradeStrategy() {
        this.tradeStrategyType = EAITradeStrategyType.PRODUCTION_CHAIN;
    }

    @PostConstruct
    private void init() {
        clientServerEventBus.register(this);
    }
    @PreDestroy
    private void destroy() {
        clientServerEventBus.unregister(this);
    }

    @Override
    public void initialzeTradeCycle(IAIPlayer player, INavigableVessel vessel) {
        // Get list of wares that have production chain
        List<IWare> waresWithProductionChain = productionChain.getWaresWithProductionChain();
        waresWithProductionChain.remove(EWare.BRICK);
        waresWithProductionChain.remove(EWare.PITCH);
        waresWithProductionChain.remove(EWare.LEATHER);
        ICity currentLocation = getCurrentCity(vessel);
        ICity productionLocation = null;
        IWare ware = null;
        while (productionLocation == null) {
            // select ware
            Collections.shuffle(waresWithProductionChain);
            ware = waresWithProductionChain.get(0);
            // select production location
            for (IWare good : currentLocation.getEffectiveProduction()) {
                if (ware.equals(good)) {
                    productionLocation = currentLocation;
                    break;
                }
            }
            if (productionLocation == null) {
                for (ICity city : mapProxy.getAllCities()) {
                    for (IWare good : city.getEffectiveProduction()) {
                        if (ware.equals(good)) {
                            productionLocation = city;
                            break;
                        }
                    }
                    if (productionLocation != null) {
                        break;
                    }
                }
            }
        }

        // set mission data
        ProductionChainMissionData missionData = new ProductionChainMissionData();
        missionData.setProduceWare(ware);
        missionData.setProductionLocation(productionLocation);
        player.setTradeMission(vessel, missionData);
        // check if needed ware can be bought at current location
        List<IWare> requiredWares = productionChain.getRequiredWares(ware);
        boolean findAllHere = true;
        for (IWare requiredWare : requiredWares) {
            if (!mapService.produces(currentLocation, requiredWare)) {
                findAllHere = false;
                break;
            }
        }
        if (findAllHere && !currentLocation.equals(productionLocation)) {
            AggregatedCheckedBuyTradeStep buyStepHometown = createAggregatedCheckedBuyTradeStep(vessel, currentLocation, requiredWares);
            append(player, vessel, buyStepHometown);
            CheckForRepairTradeStep repairStep = createCheckRepairStep(vessel, currentLocation);
            append(player, vessel, repairStep);
            HireSailorsStep hireSailors = createHireSailorStep(vessel, currentLocation);
            append(player, vessel, hireSailors);
            TravelToTradeStep travelTo = createTravelToStep(vessel, productionLocation);
            append(player, vessel, travelTo);

        } else {
            // check where to by required wares
            IProductionConsumptionKnowledge knowledge = player.getProductionAndConsumptionKnowledge();
            ICity destination = findDestinationToBuyRequiredProductionWares(player, requiredWares, vessel, Optional.of(currentLocation));
            buyWaresAndTravelToNextStop(player, vessel, currentLocation, knowledge, destination);

        }
    }

    private void buyWaresAndTravelToNextStop(IAIPlayer player, INavigableVessel vessel, ICity currentLocation, IProductionConsumptionKnowledge knowledge, ICity nextStop) {
        ICityProductionConsumptionKnowledge cityKnowledge = knowledge.getKnowlege(nextStop);
        List<Pair<IWare, Number>> sortedNeedsHometown = getMostNeededWares(cityKnowledge, vessel);
        ArrayList<IWare> waresOfInterestNextStop = findWaresOfInterest(sortedNeedsHometown);
        AggregatedCheckedBuyTradeStep buyStepHometown = createAggregatedCheckedBuyTradeStep(vessel, currentLocation, waresOfInterestNextStop);
        append(player, vessel, buyStepHometown);
        CheckForRepairTradeStep repairStep = createCheckRepairStep(vessel, currentLocation);
        append(player, vessel, repairStep);
        HireSailorsStep hireSailors = createHireSailorStep(vessel, currentLocation);
        append(player, vessel, hireSailors);
        TravelToTradeStep travelTo = createTravelToStep(vessel, nextStop);
        append(player, vessel, travelTo);
    }

    @Override
    public void handleShipArrivesInPort(IShipEntersPortEvent event) {
        INavigableVessel vessel = event.getShip();
        ICity city = event.getCity();
        IAIPlayer player = (IAIPlayer) vessel.getOwner();
        ProductionChainMissionData missionData = getMissionData(player, vessel);
        boolean inProductionTown = missionData.getProductionLocation().equals(city);
        IWare produceWare = missionData.getProduceWare();
        List<IWare> requiredWares = new LinkedList();
        requiredWares.addAll(productionChain.getRequiredWares(produceWare));
        IProductionConsumptionKnowledge knowledge = player.getProductionAndConsumptionKnowledge();
        ICity nextStop;
        List<IWare> buyWares = new ArrayList<>();
        List<IWare> sellWares;
        if (inProductionTown) {
            sellWares = getLoadedWares(vessel);
            nextStop = findDestinationToBuyRequiredProductionWares(player, requiredWares, vessel, Optional.of(city));
            ICityProductionConsumptionKnowledge cityKnowledge = knowledge.getKnowlege(nextStop);
            if (isNeeded(cityKnowledge, produceWare)) {
                buyWares.addAll(getWaresNeedIn(knowledge, knowledge.getKnowlege(city), nextStop));
                if (!buyWares.contains(produceWare)) {
                    buyWares.add(produceWare);
                }
            } else {
                nextStop = knowledge.findCitiesWithNeedMinimalDistance(city, produceWare, vessel).get(0);
                buyWares.addAll(getWaresNeedIn(knowledge, knowledge.getKnowlege(city), nextStop));
                if (!buyWares.contains(produceWare)) {
                    buyWares.add(produceWare);
                }
            }
        } else {
            sellWares = new LinkedList<>();
            sellWares.addAll(getLoadedWares(vessel));
            sellWares.removeAll(requiredWares); // Retain the wares needed for the production chain
            // can we buy the last required ware here
            ICityProductionConsumptionKnowledge cityKnowledge = knowledge.getKnowlege(city);
            List<IWare> loadedWares = new LinkedList<>();
            loadedWares.addAll(getLoadedWares(vessel));
            requiredWares.removeAll(loadedWares);
            List<IWare> canSupply = new ArrayList<>();
            for (IWare requiredWare : requiredWares) {
                int production = cityKnowledge.getProductionAmount(requiredWare);
                int consumtion = cityKnowledge.getConsumptionAmount(requiredWare);
                if (production - consumtion > 0) {
                    canSupply.add(requiredWare);
                }
            }
            requiredWares.removeAll(canSupply);
            if (requiredWares.isEmpty()) {
                nextStop = missionData.getProductionLocation();
                buyWares.addAll(getWaresNeedIn(knowledge, knowledge.getKnowlege(city), nextStop));
                buyWares.addAll(canSupply);
            } else {
                nextStop = findDestinationToBuyRequiredProductionWares(player, requiredWares, vessel, Optional.of(city));
                logger.debug("Add buy steps for city {} ", city.getName());
                buyWares.addAll(getWaresNeedIn(knowledge, knowledge.getKnowlege(city), nextStop));
            }
        }
        if (inProductionTown) {
            AggregatesDumpTradeStep dumpStep = createAggregatedDumpStep(vessel, city, sellWares);
            append(player, vessel, dumpStep);
        } else {
            AggregatesSellTradeStep sellStep = createAggregatedSellStep(vessel, city, sellWares);
            append(player, vessel, sellStep);
        }

        PayBackLoanTradeStep payBackLoan = createPaybackLoanStep(player, city);
        append(player, vessel, payBackLoan);
        CheckHireCaptainTradeStep hireCaptain = createHireCaptain(vessel, city, player);
        append(player, vessel, hireCaptain);
        GuildJoinTradeStep joinTradeStep = createJoinGuildTradeStep(vessel, city, player);
        append(player, vessel, joinTradeStep);
        TakeLoanTradeStep takeLoanStep = createCheckAndTakeLoanStep(player, city);
        append(player, vessel, takeLoanStep);
        AggregatedCheckedBuyTradeStep buyStepHometown = createAggregatedCheckedBuyTradeStep(vessel, city, buyWares);
        append(player, vessel, buyStepHometown);
        CheckForRepairTradeStep repairStep = createCheckRepairStep(vessel, city);
        append(player, vessel, repairStep);
        HireSailorsStep hireSailors = createHireSailorStep(vessel, city);
        append(player, vessel, hireSailors);
        TravelToTradeStep travelTo = createTravelToStep(vessel, nextStop);
        append(player, vessel, travelTo);
        executeTradeSteps(player, vessel);
    }

    private ICity getCurrentCity(INavigableVessel vessel) {
        Optional<ICity> optCity = shipService.findCity(vessel);
        if (!optCity.isPresent()) {
            logger.debug("The vessel {} is not in any city but {}", vessel.getName(), vessel.getLocation());
        }
        // sell all loaded wares
        return optCity.get();
    }

    private ProductionChainMissionData getMissionData(IAIPlayer player, INavigableVessel vessel) {
        return (ProductionChainMissionData) player.getTradeMission(vessel);
    }

}
