package ch.sahits.game.openpatrician.engine.player.tradesteps;

import ch.sahits.game.openpatrician.annotation.ClassCategory;
import ch.sahits.game.openpatrician.annotation.EClassCategory;
import ch.sahits.game.openpatrician.annotation.ListType;
import ch.sahits.game.openpatrician.annotation.Prototype;
import ch.sahits.game.openpatrician.model.city.ICity;
import ch.sahits.game.openpatrician.model.product.ITradeStep;
import ch.sahits.game.openpatrician.model.product.IWare;
import ch.sahits.game.openpatrician.model.ship.INavigableVessel;
import com.google.common.base.Preconditions;
import com.thoughtworks.xstream.annotations.XStreamOmitField;
import lombok.Setter;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.context.ApplicationContext;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

/**
 * Trade step aggregating several wares to buy.
 */
@ClassCategory({EClassCategory.SERIALIZABLE_BEAN, EClassCategory.PROTOTYPE_BEAN})
@Prototype
public class AggregatedCheckedBuyTradeStep implements ITradeStep {
    @Setter
    private INavigableVessel vessel;
    @Setter
    private ICity city;
    @Setter
    private boolean executeNext = true;
    @Autowired
    @XStreamOmitField
    private ApplicationContext context;
    @ListType(BuyTradeStep.class)
    private ArrayList<CheckedBuyTradeStep> tradeSteps = new ArrayList<>();

    public void addBuyStep(IWare ware, int maxPrice) {
        CheckedBuyTradeStep buyStep = context.getBean(CheckedBuyTradeStep.class);
        buyStep.setExecuteNext(executeNext);
        buyStep.setMaxBuyPrice(maxPrice);
        buyStep.setCity(city);
        buyStep.setVessel(vessel);
        buyStep.setWare(ware);
        tradeSteps.add(buyStep);
    }

    /**
     * Sort the list of wares that should be bought so that the
     * wares with the least amount are at the begining of the list.
     * @return priority sorted ware list.
     */
    private List<CheckedBuyTradeStep> getPriorityList() {
        ArrayList<CheckedBuyTradeStep> list = new ArrayList<>();
        Map<CheckedBuyTradeStep,Integer> loadedWares = new HashMap<>();
        for (CheckedBuyTradeStep step : tradeSteps) {
            IWare ware = step.getWare();
            int amount = vessel.getWare(ware).getAmount();
            loadedWares.put(step, amount);
        }
        loadedWares.entrySet()
                .stream()
                .sorted(Map.Entry.<CheckedBuyTradeStep, Integer>comparingByValue())
                .forEach(entry -> list.add(entry.getKey()));
        return list;
    }

    @Override
    public boolean execute() {
        Preconditions.checkArgument(vessel.getLocation().equals(city.getCoordinates()), "The vessel "+vessel.getName()+" is not in city "+city.getName()+" but at "+vessel.getLocation());
        List<CheckedBuyTradeStep> priorityList = getPriorityList();
        for (CheckedBuyTradeStep buyTradeStep : priorityList) {
            buyTradeStep.execute();
            if (vessel.getCapacity() == 0) {
                break;
            }
        }
        return executeNext;
    }

    @Override
    public String toString() {
        StringBuilder sb = new StringBuilder();
        tradeSteps.stream().forEach(step -> sb.append(step.getWare().name()).append(" "));
        return "AggregatedCheckedBuyTradeStep in "+city.getName()+" buying: "+sb.toString().toString();
    }
}
