package ch.sahits.game.openpatrician.engine.player.tradesteps;

import ch.sahits.game.javafx.bindings.ConstantIntegerBinding;
import ch.sahits.game.openpatrician.annotation.ClassCategory;
import ch.sahits.game.openpatrician.annotation.EClassCategory;
import ch.sahits.game.openpatrician.annotation.Prototype;
import ch.sahits.game.openpatrician.model.ICompany;
import ch.sahits.game.openpatrician.model.IPlayer;
import ch.sahits.game.openpatrician.model.building.ITradingOffice;
import ch.sahits.game.openpatrician.model.city.ICity;
import ch.sahits.game.openpatrician.model.product.ComputablePriceV2;
import ch.sahits.game.openpatrician.model.product.ITradeStep;
import ch.sahits.game.openpatrician.model.product.IWare;
import com.thoughtworks.xstream.annotations.XStreamOmitField;
import javafx.beans.property.SimpleIntegerProperty;
import lombok.AccessLevel;
import lombok.Getter;
import lombok.Setter;
import org.springframework.beans.factory.annotation.Autowired;

/**
 * Buy trade step to buy wares and store it in the trading office.
 * @author Andi Hotz, (c) Sahits GmbH, 2016
 *         Created on Dec 21, 2016
 */
@ClassCategory({EClassCategory.SERIALIZABLE_BEAN, EClassCategory.PROTOTYPE_BEAN})
@Prototype
public class BuyTradingOfficeTradeStep implements ITradeStep {
    @Setter
    @Getter(AccessLevel.PROTECTED)
    private ICity city;
    @Setter
    @Getter
    private ITradingOffice tradingOffice;
    @Setter
    @Getter
    private IWare ware;
    @Autowired
    @XStreamOmitField
    private ComputablePriceV2 computablePrice;
    @Setter
    private int maxBuyPrice;

    /**
     * Execute the transaction of buying the ware from the city and moving it in the trading office
     * @param availableAmount amount available in the city
     * @param player doing the transaction
     * @param company of the player
     * @param affordableAmount amount that should be bought
     * @return true if the next trade step can be executed.
     */
    private boolean buyWare(int availableAmount, IPlayer player, ICompany company, int affordableAmount) {
        if (affordableAmount > 0 && tradingOffice != null) {
            int buyPrice = computablePrice.buyPrice(ware, new SimpleIntegerProperty(availableAmount), new ConstantIntegerBinding(affordableAmount));
            company.updateCash(-buyPrice * affordableAmount);
            city.move(ware, -affordableAmount, player);
            tradingOffice.move(ware, affordableAmount, buyPrice);
        }
        return true;
    }
    @Override
    public boolean execute() {
        IWare ware = getWare();
        int availableAmount = getCity().getWare(ware).getAmount();
        IPlayer player = tradingOffice.getOwner();
        ICompany company = player.getCompany();
        int capacity = tradingOffice.capacityProperty().get() - tradingOffice.getOccupiedSpace();
        int maxAmount = capacity;
        if (!ware.isBarrelSizedWare()) {
            maxAmount /= 10;
        }
        int affordableAmount = computablePrice.calculateBuyAmount(ware, availableAmount, maxBuyPrice, maxAmount, company.getCash());
        return buyWare(availableAmount, player, company, affordableAmount);
    }
}
