package ch.sahits.game.openpatrician.engine.player.tradesteps;

import ch.sahits.game.openpatrician.annotation.ClassCategory;
import ch.sahits.game.openpatrician.annotation.EClassCategory;
import ch.sahits.game.openpatrician.annotation.Prototype;
import ch.sahits.game.openpatrician.model.ICompany;
import ch.sahits.game.openpatrician.model.IPlayer;
import ch.sahits.game.openpatrician.model.product.ComputablePriceV2;
import ch.sahits.game.openpatrician.model.product.IWare;
import ch.sahits.game.openpatrician.model.ship.INavigableVessel;
import com.thoughtworks.xstream.annotations.XStreamOmitField;
import lombok.Setter;
import org.springframework.beans.factory.annotation.Autowired;

/**
 * Implements the trade step to by the some goods up to a maximum buy price. It also checks that
 * the ware is only loaded up to a certain maximum of the space on the ship.
 */
@ClassCategory({EClassCategory.SERIALIZABLE_BEAN, EClassCategory.PROTOTYPE_BEAN})
@Prototype
public class CheckedBuyTradeStep extends AbstractBuyTradeStep {

    @Setter
    private int maxBuyPrice;

    @Autowired
    @XStreamOmitField
    private ComputablePriceV2 computablePrice;


    @Override
    public boolean execute() {
        IWare ware = getWare();
        INavigableVessel vessel = getVessel();
        int availableAmount = getCity().getWare(ware).getAmount();
        IPlayer player = (IPlayer) vessel.getOwner();
        ICompany company = player.getCompany();
        int capacity = vessel.getCapacity();
        int maxAmount = capacity;
        if (!ware.isBarrelSizedWare()) {
            maxAmount /= 10;
        }
        int affordableAmount = computablePrice.calculateBuyAmount(ware, availableAmount, maxBuyPrice, maxAmount, company.getCash());
        int size = vessel.getSize();
        int alreadyLoaded = vessel.getWare(ware).getAmount();
        int maxAmountBasedOnPercentage = (int) (size * ware.getAiMaxBuyPercentage()) - alreadyLoaded;
        affordableAmount = Math.min(affordableAmount, maxAmountBasedOnPercentage);
        int maxAbsoluteAmount = ware.getAiMaxBuyAbsolute() - alreadyLoaded;
        affordableAmount = Math.min(affordableAmount, maxAbsoluteAmount);
        return buyWare(availableAmount, player, company, affordableAmount);
    }

}
