package ch.sahits.game.openpatrician.engine.land;

import ch.sahits.game.event.data.ClockTickDayChange;
import ch.sahits.game.openpatrician.clientserverinterface.model.factory.PeopleFactory;
import ch.sahits.game.openpatrician.engine.AbstractEngine;
import ch.sahits.game.openpatrician.model.Date;
import ch.sahits.game.openpatrician.model.city.ICity;
import ch.sahits.game.openpatrician.model.map.IMap;
import ch.sahits.game.openpatrician.model.people.ICaptain;
import ch.sahits.game.openpatrician.model.people.impl.CaptainsState;
import ch.sahits.game.openpatrician.utilities.annotation.ClassCategory;
import ch.sahits.game.openpatrician.utilities.annotation.EClassCategory;
import ch.sahits.game.openpatrician.utilities.annotation.LazySingleton;
import com.google.common.eventbus.AsyncEventBus;
import com.google.common.eventbus.Subscribe;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.beans.factory.annotation.Value;

import javax.annotation.PostConstruct;
import javax.annotation.PreDestroy;
import java.time.LocalDateTime;
import java.util.List;
import java.util.Optional;
import java.util.Random;

import static com.google.common.collect.Lists.newArrayList;

/**
 * Engine controling all the roaming captains.
 * @author Andi Hotz, (c) Sahits GmbH, 2013
 * Created on Jan 19, 2013
 *
 */
@LazySingleton
@ClassCategory(EClassCategory.SINGLETON_BEAN)
public class CaptainEngine extends AbstractEngine {
	@Autowired
	private Date date;
    @Autowired
    private IMap map;
    @Autowired
    private Random rnd;
	@Autowired
	private CaptainsState captainsState;
    @Autowired
    private PeopleFactory peopleFactory;
    @Autowired
    @Qualifier("timerEventBus")
    private AsyncEventBus timerEventBus;
    @Value("${freeCaptain.max}")
    private int maxFreeCaptains;
    @Value("${freeCaptain.timeInTown}")
    private int maxTimeInTown;


    @PostConstruct
	public void init() {
        for (int i = 0; i<maxFreeCaptains; i++) {
            // select town
            ICity city = getRandomCity();
            // create arival date in intervall [now-maxTimeInTown,now]
            LocalDateTime currentDate = date.getCurrentDate();
            double nextGaussian = rnd.nextDouble();
            LocalDateTime arrival = currentDate.minusDays((int)Math.rint(nextGaussian*maxTimeInTown));
            // create captain
            ICaptain captain = peopleFactory.createNewCaptainProperty();
            captainsState.addCaptain(city, arrival, captain);
        }
        timerEventBus.register(this);
	}
    @PreDestroy
    private void unregister() {
        timerEventBus.unregister(this);
    }

    private ICity getRandomCity() {
        List<ICity> cities = map.getCities();
        return cities.get(rnd.nextInt(cities.size()));
    }
    @Subscribe
    public void handleDayChange(ClockTickDayChange event) {
        LocalDateTime today = date.getCurrentDate();
        LocalDateTime shouldLeaveWhenArrivedBefore = today.minusDays(captainsState.getMaxDaysPresent());
        for (ICaptain captain : captainsState.getAllFreeCaptains()) {
            Optional<ICity> city = captainsState.findCaptainsCurrentCity(captain);
            if (city.isPresent()) {
                LocalDateTime arrival = captainsState.getArrivalDate(captain);
                if (arrival.isBefore(shouldLeaveWhenArrivedBefore)) {
                    ICity nextCity = getRandomCity();
                    while (city.equals(nextCity)) {
                        nextCity = getRandomCity();
                    }
                    // calculate days in between
                    double distance = Math.abs(city.get().getCoordinates().getX() - nextCity.getCoordinates().getX());
                    double percentage = distance/map.getDimension().getWidth();
                    int days = (int) Math.rint(10*percentage);
                    captainsState.hireCaptain(captain, city.get()); // This removes the current mapping
                    LocalDateTime arrivalDate = date.getCurrentDate().plusDays(days);
                    captainsState.assignCaptainToCity(captain, city.get(), arrivalDate);
                }
            }
        }
    }

    @Override
    public List<AbstractEngine> getChildren() {
        return newArrayList();
    }

    // TODO remove captain from the list (retirement)
		// TODO add captain to the list if one was retired

}
