package ch.sahits.game.openpatrician.engine.land.city.internal;

import ch.sahits.game.openpatrician.event.data.AuctionBid;
import ch.sahits.game.openpatrician.event.data.AuctionBidLevel;
import ch.sahits.game.openpatrician.event.data.BuildingAuctionFinished;
import ch.sahits.game.openpatrician.event.data.BuildingNotAuctioned;
import ch.sahits.game.openpatrician.event.data.ShipAuctionFinished;
import ch.sahits.game.openpatrician.event.data.ShipNotAuctioned;
import ch.sahits.game.openpatrician.model.IHumanPlayer;
import ch.sahits.game.openpatrician.utilities.annotation.ClassCategory;
import ch.sahits.game.openpatrician.utilities.annotation.EClassCategory;
import ch.sahits.game.openpatrician.utilities.annotation.LazySingleton;
import ch.sahits.game.openpatrician.engine.AbstractEngine;
import ch.sahits.game.openpatrician.engine.EngineFactory;
import ch.sahits.game.openpatrician.model.IAIPlayer;
import ch.sahits.game.openpatrician.model.ICompany;
import ch.sahits.game.openpatrician.model.IPlayer;
import ch.sahits.game.openpatrician.model.city.guild.EBidLevel;
import ch.sahits.game.openpatrician.model.city.guild.IAuction;
import ch.sahits.game.openpatrician.model.city.guild.IBuildingAuction;
import ch.sahits.game.openpatrician.model.city.guild.IShipAuction;
import ch.sahits.game.openpatrician.utilities.CancelableRunnable;
import com.google.common.base.Preconditions;
import com.google.common.eventbus.AsyncEventBus;
import com.google.common.eventbus.Subscribe;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;

import javax.annotation.PostConstruct;
import javax.annotation.PreDestroy;
import java.util.ArrayList;
import java.util.Iterator;
import java.util.List;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.TimeUnit;

import static com.google.common.collect.Lists.newArrayList;

/**
 * @author Andi Hotz, (c) Sahits GmbH, 2016
 *         Created on Nov 07, 2016
 */
@LazySingleton
@ClassCategory(EClassCategory.SINGLETON_BEAN)
public class AuctionEngine extends AbstractEngine {
    public static final int BID_DELTA = 500;

    @Autowired
    @Qualifier("serverTimer")
    private ScheduledExecutorService timer;
    @Autowired
    @Qualifier("serverClientEventBus")
    private AsyncEventBus clientServerEventBus;
    @Autowired
    private EngineFactory factory;

    @PostConstruct
    private void init() {
        clientServerEventBus.register(this);
    }
    @PreDestroy
    private void unregister() {
        clientServerEventBus.unregister(this);
    }

    public void startAuction(IAuction auction, List<IAIPlayer> players) {
        // Chose player for first bid;
        auction.setCurrentBid(auction.getStartingBid() - BID_DELTA);
        for (Iterator<IAIPlayer> iterator = players.iterator(); iterator.hasNext(); ) {
            IAIPlayer player = iterator.next();
            ICompany company = player.getCompany();
            if (company.getCash() < auction.getStartingBid() * 3) {
                iterator.remove();
            }
        }
        auction.setBiddingPlayers(players);
        if (players.isEmpty()) {
            // long running task waiting for Human player bid
            CancelableRunnable task = new CancelableRunnable() {
                private boolean run = true;
                @Override
                public void cancel() {
                    run = false;
                }

                @Override
                public void run() {
                    if (run) {
                        if (auction instanceof IShipAuction) {
                            if (auction.getOwner().isPresent()) {
                                IShipAuction shipAuction = (IShipAuction) auction;
                                clientServerEventBus.post(new ShipNotAuctioned(shipAuction.getAuctionedShip(), shipAuction.getOwner().get()));
                            }
                        } else {
                            IBuildingAuction buildingAuction = (IBuildingAuction) auction;
                            clientServerEventBus.post(new BuildingNotAuctioned(buildingAuction.getAuctionedBuilding(), buildingAuction.getOwner().get()));
                        }
                    }
                }
            };
            auction.setCurrentTask(task);
            timer.schedule(task, 2, TimeUnit.MINUTES);
        } else {
            CancelableRunnable task = factory.createNextBidTask(auction);
            auction.setCurrentTask(task);
            timer.schedule(task, 30, TimeUnit.SECONDS);
        }
    }


    @Subscribe
    public void handleBid(AuctionBid bid) {
        Preconditions.checkArgument(!bid.getBidder().equals(bid.getAuction().getBiddingPlayer()), "The bidder has to change");
        IAuction auction = bid.getAuction();
        auction.getCurrentTask().cancel();
        auction.setBidLevel(EBidLevel.INITIAL);
        auction.setBiddingPlayer(bid.getBidder());
        auction.setCurrentBid(bid.getAmount());
        clientServerEventBus.post(new AuctionBidLevel(auction.getBidLevel(), EBidLevel.INITIAL, bid.getBidder(), auction));
        CancelableRunnable task = factory.createNextBidTask(auction);
        auction.setCurrentTask(task);
        timer.schedule(task, 30, TimeUnit.SECONDS);
    }

    @Subscribe
    public void handleBidLevelChange(AuctionBidLevel change) {
        IAuction auction = change.getAuction();
        auction.setBidLevel(change.getNewLevel());
        auction.setBiddingPlayer(change.getBidder());
        if (change.getNewLevel() == EBidLevel.DONE) {
            // assign new owner
            IPlayer newOwner = auction.getBiddingPlayer();
            int amount = auction.getCurrentBid();
            if (auction.getOwner().isPresent() && auction.getOwner().get() instanceof IPlayer) {
                if (auction.getOwner().get() instanceof IHumanPlayer) {
                    ((IPlayer) auction.getOwner().get()).getCompany().updateCash(amount);
                } else {
                    ((IPlayer) auction.getOwner().get()).getCompany().updateCashDirectly(amount);
                }
            }
            if (newOwner instanceof IHumanPlayer) {
                newOwner.getCompany().updateCash(-amount);
            } else {
                newOwner.getCompany().updateCashDirectly(-amount);
            }
            // post message to new owner
            if (change.getAuction() instanceof IShipAuction) {
                IShipAuction shipAuction = (IShipAuction) auction;
                clientServerEventBus.post(new ShipAuctionFinished(shipAuction.getAuctionedShip(), newOwner));
            } else {
                IBuildingAuction buildingAuction = (IBuildingAuction) auction;
                clientServerEventBus.post(new BuildingAuctionFinished(buildingAuction.getAuctionedBuilding(), newOwner));
            }
        } else {
            CancelableRunnable task = factory.createNextBidTask(change.getAuction());
            timer.schedule(task, 30, TimeUnit.SECONDS);
        }
    }

    @Override
    public List<AbstractEngine> getChildren() {
        ArrayList<AbstractEngine> engines = newArrayList();
        return engines;
    }


}
