package ch.sahits.game.openpatrician.engine.player;

import ch.sahits.game.openpatrician.utilities.annotation.ClassCategory;
import ch.sahits.game.openpatrician.utilities.annotation.EClassCategory;
import ch.sahits.game.openpatrician.utilities.annotation.MapType;
import ch.sahits.game.openpatrician.utilities.annotation.Prototype;
import ch.sahits.game.openpatrician.clientserverinterface.service.CityProductionAndConsumptionService;
import ch.sahits.game.openpatrician.clientserverinterface.service.MapProxy;
import ch.sahits.game.openpatrician.utilities.annotation.IgnoreOnDeserialisation;
import ch.sahits.game.openpatrician.model.city.ICity;
import ch.sahits.game.openpatrician.model.city.PopulationConsume;
import ch.sahits.game.openpatrician.model.map.IMap;
import ch.sahits.game.openpatrician.model.player.ICityProductionConsumptionKnowledge;
import ch.sahits.game.openpatrician.model.player.IProductionConsumptionKnowledge;
import ch.sahits.game.openpatrician.model.product.EWare;
import ch.sahits.game.openpatrician.model.product.IWare;
import ch.sahits.game.openpatrician.model.sea.BlockadeState;
import ch.sahits.game.openpatrician.model.sea.IBlockade;
import ch.sahits.game.openpatrician.model.ship.INavigableVessel;
import com.google.common.base.Preconditions;
import com.google.common.collect.Lists;
import com.thoughtworks.xstream.annotations.XStreamOmitField;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.context.ApplicationContext;

import javax.annotation.PostConstruct;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Map.Entry;
import java.util.Set;
import java.util.TreeMap;
import java.util.stream.Collectors;

/**
 * Implementation of the knowlege of all cities
 */
@Prototype
@ClassCategory({EClassCategory.SERIALIZABLE_BEAN, EClassCategory.PROTOTYPE_BEAN})
public class ProductionConsumptionKnowledge implements IProductionConsumptionKnowledge {
    @MapType(key = ICity.class, value = ICityProductionConsumptionKnowledge.class)
    private Map<ICity, ICityProductionConsumptionKnowledge> productionAndConsumption = new HashMap<>();

    @Autowired
    @XStreamOmitField
    private CityProductionAndConsumptionService productionAndConsumptionService;
    @Autowired
    private IMap map;
    @Autowired
    @XStreamOmitField
    private ApplicationContext context;
    @Autowired
    @XStreamOmitField
    private PopulationConsume consume;
    @Autowired
    private BlockadeState blockade;
    @Autowired
    @XStreamOmitField
    private MapProxy proxy;

    @PostConstruct
    @IgnoreOnDeserialisation
    private void init() {
        for (ICity city : map.getCities()) {
            CityProductionConsumptionKnowledge cityKnoledge = context.getBean(CityProductionConsumptionKnowledge.class);
            for (EWare ware : EWare.values()) {
                int amount = productionAndConsumptionService.getProductionOutputCurrentWeek(ware, city);
                cityKnoledge.updateProduction(ware, amount);
                amount = consume.getBasicWeeklyConsumtion(ware);
                cityKnoledge.updateConsumption(ware, amount);
            }
            productionAndConsumption.put(city, cityKnoledge);
        }
    }

    @Override
    public void updateKnowledge(ICity city) {
        CityProductionConsumptionKnowledge knowledge = (CityProductionConsumptionKnowledge) productionAndConsumption.get(city);
        for (EWare ware : EWare.values()) {
            int amount = productionAndConsumptionService.getProductionOutputCurrentWeek(ware, city);
            knowledge.updateProduction(ware, amount);
            amount = (int) Math.floor(consume.getWeeklyConsumption(ware, city));
            knowledge.updateConsumption(ware, amount);
            amount = city.getWare(ware).getAmount();
            knowledge.updateStored(ware, amount);
        }
    }

    @Override
    public ICityProductionConsumptionKnowledge getKnowlege(ICity city) {
        Preconditions.checkNotNull(city, "The city must not be null");
        ICityProductionConsumptionKnowledge knowledge = productionAndConsumption.get(city);
        return knowledge;
    }

    @Override
    public List<ICity> findListWithProductionsMinimalDistance(ICity distanceToCity, IWare ware, INavigableVessel vessel) {
        TreeMap<Double, ICity> cityMap = new TreeMap<>();
        List<ICity> reachableCities = proxy.getAllReachableNonBlockadedCities(vessel);

        double mapWidth = map.getDimension().getWidth();
        for (ICity city : productionAndConsumption.keySet() ) {
            if (!city.equals(distanceToCity)) {
                int productionAmount = productionAndConsumption.get(city).getProductionAmount(ware);
                double distance = distanceToCity.getCoordinates().distance(city.getCoordinates());
                double distanceFactor = distance/mapWidth;
                if (reachableCities.contains(city)) {
                    cityMap.put(1 / (distanceFactor * productionAmount), city);
                }
            }
        }
        return cityMap.values().stream().collect(Collectors.toList());
    }

    @Override
    public List<ICity> findListWithConsumptionMinimalDistance(ICity distanceToCity, IWare ware, INavigableVessel vessel) {
        TreeMap<Double, ICity> cityMap = new TreeMap<>();
        List<ICity> reachableCities = proxy.getAllReachableNonBlockadedCities(vessel);
        double mapWidth = map.getDimension().getWidth();
        for (ICity city : productionAndConsumption.keySet() ) {
            if (!city.equals(distanceToCity)) {
                int consumptionAmount = productionAndConsumption.get(city).getConsumptionAmount(ware);
                double distance = distanceToCity.getCoordinates().distance(city.getCoordinates());
                double distanceFactor = distance/mapWidth;
                if (reachableCities.contains(city)) {
                    cityMap.put(1 / (distanceFactor * consumptionAmount), city);
                }
            }
        }
        return cityMap.values().stream().collect(Collectors.toList());
    }

    @Override
    public List<ICity> findCitiesWithNeedMinimalDistance(ICity distanceToCity, IWare ware, INavigableVessel vessel) {
        TreeMap<Double, ICity> cityMap = new TreeMap<>();
        double mapWidth = map.getDimension().getWidth();
        List<ICity> reachableCities = proxy.getAllReachableCities(vessel);
        for (ICity city : productionAndConsumption.keySet() ) {
            if (!city.equals(distanceToCity)) {
                int productionAmount = productionAndConsumption.get(city).getProductionAmount(ware);
                int consumtionAmount = productionAndConsumption.get(city).getConsumptionAmount(ware);
                double consumedRation = productionAmount - consumtionAmount;
                double distance = distanceToCity.getCoordinates().distance(city.getCoordinates());
                double distanceFactor = distance/mapWidth;
                if (reachableCities.contains(city)) {
                    cityMap.put(1 / (distanceFactor * consumedRation), city);
                }
            }
        }
        return Lists.reverse(cityMap.values().stream().collect(Collectors.toList()));
    }
}
