package ch.sahits.game.openpatrician.engine.player.strategy;

import ch.sahits.game.openpatrician.clientserverinterface.service.MapProxy;
import ch.sahits.game.openpatrician.clientserverinterface.service.ShipService;
import ch.sahits.game.openpatrician.engine.player.CentralStorageDeliverMissionData;
import ch.sahits.game.openpatrician.engine.player.tradesteps.AggregatedCheckedBuyTradeStep;
import ch.sahits.game.openpatrician.engine.player.tradesteps.AggregatesCheckedSellTradeStep;
import ch.sahits.game.openpatrician.engine.player.tradesteps.BuyWeaponTradeStep;
import ch.sahits.game.openpatrician.engine.player.tradesteps.CheckForRepairTradeStep;
import ch.sahits.game.openpatrician.engine.player.tradesteps.CheckHireCaptainTradeStep;
import ch.sahits.game.openpatrician.engine.player.tradesteps.CheckedTransferToOfficeTradeStep;
import ch.sahits.game.openpatrician.engine.player.tradesteps.GuildJoinTradeStep;
import ch.sahits.game.openpatrician.engine.player.tradesteps.HireDismissTradeManagerTradeStep;
import ch.sahits.game.openpatrician.engine.player.tradesteps.HireSailorsStep;
import ch.sahits.game.openpatrician.engine.player.tradesteps.PayBackLoanTradeStep;
import ch.sahits.game.openpatrician.engine.player.tradesteps.TakeLoanTradeStep;
import ch.sahits.game.openpatrician.engine.player.tradesteps.TransferToOfficeTradeStep;
import ch.sahits.game.openpatrician.engine.player.tradesteps.TransferToShipTradeStep;
import ch.sahits.game.openpatrician.engine.player.tradesteps.TravelToTradeStep;
import ch.sahits.game.openpatrician.engine.player.tradesteps.UpgradeShipTradeStep;
import ch.sahits.game.openpatrician.model.Date;
import ch.sahits.game.openpatrician.model.IAIPlayer;
import ch.sahits.game.openpatrician.model.building.ITradingOffice;
import ch.sahits.game.openpatrician.model.city.ICity;
import ch.sahits.game.openpatrician.model.event.IShipEntersPortEvent;
import ch.sahits.game.openpatrician.model.player.ICityProductionConsumptionKnowledge;
import ch.sahits.game.openpatrician.model.player.IProductionConsumptionKnowledge;
import ch.sahits.game.openpatrician.model.player.ITradeStategyHint;
import ch.sahits.game.openpatrician.model.product.EWare;
import ch.sahits.game.openpatrician.model.product.IWare;
import ch.sahits.game.openpatrician.model.ship.INavigableVessel;
import ch.sahits.game.openpatrician.utilities.annotation.ClassCategory;
import ch.sahits.game.openpatrician.utilities.annotation.EClassCategory;
import ch.sahits.game.openpatrician.utilities.annotation.LazySingleton;
import com.google.common.annotations.VisibleForTesting;
import com.google.common.base.Preconditions;
import org.springframework.beans.factory.annotation.Autowired;

import java.time.Duration;
import java.time.LocalDateTime;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.Set;
import java.util.stream.Collectors;

import static java.util.Arrays.asList;

/**
 * Strategy to deliver the required wares out of a central storage.
 * @author Andi Hotz, (c) Sahits GmbH, 2017
 * Created on Oct 23, 2017
 */
@LazySingleton
@ClassCategory(EClassCategory.SINGLETON_BEAN)
public class CentralStorageDeliveryTradeStrategy extends BasePlayerTradeStrategy {
    @Autowired
    private MapProxy map;
    @Autowired
    private ShipService shipService;
    @Autowired
    private Date date;

    public CentralStorageDeliveryTradeStrategy() {
        tradeStrategyType = EAITradeStrategyType.DELIVER_WARES_FROM_CENTRAL_STORAGE;
    }

    @Override
    public void initializeTradeCycle(IAIPlayer player, INavigableVessel vessel) {
        CentralStorageStrategyHint hint = null;
        ICity centralStorage = null;
        for (ICity city : map.getAllReachableNonBlockadedCities(vessel)) {
            for (ITradeStategyHint h : player.getPlayerContext().getHints(city)) {
                if (h instanceof CentralStorageStrategyHint) {
                    CentralStorageStrategyHint hint1 = (CentralStorageStrategyHint) h;
                    CentralStorageHintContext context = (CentralStorageHintContext) hint1.getContext();
                    if (context.getDistributingVessels().isEmpty()) {
                        hint = hint1;
                        centralStorage = city;
                        break;
                    }
                }
            }
            if (hint != null) {
                break;
            }
        }
        Preconditions.checkNotNull(hint, "The CentralStorageStrategyHint could not be found");
        CentralStorageHintContext context = (CentralStorageHintContext) hint.getContext();
        Optional<ICity> optCity = shipService.findCity(vessel);
        CentralStorageDeliverMissionData missionData = new CentralStorageDeliverMissionData();
        if (optCity.isPresent()) {
             ICity city = optCity.get();
             if (centralStorage.equals(city)) {
                 handleTradeInCentralStorage(player, vessel, city, context);
                 missionData.setLastVisit(date.getCurrentDate());
             } else {
                 // travel to central storage
                 addDefaultTradeSteps(vessel, player, city, centralStorage, new HashSet<>(), new HashSet<>(context.getWares()), false);
             }
        } else {
            TravelToTradeStep travelTo = createTravelToStep(vessel, centralStorage);
            append(player, vessel, travelTo);
        }
        // set trade mission
        missionData.setContext(context);
        missionData.setSourceCity(centralStorage);
        player.setTradeMission(vessel, missionData);
    }
    @VisibleForTesting
    void handleTradeInCentralStorage(IAIPlayer player, INavigableVessel vessel, ICity centralStorageCity, CentralStorageHintContext context) {
        CentralStorageDeliverMissionData missionData = (CentralStorageDeliverMissionData) player.getTradeMission(vessel);
        // transfer everything to storage
        TransferToOfficeTradeStep trasferToOfficeStep = createTransferToOfficeTradeStep(vessel, centralStorageCity);
        append(player, vessel, trasferToOfficeStep);
        Map<IWare, Integer> requiredWares = new HashMap<>();
        ITradingOffice office = player.findTradingOffice(centralStorageCity).get();
        IProductionConsumptionKnowledge globalKnowledge = player.getProductionAndConsumptionKnowledge();
        if (missionData.getLastVisit() == null) {
            fillRequiredWaresAvailableBased(requiredWares, context, office);
        } else {
            // caluclate duration in weeks and the consumption of the cities to get approx numbers
            fillRequiredWaresConsumptionBased(requiredWares, context, missionData, globalKnowledge);
        }
        // transfer required wares to ship
        ICity nextStop = getNextStop(centralStorageCity, missionData);
        PayBackLoanTradeStep payBackLoan = createPaybackLoanStep(player, centralStorageCity);
        append(player, vessel, payBackLoan);
        CheckHireCaptainTradeStep hireCaptain = createHireCaptain(vessel, centralStorageCity, player);
        append(player, vessel, hireCaptain);
        GuildJoinTradeStep joinTradeStep = createJoinGuildTradeStep(vessel, centralStorageCity, player);
        append(player, vessel, joinTradeStep);
        HireDismissTradeManagerTradeStep hireTradeManager = createHireDismissTradeManagerTradeStep(centralStorageCity, player);
        append(player, vessel, hireTradeManager);
        TakeLoanTradeStep takeLoanStep = createCheckAndTakeLoanStep(player, centralStorageCity);
        append(player, vessel, takeLoanStep);
        BuyWeaponTradeStep buyWeaponsStep = createWeaponBuyTradeStep(vessel, player, centralStorageCity);
        append(player, vessel, buyWeaponsStep);
        TransferToShipTradeStep transferToShipStep = createTransferToShipTradeStep(vessel, centralStorageCity, requiredWares);
        append(player, vessel, transferToShipStep);
        UpgradeShipTradeStep upgradeShipStep = createUpgradeShipTradeStep(vessel, player);
        append(player, vessel, upgradeShipStep);
        CheckForRepairTradeStep repairStep = createCheckRepairStep(vessel, centralStorageCity);
        append(player, vessel, repairStep);
        HireSailorsStep hireSailors = createHireSailorStep(vessel, centralStorageCity);
        append(player, vessel, hireSailors);
        // travel to first stop
        TravelToTradeStep travelTo = createTravelToStep(vessel, nextStop);
        append(player, vessel, travelTo);
    }

    private ICity getNextStop(ICity city, CentralStorageDeliverMissionData missionData) {
        CentralStorageHintContext context = missionData.getContext();
        List<ICity> supplideCities = context.getSuppliedCities();
        for (int i = 0; i < supplideCities.size(); i++) {
            ICity currentCity = supplideCities.get(i);
            if (currentCity.equals(city)) {
                if (i == supplideCities.size() - 1) {
                    return supplideCities.get(0);
                } else {
                    return supplideCities.get(i + 1);
                }
            }
        }
        return null; // only if the list is empty or the city is not contained
    }

    @VisibleForTesting
    void fillRequiredWaresAvailableBased(Map<IWare, Integer> requiredWares, CentralStorageHintContext context, ITradingOffice office) {
        asList(EWare.values()).stream()
                .filter(ware -> context.getWares().contains(ware))
                .filter(ware -> !office.getSteward().isPresent() || office.getOfficeTrading().isMovableToShip(ware))
                .forEach(ware -> {
                    int availableInStore = office.getWare(ware).getAmount();
                    requiredWares.put(ware, availableInStore);
                });
    }
    @VisibleForTesting
    void fillRequiredWaresConsumptionBased(Map<IWare, Integer> requiredWares, CentralStorageHintContext context, CentralStorageDeliverMissionData missionData, IProductionConsumptionKnowledge globalKnowledge) {
        LocalDateTime now = date.getCurrentDate();
        int weeks = Math.max(1, (int) (Duration.between(missionData.getLastVisit(), now).toDays() / 7));
        for (ICity suppliedCity : context.getSuppliedCities()) {
            if (suppliedCity.equals(missionData.getSourceCity())) {
                continue;
            }
            for (IWare ware : context.getWares()) {
                if (!requiredWares.containsKey(ware)) {
                     requiredWares.put(ware, 0);
                }
                ICityProductionConsumptionKnowledge knowledge = globalKnowledge.getKnowlege(suppliedCity);
                int consumption = knowledge.getConsumptionAmount(ware) * weeks;
                requiredWares.put(ware, requiredWares.get(ware) + consumption);
            }
        }
    }

    @Override
    public void handleShipArrivesInPort(IShipEntersPortEvent event) {
        INavigableVessel vessel = event.getShip();
        if (isMatchingTradeStrategy(vessel)) {
            IAIPlayer player = (IAIPlayer) vessel.getOwner();
            ICity city = event.getCity();
            CentralStorageDeliverMissionData missionData = (CentralStorageDeliverMissionData) player.getTradeMission(vessel);
            if (city.equals(missionData.getSourceCity())) {
                // Sell the wares not on the list
                // refill and restart
                handleTradeInCentralStorage(player, vessel, city, missionData.getContext());
            } else {
                // calculate the wares needed in cities down the road and sell accordingly
                // buy wares that are needed and not in the delivery list
                handleTradeInSuppliedCity(player, vessel, city, missionData);
            }
            executeTradeSteps(player, vessel);
        }
    }
    @VisibleForTesting
    void handleTradeInSuppliedCity(IAIPlayer player, INavigableVessel vessel, ICity city, CentralStorageDeliverMissionData missionData) {
        CentralStorageHintContext context = missionData.getContext();
        IProductionConsumptionKnowledge globalKnowledge = player.getProductionAndConsumptionKnowledge();
        List<ICity> forwardCities = calculateForwardCities(city, context);
        LinkedList<ICity> citiesToSell = new LinkedList<>(forwardCities);
        citiesToSell.addFirst(city);
        Map<IWare, Integer> amountToSellOrTransfer = calculateAmountsToSell(vessel, city, context, globalKnowledge, citiesToSell);
        Set<IWare> buyWares = calculateWaresToBuy(city, context, globalKnowledge);

        boolean sell = true; // sell if there is no trading office with store manager
        Optional<ITradingOffice> optOffice = player.findTradingOffice(city);
        sell = !(optOffice.isPresent() && optOffice.get().getSteward().isPresent());
        if (sell) {
           // Checked aggregate sell step
            AggregatesCheckedSellTradeStep aggSellStep = createAggregatedCheckedSellStep(vessel, city, amountToSellOrTransfer);
            append(player, vessel, aggSellStep);
        } else {
            // Check transfer to office step
            CheckedTransferToOfficeTradeStep transferToOfficeStep = createCheckedTransferToOfficeTradeStep(vessel, city, amountToSellOrTransfer);
            append(player, vessel, transferToOfficeStep);
        }
        ICity nextStop = getNextStop(city, missionData);
        PayBackLoanTradeStep payBackLoan = createPaybackLoanStep(player, city);
        append(player, vessel, payBackLoan);
        CheckHireCaptainTradeStep hireCaptain = createHireCaptain(vessel, city, player);
        append(player, vessel, hireCaptain);
        GuildJoinTradeStep joinTradeStep = createJoinGuildTradeStep(vessel, city, player);
        append(player, vessel, joinTradeStep);
        HireDismissTradeManagerTradeStep hireTradeManager = createHireDismissTradeManagerTradeStep(city, player);
        append(player, vessel, hireTradeManager);
        TakeLoanTradeStep takeLoanStep = createCheckAndTakeLoanStep(player, city);
        append(player, vessel, takeLoanStep);
        BuyWeaponTradeStep buyWeaponsStep = createWeaponBuyTradeStep(vessel, player, city);
        append(player, vessel, buyWeaponsStep);
        if (!sell) {
            Map<IWare, Integer> buyAmounts = calculateBuyAmounts(buyWares);
            TransferToShipTradeStep transfertoShipStep = createTransferToShipTradeStep(vessel, city, buyAmounts);
            append(player, vessel, transfertoShipStep);
        }
        AggregatedCheckedBuyTradeStep buyStepHometown = createAggregatedCheckedBuyTradeStep(vessel, city, new ArrayList<>(buyWares));
        append(player, vessel, buyStepHometown);
        UpgradeShipTradeStep upgradeShipStep = createUpgradeShipTradeStep(vessel, player);
        append(player, vessel, upgradeShipStep);
        CheckForRepairTradeStep repairStep = createCheckRepairStep(vessel, city);
        append(player, vessel, repairStep);
        HireSailorsStep hireSailors = createHireSailorStep(vessel, city);
        append(player, vessel, hireSailors);
        TravelToTradeStep travelTo = createTravelToStep(vessel, nextStop);
        append(player, vessel, travelTo);
    }
    @VisibleForTesting
    Map<IWare, Integer> calculateBuyAmounts(Set<IWare> buyWares) {
        Map<IWare, Integer> buyAmounts = new HashMap<>();
        for (IWare buyWare : buyWares) {
            int amount = 5;
            if (buyWare.isBarrelSizedWare()) {
                amount *= 10;
            }
            buyAmounts.put(buyWare, amount);
        }
        return buyAmounts;
    }

    @VisibleForTesting
    Set<IWare> calculateWaresToBuy(ICity city, CentralStorageHintContext context, IProductionConsumptionKnowledge globalKnowledge) {
        Set<IWare> buyWares = new HashSet<>();
        for (ICity c : context.getSuppliedCities()) {
            if (c.equals(city)) {
                continue;
            }
            final ICityProductionConsumptionKnowledge knowledge = globalKnowledge.getKnowlege(c);
            asList(EWare.values()).stream()
                    .filter(ware -> knowledge.getConsumptionAmount(ware) > 0)
                    .forEach(ware -> buyWares.add(ware));
        }
        return buyWares;
    }

    @VisibleForTesting
    Map<IWare, Integer> calculateAmountsToSell(INavigableVessel vessel, ICity city, CentralStorageHintContext context, IProductionConsumptionKnowledge globalKnowledge, List<ICity> cities) {
        Map<IWare, Integer> amountToSellOrTransfer = new HashMap<>();
        for (IWare ware : context.getWares()) {
            double totalConsumedAmount = 0;
            for (ICity forwardCity : cities) {
                ICityProductionConsumptionKnowledge knowledge = globalKnowledge.getKnowlege(forwardCity);
                totalConsumedAmount += knowledge.getConsumptionAmount(ware);
            }
            ICityProductionConsumptionKnowledge knowledge = globalKnowledge.getKnowlege(city);
            int consumedHere = knowledge.getConsumptionAmount(ware);
            double percentage = consumedHere / totalConsumedAmount;
            int avalable = vessel.getWare(ware).getAmount();
            int amount = (int) Math.rint(avalable * percentage);
            if (amount > 0) {
                amountToSellOrTransfer.put(ware, amount);
            }
        }
        return amountToSellOrTransfer;
    }

    private List<ICity> calculateForwardCities(ICity city, CentralStorageHintContext context) {
        List<ICity> forwardCities = new ArrayList<>(context.getSuppliedCities());
        for (Iterator<ICity> iterator = forwardCities.iterator(); iterator.hasNext(); ) {
            ICity next = iterator.next();
            if (next.equals(city)) {
                break;
            }
            iterator.remove();
        }
        return forwardCities;
    }

    @Override
    public boolean isSelectable(IAIPlayer player, INavigableVessel vessel) {
        // There are at least 4 ships
        if (player.getFleet().size() < 4) {
            return false;
        }
        List<ICity> citiesWithTradingOffice = map.getAllReachableCities(vessel).stream()
                .filter(city -> player.findTradingOffice(city).isPresent())
                .collect(Collectors.toList());
        // This city should not yet be supplied
        for (ICity city : citiesWithTradingOffice) {
            List<ITradeStategyHint> hints = player.getPlayerContext().getHints(city);
            for (ITradeStategyHint hint : hints) {
                if (hint instanceof CentralStorageStrategyHint) {
                    CentralStorageHintContext hintContext = (CentralStorageHintContext) hint.getContext();
                    return hintContext.getDistributingVessels().isEmpty();
                }
            }
        }
        return false;
    }
}
