package ch.sahits.game.openpatrician.engine.sea;

import ch.sahits.game.graphic.image.ECircleSegment;
import ch.sahits.game.graphic.image.IMapImageServiceFacade;
import ch.sahits.game.graphic.image.model.MapGrid;
import ch.sahits.game.openpatrician.utilities.annotation.MapType;
import ch.sahits.game.openpatrician.model.map.IMap;
import com.carrotsearch.hppc.ObjectByteMap;
import com.carrotsearch.hppc.ObjectByteScatterMap;
import com.google.common.annotations.VisibleForTesting;
import com.google.common.collect.Interner;
import javafx.geometry.Point2D;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.springframework.beans.factory.annotation.Autowired;

import java.util.ArrayList;
import java.util.List;

/**
 * Service providing methods for the calculation of the heuristic and the graph.
 * @author Andi Hotz, (c) Sahits GmbH, 2016
 *         Created on Jan 01, 2016
 */
public abstract class BaseGraphCalulationService implements MapGrid {
    private final Logger logger = LogManager.getLogger(getClass());


    @Autowired
    protected IMapImageServiceFacade imageService;
    @Autowired
    protected IMap map;
    @Autowired
    protected Interner<Point2D> pointInterner;

    @MapType(key = Point2D.class, value = Byte.class)
//    private HashMap<Point2D, Boolean> onSea = new HashMap<>();
    private ObjectByteMap<Point2D> onSea = new ObjectByteScatterMap();

    protected Point2D getPoint(double x, double y) {
        return pointInterner.intern(new Point2D(x, y));
    }

    /**
     * Calculate the weight between two nodes.
     * @param from source node
     * @param to destination node
     * @return weight of the edge from node <code>from</code> to <code>to</code>
     */
    protected abstract double calculateWeight(Point2D from, Point2D to);
    @VisibleForTesting
    protected List<ECircleSegment> getSegments(Point2D from, Point2D to) {
        ArrayList<ECircleSegment> segments = new ArrayList<>();
        if (from.equals(to)) {
            return segments;
        }
        double xVec = to.getX() - from.getX();
        double yVec = to.getY() - from.getY();
        double angle = Math.toDegrees( Math.atan(yVec/xVec)) + 90;
        if (xVec < 0) {
            angle += 180;
        }
        if (angle < 22.5) {
            segments.add(ECircleSegment.TOPLEFT);
            segments.add(ECircleSegment.TOPRIGHT);
        } else if (angle < 45 + 22.5) {
            segments.add(ECircleSegment.TOPRIGHT);
        } else if (angle < 90 + 22.5) {
            segments.add(ECircleSegment.TOPRIGHT);
            segments.add(ECircleSegment.BOTTOMRIGHT);
        } else if (angle < 180 - 22.5) {
            segments.add(ECircleSegment.BOTTOMRIGHT);
        } else if (angle < 180 + 22.5) {
            segments.add(ECircleSegment.BOTTOMRIGHT);
            segments.add(ECircleSegment.BOTTOMLEFT);
        } else if (angle < 270 - 22.5) {
            segments.add(ECircleSegment.BOTTOMLEFT);
        } else if (angle < 270 + 22.5) {
            segments.add(ECircleSegment.BOTTOMLEFT);
            segments.add(ECircleSegment.TOPLEFT);
        } else if (angle < 360 - 22.5) {
            segments.add(ECircleSegment.TOPLEFT);
        } else {
            segments.add(ECircleSegment.TOPLEFT);
            segments.add(ECircleSegment.TOPRIGHT);
        }
        return segments;
    }
    @VisibleForTesting
    protected List<ECircleSegment> getTangentialSegments(Point2D from, Point2D to) {
        ArrayList<ECircleSegment> segments = new ArrayList<>();
        if (from.equals(to)) {
            return segments;
        }
        double xVec = to.getX() - from.getX();
        double yVec = to.getY() - from.getY();
        double angle = Math.toDegrees( Math.atan(yVec/xVec)) + 90;
        if (xVec < 0) {
            angle += 180;
        }
        if (angle < 22.5) {
            segments.add(ECircleSegment.TOPLEFT);
            segments.add(ECircleSegment.TOPRIGHT);
            segments.add(ECircleSegment.BOTTOMRIGHT);
            segments.add(ECircleSegment.BOTTOMLEFT);
        } else if (angle < 45 + 22.5) {
            segments.add(ECircleSegment.TOPLEFT);
            segments.add(ECircleSegment.BOTTOMRIGHT);
        } else if (angle < 90 + 22.5) {
            segments.add(ECircleSegment.TOPLEFT);
            segments.add(ECircleSegment.TOPRIGHT);
            segments.add(ECircleSegment.BOTTOMRIGHT);
            segments.add(ECircleSegment.BOTTOMLEFT);
        } else if (angle < 180 - 22.5) {
            segments.add(ECircleSegment.TOPRIGHT);
            segments.add(ECircleSegment.BOTTOMLEFT);
        } else if (angle < 180 + 22.5) {
            segments.add(ECircleSegment.TOPLEFT);
            segments.add(ECircleSegment.TOPRIGHT);
            segments.add(ECircleSegment.BOTTOMRIGHT);
            segments.add(ECircleSegment.BOTTOMLEFT);
        } else if (angle < 270 - 22.5) {
            segments.add(ECircleSegment.TOPLEFT);
            segments.add(ECircleSegment.BOTTOMRIGHT);
        } else if (angle < 270 + 22.5) {
            segments.add(ECircleSegment.TOPLEFT);
            segments.add(ECircleSegment.TOPRIGHT);
            segments.add(ECircleSegment.BOTTOMRIGHT);
            segments.add(ECircleSegment.BOTTOMLEFT);
        } else if (angle < 360 - 22.5) {
            segments.add(ECircleSegment.TOPRIGHT);
            segments.add(ECircleSegment.BOTTOMLEFT);
        } else {
            segments.add(ECircleSegment.TOPLEFT);
            segments.add(ECircleSegment.TOPRIGHT);
            segments.add(ECircleSegment.BOTTOMRIGHT);
            segments.add(ECircleSegment.BOTTOMLEFT);
        }
        return segments;
    }

    /**
     * Verify that a location is actually on sea.
     * @param point to be checked
     * @return true if the point is at sea.
     */
    public boolean isOnSea(Point2D point) {

        if (onSea.containsKey(point)) {
            return mapByteToBoolean(onSea.get(point));
        } else {
            final int x = (int) point.getX();
            final int y = (int) point.getY();
            boolean onSea = imageService.isOnSea(x, y);
            this.onSea.put(point, mapBooleanToByte(onSea));
            return onSea;
        }
    }

    /**
     * If the byte is 0 it is considered true, otherwise false.
     * @param b
     * @return
     */
    private boolean mapByteToBoolean(Byte b) {
        return b == (byte)0x00;
    }

    private byte mapBooleanToByte(boolean b) {
        if (b) {
            return (byte)0x00;
        } else {
            return (byte)0x01;
        }
    }
}
