package ch.sahits.game.openpatrician.engine.sea.model;

import ch.sahits.game.openpatrician.model.Date;
import ch.sahits.game.openpatrician.utilities.annotation.ClassCategory;
import ch.sahits.game.openpatrician.utilities.annotation.EClassCategory;
import ch.sahits.game.openpatrician.utilities.annotation.LazySingleton;
import ch.sahits.game.openpatrician.utilities.annotation.MapType;
import org.springframework.beans.factory.annotation.Autowired;

import java.time.LocalDateTime;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;

/**
 * Storage container for keeping track of pirate activity.
 * @author Andi Hotz, (c) Sahits GmbH, 2017
 * Created on Oct 17, 2017
 */
@ClassCategory({EClassCategory.SERIALIZABLE_BEAN, EClassCategory.SINGLETON_BEAN})
@LazySingleton
public class PirateActivity {

    private static final int NUMBER_OF_CACHED_DAYS = 60;

    @Autowired
    private Date date;
    @MapType(key = LocalDateTime.class, value = PirateActivityEntry.class)
    private Map<LocalDateTime, PirateActivityEntry> map = new HashMap<>();

    /**
     * Add a new pirate activity with the current date.
      * @param pirateActivity
     */
    public void add(PirateActivityEntry pirateActivity) {
        map.put(date.getCurrentDate(), pirateActivity);
    }

    /**
     * Ensure that only the most recent events are kept.
     */
    private void clearCache() {
        LocalDateTime now = date.getCurrentDate();
        for (Iterator<LocalDateTime> iterator = map.keySet().iterator(); iterator.hasNext(); ) {
            LocalDateTime next = iterator.next();
            if (next.isBefore(now.minusDays(NUMBER_OF_CACHED_DAYS))) {
                iterator.remove();
            }
        }
    }

    /**
     * Retrieve the most recent pirate activities.
     * @return
     */
    public List<PirateActivityEntry> getPirateActivity() {
        clearCache();
        return new ArrayList<>(map.values());
    }

    /**
     * Retrieve the number of days for which attacks are tracked back into the past.
     */
    public int getObservationPeriodInDays() {
        return NUMBER_OF_CACHED_DAYS;
    }
}
