package ch.sahits.game.openpatrician.engine.event.task;

import ch.sahits.game.openpatrician.engine.player.DailyPlayerUpdater;
import ch.sahits.game.openpatrician.engine.time.DailyUpdateTask;
import ch.sahits.game.openpatrician.model.city.EPopulationClass;
import ch.sahits.game.openpatrician.model.ui.IDialogState;
import ch.sahits.game.openpatrician.model.weapon.EWeapon;
import ch.sahits.game.openpatrician.model.weapon.IArmory;
import ch.sahits.game.openpatrician.utilities.annotation.ClassCategory;
import ch.sahits.game.openpatrician.utilities.annotation.DependentInitialisation;
import ch.sahits.game.openpatrician.utilities.annotation.EClassCategory;
import ch.sahits.game.openpatrician.clientserverinterface.client.ICityPlayerProxyJFX;
import ch.sahits.game.openpatrician.clientserverinterface.model.event.MarriageBrokerAnnouncementState;
import ch.sahits.game.openpatrician.clientserverinterface.service.ClientServerTaskFactory;
import ch.sahits.game.openpatrician.model.IAIPlayer;
import ch.sahits.game.openpatrician.model.IPlayer;
import ch.sahits.game.openpatrician.model.city.IChurch;
import ch.sahits.game.openpatrician.model.city.ICity;
import ch.sahits.game.openpatrician.model.city.IShipyard;
import ch.sahits.game.openpatrician.model.city.impl.IDebt;
import ch.sahits.game.openpatrician.model.city.impl.IShipBuildTask;
import ch.sahits.game.openpatrician.model.event.TimedTask;
import ch.sahits.game.openpatrician.model.people.ISeaPirate;
import ch.sahits.game.openpatrician.model.ship.EShipType;
import ch.sahits.game.openpatrician.model.ship.EShipUpgrade;
import ch.sahits.game.openpatrician.model.ship.INavigableVessel;
import ch.sahits.game.openpatrician.model.ship.IShip;
import ch.sahits.game.openpatrician.model.initialisation.StartNewGameBean;
import javafx.geometry.Point2D;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.context.ApplicationContext;
import org.springframework.context.annotation.Lazy;
import org.springframework.stereotype.Service;

import java.time.LocalDateTime;

/**
 * @author Andi Hotz, (c) Sahits GmbH, 2014
 *         Created on Mar 01, 2014
 */
@Service
@Lazy
@DependentInitialisation(StartNewGameBean.class)
@ClassCategory(EClassCategory.SINGLETON_BEAN)
public class ServerSideTaskFactory extends ClientServerTaskFactory {
    @Autowired
    private ApplicationContext context;
    /**
     * Retrieve the ship repair task.
     * @return
     */
    public TimedTask getRepairTask(LocalDateTime executionTime, IShip ship, ICityPlayerProxyJFX proxy, IShipyard shipyard) {
        Object[] args = new Object[]{executionTime, ship, proxy, shipyard};
        return (RepairTask)  context.getBean("repairTask", args);
    }

    /**
     * Retrieve the ship build task.
     * @return
     */
    public IShipBuildTask getShipBuildTask(LocalDateTime executionTime, EShipType type, IPlayer owner, Point2D location, IShipyard shipyard) {
        Object[] args = new Object[]{executionTime, type, owner, location, shipyard};
        return (ShipBuildTask)  context.getBean("shipBuildTask", args);
    }
    /**
     * Retrieve the ship refit task.
     * @return
     */
    public TimedTask getRefitShipTask(LocalDateTime executionTime, IShip ship, EShipUpgrade upgrade, ICityPlayerProxyJFX proxy, IShipyard shipyard) {
        Object[] args = new Object[]{executionTime, ship, upgrade, proxy, shipyard};
        return (RefitShipTask)  context.getBean("refitShipTask", args);
    }

    /**
     * Retrieve the repair task for a pirate vessel
     * @param dueDate execution time
     * @param vessel vessel to be repaired
     * @param owner pirate owner
     * @return special repair task
     */
    public TimedTask getPirateRepairTask(LocalDateTime dueDate, INavigableVessel vessel, ISeaPirate owner, IShipyard shipyard) {
        Object[] args = new Object[]{dueDate, vessel, owner, shipyard};
        return (RepairPirateShipTask) context.getBean("repairPirateShipTask", args);
    }

    /**
     * Retrieve the repair task for an AI ship.
     * @param dueDate execution time
     * @param ship to be repaired
     * @return repair task for AI ships
     */
    public TimedTask getAIShipRepairTask(LocalDateTime dueDate, IShip ship, IShipyard shipyard) {
        Object[] args = new Object[]{dueDate, ship, shipyard};
        return context.getBean(AIRepairTask.class, args);
    }

    /**
     * Retrieve the refit task for an AI ship.
     * @param dueDate execution time
     * @param ship to be upgraded
     * @return upgrade task for AI ships
     */
    public TimedTask getAIShipRefitTask(LocalDateTime dueDate, IShip ship, EShipUpgrade upgrade, IShipyard shipyard) {
        Object[] args = new Object[]{dueDate, ship, upgrade, shipyard};
        return context.getBean(AIRefitTask.class, args);
    }

    /**
     * Retrieve a weekly loaner check task.
     * @return
     */
    public TimedTask getWeeklyLoanerCheck() {
        return context.getBean(WeeklyLoanerCheckTask.class);
    }

    /**
     * Retrieve the daily check for the AI player
     * @return
     */
    public TimedTask getDailyAIPlayerCheck() {
        return context.getBean(DailyPlayerUpdater.class);
    }

    /**
     * Update task that issues a PeriodicalDailyUpdate event on the event bus.
     * @return
     */
    public TimedTask getPeriodicalDailyUpdateTask() {
        return context.getBean(DailyUpdateTask.class);
    }

    /**
     * Retrieve a weekly city check task.
     * @return
     */
    public TimedTask getWeeklyCityCheck() {
        return context.getBean(WeeklyCityCheckTask.class);
    }

    /**
     * Retrieve a timed task for the marriage offer.
     * @param player that is addressed
     * @param state of the marraiage offer announcement
     * @param deadLine when the offer will happen.
     * @return timed task to be submitted to the timed task list.
     */
    public MarriageOfferTask getMarriageOfferTask(IAIPlayer player, MarriageBrokerAnnouncementState state, LocalDateTime deadLine) {
        return (MarriageOfferTask) context.getBean("marriageOfferTask", new Object[]{player, state, deadLine});
    }

    /**
     * Create the timed task for the church upgrade.
     * @param church to be upgreaded
     * @param deadline when it will happen
     * @return timed task to be supplied to the timed task list.
     */
    public ChurchUpgradeTimedTask getChurchUpgradeTask(IChurch church, LocalDateTime deadline) {
        return (ChurchUpgradeTimedTask) context.getBean("churchUpgradeTimedTask", church, deadline);
    }

    /**
     * Create the timed task for the loaner to collect money from the debitor after an auction.
     * @param debt debt contract
     * @param deadline when this will happen.
     * @return timed task to be supplied to the timed task list.
     */
    public LoanerCollectMoneyAfterAuctionTimedTask getLoanerCollectMoneyAfterAuctionTask(IDebt debt, LocalDateTime deadline) {
        return (LoanerCollectMoneyAfterAuctionTimedTask) context.getBean("loanerCollectMoneyAfterAuctionTimedTask", debt, deadline);
    }

    /**
     * Create a timed task for the loaner to collect money.
     * @param debt debt contract
     * @param debitor from whom to collect the money.
     * @param deadline when this will happen.
     * @return timed task to be supplied to the timed task list.
     */
    public LoanerCollectMoneyWithoutAuctionTimedTask getLoanerCollectMoneyTask(IDebt debt, IPlayer debitor, LocalDateTime deadline) {
        return (LoanerCollectMoneyWithoutAuctionTimedTask) context.getBean("loanerCollectMoneyWithoutAuctionTimedTask", debt, debitor, deadline);
    }

    /**
     * Retrieve the timed task for the declined marriage offer.
     * @param player who declined the offer.
     * @return timed task to be supplied to the timed task list.
     */
    public MarriageOfferDeclinedTimedTask getMarriageOfferDeclinedTask(IPlayer player) {
        return (MarriageOfferDeclinedTimedTask) context.getBean("marriageOfferDeclinedTimedTask", player);
    }

    /**
     * Retrieve the timed task for finishing a weapon construction.
      * @param armory in which the weapon was buildt
     * @param weapon type that was built.
     * @return timed task to be supplied to the timed task list.
     */
    public WeaponConstructionTask getWeaponConstructionFinishedTask(IArmory armory, EWeapon weapon) {
        return (WeaponConstructionTask) context.getBean("weaponConstructionTask", armory, weapon);
    }

    /**
     * Retrieve the timed task to update the population of a certain class.
     * @param executionTime when the task is to be executed.
     * @param populationClass population class to be updated
     * @param delta of the population change (can be negative)
     * @param city in which the population should be updated.
     * @return timed task for the population update.
     */
    public UpdatePopulationTask getUpdatePopulationTask(LocalDateTime executionTime, EPopulationClass populationClass, int delta, ICity city) {
        return context.getBean(UpdatePopulationTask.class, executionTime, populationClass, delta, city);
    }

    /**
     * Retrieve the timed task to clear the city event.
     * @param executionTime when the task is to be executed.
     * @param city for which the event is to be cleared.
     * @return timed task for the event clearing.
     */
    public ClearCityEventTask getClearCityEventTask(LocalDateTime executionTime, ICity city) {
        return context.getBean(ClearCityEventTask.class, executionTime, city);
    }

    /**
     * Retrieve the timed task to post a state based dialog message if the player is human.
     * @param executionTime when the task is to be executed.
     * @param state on which the dialog is based
     * @param player for which the dialog is to be posted
     * @param messageKey key of the dialog
     * @param messageParams message parameters
     * @return timed task for the state based display of a dialog.
     */
    public PostStateDialogToHumanPlayerTask getPostStateDialogMessageTask(LocalDateTime executionTime, IDialogState state, IPlayer player, String messageKey, Object[] messageParams) {
        return context.getBean(PostStateDialogToHumanPlayerTask.class, executionTime, state, player, messageKey, messageParams);
    }

}
