package ch.sahits.game.openpatrician.engine.event.task;

import ch.sahits.game.openpatrician.utilities.annotation.ClassCategory;
import ch.sahits.game.openpatrician.utilities.annotation.EClassCategory;
import ch.sahits.game.openpatrician.utilities.annotation.Prototype;
import ch.sahits.game.openpatrician.clientserverinterface.model.factory.ShipFactory;
import ch.sahits.game.openpatrician.engine.AIPlayerEngine;
import ch.sahits.game.openpatrician.engine.sea.LocationTracker;
import ch.sahits.game.openpatrician.model.DisplayMessage;
import ch.sahits.game.openpatrician.model.IAIPlayer;
import ch.sahits.game.openpatrician.model.IHumanPlayer;
import ch.sahits.game.openpatrician.model.IPlayer;
import ch.sahits.game.openpatrician.model.city.IShipyard;
import ch.sahits.game.openpatrician.model.city.impl.IShipBuildTask;
import ch.sahits.game.openpatrician.model.event.TargetedEvent;
import ch.sahits.game.openpatrician.model.event.TimedTask;
import ch.sahits.game.openpatrician.model.ship.EShipType;
import ch.sahits.game.openpatrician.model.ship.IShip;
import ch.sahits.game.openpatrician.utilities.service.RandomNameLoader;
import com.google.common.eventbus.AsyncEventBus;
import com.thoughtworks.xstream.annotations.XStreamOmitField;
import javafx.geometry.Point2D;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;

import javax.annotation.PostConstruct;
import java.time.LocalDateTime;

/**
 * @author Andi Hotz, (c) Sahits GmbH, 2014
 *         Created on Feb 23, 2014
 */
@Prototype
@ClassCategory({EClassCategory.SERIALIZABLE_BEAN, EClassCategory.PROTOTYPE_BEAN})
class ShipBuildTask extends TimedTask implements IShipBuildTask {
    private final EShipType type;
    private final IPlayer owner;
    @Autowired
    @XStreamOmitField
    private ShipFactory shipUtility;
    @Autowired
    @XStreamOmitField
    private AIPlayerEngine aiPlayerEngine;
    private final double x;
    private final Point2D location;
    private IShip ship;
    private final IShipyard shipyard;

    private static RandomNameLoader shipLoader = new RandomNameLoader("shipnames.properties");

    @Autowired
    @Qualifier("serverClientEventBus")
    @XStreamOmitField
    private AsyncEventBus clientServerEventBus;
    @Autowired
    private LocationTracker locationTracker;

    public ShipBuildTask(LocalDateTime executionTime, EShipType type, IPlayer owner, Point2D location, IShipyard shipyard) {
        super();
        setExecutionTime(executionTime);
        this.owner = owner;
        this.type = type;
        x = location.getX();
        this.location = location;
        this.shipyard = shipyard;
    }
    @PostConstruct
    void initializeShip() {
        ship = createShip();
    }

    @Override
    public void run() {
        ship.setLocation(location);
        locationTracker.add(ship);
        shipyard.removeCompletedConstruction(ship);
        if (owner instanceof IHumanPlayer) {
            DisplayMessage msg = new DisplayMessage("ch.sahits.game.openpatrician.engine.event.task.ShipBuildTask.message", new Object[]{type, ship.getName()});
            TargetedEvent displayMessage = new TargetedEvent((IHumanPlayer) owner, msg);
            clientServerEventBus.post(displayMessage);
        }  else {
            aiPlayerEngine.handleShipConstructionFinished(ship, (IAIPlayer) owner);
        }
    }
    /**
     * Retrieve the ship that is to be built.
     * @return
     */
    @Override
    public IShip getShipToBeBuilt(){
        return ship;
    }
    /**
     * Create the ship based on the ships type.
     * @return
     */
    private IShip createShip() {
        IShip ship = null;
        switch (type) {
            case SNAIKKA:
                ship = shipUtility.createSnaikka(shipLoader.getRandomName(), calculateShipCapacity());
                break;
            case CRAYER:
                ship = shipUtility.createCrayer(shipLoader.getRandomName(), calculateShipCapacity());
                break;
            case COG:
                ship = shipUtility.createCog(shipLoader.getRandomName(), calculateShipCapacity());
                break;
            case HOLK:
                ship = shipUtility.createHolk(shipLoader.getRandomName(), calculateShipCapacity());
                break;
        }
        ship.setOwner(owner);
        return ship;
    }

    private int calculateShipCapacity() {
        return shipUtility.calculateInitialCapacity(type, (int)x);
    }
}
