package ch.sahits.game.openpatrician.engine.land.city;

import ch.sahits.game.openpatrician.utilities.annotation.ClassCategory;
import ch.sahits.game.openpatrician.utilities.annotation.EClassCategory;
import ch.sahits.game.openpatrician.clientserverinterface.service.CityProductionAndConsumptionService;
import ch.sahits.game.openpatrician.model.IPlayer;
import ch.sahits.game.openpatrician.model.city.EPopulationClass;
import ch.sahits.game.openpatrician.model.city.ICity;
import ch.sahits.game.openpatrician.model.city.PopulationConsume;
import ch.sahits.game.openpatrician.model.personal.EEconomicCareer;
import ch.sahits.game.openpatrician.model.personal.EMilitantCareer;
import ch.sahits.game.openpatrician.model.personal.ESocialRank;
import ch.sahits.game.openpatrician.model.personal.ICareer;
import ch.sahits.game.openpatrician.model.product.EWare;
import com.google.common.annotations.VisibleForTesting;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.context.annotation.Lazy;
import org.springframework.stereotype.Component;

/**
 * Calculator for a players reputation
 * @author Andi Hotz, (c) Sahits GmbH, 2012
 * Created on Jul 23, 2012
 *
 */
@Lazy
@Component
@ClassCategory(EClassCategory.SINGLETON_BEAN)
public class ReputationCalculator {
	@Autowired
	private PopulationConsume consumer;
	@Autowired
	private CityProductionAndConsumptionService cityProductionAndConsumptionService;
	/**
	 * Check each ware if the player contributes and what amount.
	 * If the city contributes enough to cover the needs, this ware is
	 * irrelevant for the computation
	 * If the player contributes enough to cover all the need he gets
	 * 100 points for the ware. If he contributes less he gets a fraction
	 * according to what he contributes.
	 * @param city for which the ware based reputation should be checked
	 * @param player for whom the reputation is calculated
	 * @return value between 0 and 100
	 */
	public int calculateWareReputation(ICity city, IPlayer player){
		double reputation = 0;
		EWare[] wares = EWare.values();
		double negativeFactor = getNegativeReputationFactor(player, city);
		for (EWare ware : wares){
			double need = consumer.getNeed(ware, EPopulationClass.POOR, city.getPopulation(EPopulationClass.POOR));
			need += consumer.getNeed(ware, EPopulationClass.MEDIUM, city.getPopulation(EPopulationClass.MEDIUM));
			need += consumer.getNeed(ware, EPopulationClass.RICH, city.getPopulation(EPopulationClass.RICH));
			int cityProduction = cityProductionAndConsumptionService.getProductionOutputCurrentWeek(ware, city);
			if (need>cityProduction) {
				int contribution = city.getContribution(player, ware);
				// TODO with a certain difficulty negative values might result in a penalty in the reputation
				if (contribution > 0) {
					if (contribution > need) {
						reputation += 100;
					} else {
						reputation += contribution / need * 100;
					}
				} else {
					int inStorage = city.getWare(ware).getAmount();
					if (inStorage < 20) {
						if (contribution == 0) {
							reputation += -100 * negativeFactor;
						} else if (contribution < 0 && need > 0) {
							reputation += Math.min(-100, contribution / need * 100) * negativeFactor;
						}
					}
				}
			}
		} // end for
		return (int) Math.rint(reputation/wares.length);
	}
	@VisibleForTesting
	double getNegativeReputationFactor(IPlayer player, ICity city) {
		if (city.equals(player.getHometown())) {
			return 1;
		}
		// Building Permission
        if (!city.hasBuildingPermission(player)) {
		    return 0.05;
        }
		if (!player.findTradingOffice(city).isPresent()) {
		    return 0.12;
        }
        int nbBuildings = player.findBuildings(city).size();
		if (nbBuildings == 1) { // trading office
		    return 0.18;
        }
        if (nbBuildings <= 10) {
		    return 0.3;
        }
        if (nbBuildings <= 20) {
		    return 0.6;
        }
        if (nbBuildings <= 30) {
		    return 0.8;
        }
        return 1;
	}

    /**
     * Remap the linear reputation stored in a city to value that are used in
     * ranks.
     * @param city
     * @param player
     * @return
     */
	public double remapReputation(ICity city, IPlayer player) {
       int reputation = city.getReputation(player).getPopularity();
       return reputation < 1 ? 0 : Math.log10(reputation);
    }

	/**
	 * Calculate the current career level of the player.
	 * @param city
	 * @param player
	 * @return
	 */
    public ICareer calculateCareer(ICity city, IPlayer player) {
		ESocialRank rank = player.getRank();
		int nbAvailableLevels = getAvailableLevels(rank, player);
		double totalReputationDelta = calculateDelta(rank);
		double reputation = remapReputation(city, player);
		double startLevelRep = rank.getSocialRank();
		double reqiredRepTillNextRank = Math.max(reputation - startLevelRep, 0);
		double percentage = reqiredRepTillNextRank / totalReputationDelta;
		int index = (int) Math.rint(nbAvailableLevels * percentage);
		if (player.getCriminalDrive() > 0) {
			return EMilitantCareer.values()[rank.getCareerLowerOffset()+index];
		} else {
			return EEconomicCareer.values()[rank.getCareerLowerOffset()+index];
		}
	}
	@VisibleForTesting
	double calculateDelta(ESocialRank rank) {
		if (rank == ESocialRank.ALDERMAN) {
			return calculateDelta(ESocialRank.MAYOR);
		}
		double upperEnd = rank.getNextRank().getSocialRank();
		return upperEnd - rank.getSocialRank();
	}

	@VisibleForTesting
	int getAvailableLevels(ESocialRank rank, IPlayer player) {
		int total = 0;
		if (player.getCriminalDrive() > 0) {
			total = EMilitantCareer.values().length;
		} else {
			total = EEconomicCareer.values().length;
		}
		total -= rank.getCareerLowerOffset();
		total -= rank.getCareerUpperOffset();
		return total;
	}


}
