package ch.sahits.game.openpatrician.engine.player;

import ch.sahits.game.openpatrician.utilities.annotation.ClassCategory;
import ch.sahits.game.openpatrician.utilities.annotation.EClassCategory;
import ch.sahits.game.openpatrician.utilities.annotation.LazySingleton;
import ch.sahits.game.openpatrician.clientserverinterface.service.ShipService;
import ch.sahits.game.openpatrician.engine.AIPlayerEngine;
import ch.sahits.game.openpatrician.model.IAIPlayer;
import ch.sahits.game.openpatrician.model.city.ICity;
import ch.sahits.game.openpatrician.model.city.IShipDueDate;
import ch.sahits.game.openpatrician.model.city.IShipyard;
import ch.sahits.game.openpatrician.model.map.IMap;
import ch.sahits.game.openpatrician.model.product.ISpecialMission;
import ch.sahits.game.openpatrician.model.product.ITradeMissionData;
import ch.sahits.game.openpatrician.model.sea.TravellingVessels;
import ch.sahits.game.openpatrician.model.ship.INavigableVessel;
import org.springframework.beans.factory.annotation.Autowired;

import java.util.List;

/**
 * This trade checker verifies that each ship is currently in active tradeing or there is another good reason for it.
 * @author Andi Hotz, (c) Sahits GmbH, 2016
 *         Created on Jul 18, 2016
 */
@LazySingleton
@ClassCategory(EClassCategory.SINGLETON_BEAN)
public class AITradeChecker {
    @Autowired
    private TravellingVessels vessels;
    @Autowired
    private IMap map;
    @Autowired
    private AIPlayerEngine playerEngine;
    @Autowired
    private ShipService shipService;

    /**
     * Check if the vessel is engaged in trading. That means it is either traveling, trading or repairing.
     * @param vessel
     * @return
     */
    public boolean isEngagedInTrading(INavigableVessel vessel) {
//        IAIPlayer player = (IAIPlayer) vessel.getOwner();
//        if (player.waitingForTradeStepToFinish(vessel)) {
//            return true;
//        }
        // Is travelling (Check TravelingVessels)
        if (vessels.isTravelling(vessel)) {
            return true;
        }
        // Is repairing (check shipyards)
        for (ICity city : map.getCities()) {
            IShipyard shipyard = city.getCityState().getShipyardState();
            List<IShipDueDate> repairlist = shipyard.getShipRepairList();
            for (IShipDueDate dueDate : repairlist) {
                if (dueDate.getShip().equals(vessel)) {
                    return true;
                }
            }
        }
        if (vessel.getNumberOfSailors() < shipService.getMinNumberOfSailors(vessel)) {
            return true; // needs more sailors
        }
        // is trading the AIEngine is currently executing tasks
        return playerEngine.executeCurrentTrade(vessel);
    }

    /**
     * Check weather the vessel should be engaged in trading.
     * @param vessel not engaged in trading
     * @return
     */
    public boolean shouldBeTrading(INavigableVessel vessel) {
        IAIPlayer player = (IAIPlayer) vessel.getOwner();
        ITradeMissionData missionData = player.getTradeMission(vessel);
        if (missionData != null && missionData instanceof ISpecialMission) {
            return false;
        }
        return true;
    }
}
