package ch.sahits.game.openpatrician.engine.player;

import ch.sahits.game.openpatrician.utilities.annotation.ClassCategory;
import ch.sahits.game.openpatrician.utilities.annotation.EClassCategory;
import ch.sahits.game.openpatrician.utilities.annotation.LazySingleton;
import ch.sahits.game.openpatrician.clientserverinterface.model.event.BuildingPermissionState;
import ch.sahits.game.openpatrician.clientserverinterface.model.event.SocialAdvancementDeficitState;
import ch.sahits.game.openpatrician.clientserverinterface.model.event.SocialAdvancementState;
import ch.sahits.game.openpatrician.engine.land.city.ReputationCalculator;
import ch.sahits.game.openpatrician.model.Date;
import ch.sahits.game.openpatrician.model.DisplayStateMessage;
import ch.sahits.game.openpatrician.model.ICompany;
import ch.sahits.game.openpatrician.model.IHumanPlayer;
import ch.sahits.game.openpatrician.model.IPlayer;
import ch.sahits.game.openpatrician.model.PlayerList;
import ch.sahits.game.openpatrician.model.city.ICity;
import ch.sahits.game.openpatrician.model.city.guild.GuildList;
import ch.sahits.game.openpatrician.model.city.guild.IGuild;
import ch.sahits.game.openpatrician.model.event.TargetedEvent;
import ch.sahits.game.openpatrician.model.map.IMap;
import ch.sahits.game.openpatrician.model.personal.ESocialRank;
import ch.sahits.game.openpatrician.model.personal.ICareer;
import com.google.common.eventbus.AsyncEventBus;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;

/**
 * This checker checks at the end of month if the next social rank can be reached
 * @author Andi Hotz, (c) Sahits GmbH, 2012
 * Created on Jul 24, 2012
 *
 */
@LazySingleton
@ClassCategory(EClassCategory.SINGLETON_BEAN)
public class SocialRankChecker {
	private final Logger logger = LogManager.getLogger(getClass());
	@Autowired
	@Qualifier("serverClientEventBus")
	private AsyncEventBus clientServerEventBus;
	@Autowired
	private PlayerList players;
	@Autowired
	private ReputationCalculator repCalculator;
	@Autowired
	private GuildList guildList;
	@Autowired
	private Date date;
	@Autowired
	private IMap map;


	public void doMonthlyCheck() { // TODO: andi 1/9/17 appropriate messages for mayor, alderman, tradesman
        for (IPlayer player : players) {
            ESocialRank rank = player.getRank();
            if (isNextRankAchievable(player)){
				ESocialRank nextRank = rank.getNextRank();
                player.updateRank(nextRank);
                if (player instanceof IHumanPlayer) {
					SocialAdvancementState state = SocialAdvancementState.builder()
							.location(player.getHometown().getName())
							.rank(nextRank)
							.date(date.getCurrentDate())
							.build();
					DisplayStateMessage message = new DisplayStateMessage("ch.sahits.game.openpatrician.engine.event.EventEngine.message.socialAdvancement.title", state, new Object[0]);
					clientServerEventBus.post(new TargetedEvent((IHumanPlayer) player, message));
//					clientServerEventBus.post(new TargetedEvent((IHumanPlayer) player, state));
				}
            } else if (player instanceof IHumanPlayer && rank!=ESocialRank.PATRICIAN && rank!=ESocialRank.MAYOR && rank!=ESocialRank.ALDERMAN) {
				boolean moneyMissing = notEnoughMoney(player);
				boolean reputationLow = isReputationTooLow(player);
				SocialAdvancementDeficitState state = SocialAdvancementDeficitState.builder()
						.date(date.getCurrentDate())
						.location(player.getHometown().getName())
						.money(moneyMissing)
						.reputation(reputationLow)
						.build();
				DisplayStateMessage message = new DisplayStateMessage("ch.sahits.game.openpatrician.engine.event.EventEngine.message.socialAdvancement.title", state, new Object[0]);
				clientServerEventBus.post(new TargetedEvent((IHumanPlayer) player, message));
//				clientServerEventBus.post(new TargetedEvent((IHumanPlayer) player, state));
			}
            ICareer career = repCalculator.calculateCareer(player.getHometown(), player);
            player.setCareerLevel(career);
            for (ICity city : map.getCities()) {
                if (!city.hasBuildingPermission(player)) {
                    double rep = repCalculator.remapReputation(city, player);
                    if (rep > ESocialRank.BARGAINER.getSocialRank()) {
                        city.addBuildingPermission(player);
                        BuildingPermissionState state = BuildingPermissionState.builder()
                                .date(date.getCurrentDate())
                                .location(city.getName())
                                .player(player)
                                .build();
                        if (player instanceof IHumanPlayer) {
                            DisplayStateMessage message = new DisplayStateMessage("ch.sahits.game.openpatrician.engine.event.EventEngine.message.buildingPermission.title", state, new Object[]{city.getName()});
                            clientServerEventBus.post(new TargetedEvent((IHumanPlayer) player, message));
                        }
                    }
                }
            }
        }
	}

	private boolean isNextRankAchievable(IPlayer player){
		ESocialRank rank = player.getRank();
		if (rank==ESocialRank.PATRICIAN || rank==ESocialRank.MAYOR || rank==ESocialRank.ALDERMAN){
			return false;
		}
		final ICompany company = player.getCompany();
		ICity homeTown = player.getHometown();
		long compValue = company.getCompanyValue();
		ESocialRank nextRank = rank.getNextRank();
		double remapedRep = repCalculator.remapReputation(homeTown, player);
		if (compValue>=nextRank.getCompanyValue() && remapedRep>=nextRank.getSocialRank()){
			if (nextRank==ESocialRank.COUNCILMAN){
                IGuild guild = guildList.findGuild(homeTown).get();
                return guild.getMembers().contains(player);
			}
			return true;
		}
		return false;
	}

	private boolean isReputationTooLow(IPlayer player) {
		ESocialRank rank = player.getRank();
		ESocialRank nextRank = rank.getNextRank();
		double remapedRep = repCalculator.remapReputation(player.getHometown(), player);
		return remapedRep < nextRank.getSocialRank();
	}

	private boolean notEnoughMoney(IPlayer player) {
		ESocialRank rank = player.getRank();
		ESocialRank nextRank = rank.getNextRank();
		final ICompany company = player.getCompany();
		long compValue = company.getCompanyValue();
		return compValue < nextRank.getCompanyValue();
	}

}
