package ch.sahits.game.openpatrician.engine.player.strategy;

import ch.sahits.game.openpatrician.utilities.annotation.ClassCategory;
import ch.sahits.game.openpatrician.utilities.annotation.EClassCategory;
import ch.sahits.game.openpatrician.utilities.annotation.Prototype;
import ch.sahits.game.openpatrician.clientserverinterface.service.CelebrationService;
import ch.sahits.game.openpatrician.engine.player.CollectWaresForCity;
import ch.sahits.game.openpatrician.engine.player.CollectWaresMissionData;
import ch.sahits.game.openpatrician.model.IAIPlayer;
import ch.sahits.game.openpatrician.model.building.ITradingOffice;
import ch.sahits.game.openpatrician.model.city.EPopulationClass;
import ch.sahits.game.openpatrician.model.city.ICity;
import ch.sahits.game.openpatrician.model.player.IAITradeStrategyType;
import ch.sahits.game.openpatrician.model.product.IWare;
import ch.sahits.game.openpatrician.model.ship.INavigableVessel;
import com.google.common.base.Preconditions;
import com.thoughtworks.xstream.annotations.XStreamOmitField;
import lombok.Setter;
import org.springframework.beans.factory.annotation.Autowired;

import java.util.HashMap;
import java.util.Map;
import java.util.Optional;

/**
 * @author Andi Hotz, (c) Sahits GmbH, 2016
 *         Created on Dec 18, 2016
 */
@Prototype
@ClassCategory(EClassCategory.PROTOTYPE_BEAN)
public class CollectCelebrationWaresStrategy extends CollectionWaresStrategy {
    @XStreamOmitField // Passed trhough constructor and then really stored in the mission data.
    @Setter
    private ICity celebrationLocation;
    @Autowired
    private CelebrationService celebrationService;


    @Override
    protected CollectWaresMissionData createMissionData(IAIPlayer player) {
        Preconditions.checkNotNull(celebrationLocation);
        ICity where = celebrationLocation;
        CollectWaresForCity missionData = new CollectWaresForCity();
        missionData.setCity(where);
        Map<EPopulationClass, Integer> population = new HashMap<>();
        for (EPopulationClass populationClass : EPopulationClass.values()) {
            population.put(populationClass, where.getPopulation(populationClass));
        }
        Map<IWare, Integer> requirements = celebrationService.calculateRequiredAmounts(population);
        Optional<ITradingOffice> optOffice = player.findTradingOffice(where);
        for (IWare ware : requirements.keySet()) {
            int amount;
            if (!optOffice.isPresent()) {
               amount = requirements.get(ware);
            } else {
                amount = Math.max(requirements.get(ware) - optOffice.get().getWare(ware).getAmount(), 0);
            }
            missionData.require(ware, amount);
        }
        missionData.setInitialRequirements(missionData.getRequiredWareAmounts());
        return missionData;
    }

    @Override
    protected boolean checkAllWaresCollected(ICity city, INavigableVessel vessel, CollectWaresMissionData tradeMission, Optional<ITradingOffice> tradingOffice) {
        // wares are either on ship or in storage in target city
        Map<IWare, Integer> initialRequirements = ((CollectWaresForCity) tradeMission).getInitialRequirements();
        for (IWare ware : initialRequirements.keySet()) {
            int amount;
            if (!tradingOffice.isPresent() || !city.equals(celebrationLocation)) {
                amount = initialRequirements.get(ware);
            } else {
                amount = Math.max(initialRequirements.get(ware) - tradingOffice.get().getWare(ware).getAmount(), 0);
            }
            amount = Math.max(amount - vessel.getWare(ware).getAmount(), 0);
            tradeMission.require(ware, amount);
        }
        for (Integer amount : tradeMission.getRequiredWareAmounts().values()) {
            if (amount > 0) {
                return false;
            }
        }
        return true;
    }

    @Override
    protected IAITradeStrategyType getStrategyType() {
        return ECollectWareStrategyType.CELEBRATION;
    }
}
