package ch.sahits.game.openpatrician.engine.player.strategy;

import ch.sahits.game.openpatrician.model.building.ISteward;
import ch.sahits.game.openpatrician.utilities.annotation.ClassCategory;
import ch.sahits.game.openpatrician.utilities.annotation.EClassCategory;
import ch.sahits.game.openpatrician.utilities.annotation.LazySingleton;
import ch.sahits.game.openpatrician.clientserverinterface.model.factory.ShipFactory;
import ch.sahits.game.openpatrician.clientserverinterface.service.TradeService;
import ch.sahits.game.openpatrician.engine.player.CollectConstructionWaresMissionData;
import ch.sahits.game.openpatrician.engine.player.CollectWaresMissionData;
import ch.sahits.game.openpatrician.model.IAIPlayer;
import ch.sahits.game.openpatrician.model.building.ITradingOffice;
import ch.sahits.game.openpatrician.model.city.ICity;
import ch.sahits.game.openpatrician.model.product.EWare;
import ch.sahits.game.openpatrician.model.ship.EShipType;
import ch.sahits.game.openpatrician.model.ship.INavigableVessel;
import org.springframework.beans.factory.annotation.Autowired;

import java.util.Optional;

/**
 * Base strategy for constructing a ship.
 * @author Andi Hotz, (c) Sahits GmbH, 2016
 *         Created on Jul 26, 2016
 */
@LazySingleton
@ClassCategory(EClassCategory.SINGLETON_BEAN)
public abstract class CollectConstructionWareStrategy extends CollectionWaresStrategy {
    private final EShipType constructionType;
    @Autowired
    private ShipFactory shipFactory;
    @Autowired
    private TradeService tradeService;

    public CollectConstructionWareStrategy(EShipType constructionType) {
        this.constructionType = constructionType;
    }
    protected CollectWaresMissionData createMissionData(IAIPlayer player) {
        ICity where = getConstructionCity(player);
        CollectConstructionWaresMissionData missionData = new CollectConstructionWaresMissionData();
        missionData.setCity(where);
        missionData.setShipType(constructionType);
        for (EWare ware : EWare.values()) {
            int amount = shipFactory.getConstructionAmount(constructionType, ware);
            if (amount > 0) {
                Optional<ITradingOffice> optOffice = player.findTradingOffice(where);
                if (optOffice.isPresent()) {
                    Optional<ISteward> steward = optOffice.get().getSteward();
                    tradeService.buyFromCityToStorage(optOffice.get(), player, where, ware, amount, Optional.of(ware.getMaxBuyPriceModerate()), steward);
                    amount = Math.max(0,amount - optOffice.get().getWare(ware).getAmount());
                }
                missionData.require(ware, amount);
            }
        }
        return missionData;
    }

    /**
     * Define where the ship should be constructed.
     * @param player
     * @return
     */
    protected abstract ICity getConstructionCity(IAIPlayer player);
    @Override
    protected boolean checkAllWaresCollected(ICity city, INavigableVessel vessel, CollectWaresMissionData tradeMission, Optional<ITradingOffice> tradingOffice) {
        boolean haveEverything;
        haveEverything = true;
        for (EWare ware : EWare.values()) {
            int amount = shipFactory.getConstructionAmount(((CollectConstructionWaresMissionData)tradeMission).getShipType(), ware);
            // check amount on ship and in storage and city
            if (amount > 0) {
                int availableAmount = 0;
                if (tradingOffice.isPresent()) {
                    availableAmount += tradingOffice.get().getWare(ware).getAmount();
                }
                availableAmount += vessel.getWare(ware).getAmount();
                availableAmount += city.getWare(ware).getAmount();
                if (availableAmount < amount) {
                    haveEverything = false;
                    break;
                }
            }
        }
        return haveEverything;
    }
}
