package ch.sahits.game.openpatrician.engine.player.strategy;

import ch.sahits.game.openpatrician.clientserverinterface.service.ShipService;
import ch.sahits.game.openpatrician.engine.land.city.ShipyardEngine;
import ch.sahits.game.openpatrician.engine.sea.DangerService;
import ch.sahits.game.openpatrician.engine.sea.model.PirateActivity;
import ch.sahits.game.openpatrician.engine.sea.model.PirateActivityEntry;
import ch.sahits.game.openpatrician.model.city.ICity;
import ch.sahits.game.openpatrician.model.city.IShipyard;
import ch.sahits.game.openpatrician.model.player.IAIShipUpgradeStrategy;
import ch.sahits.game.openpatrician.model.ship.EShipUpgrade;
import ch.sahits.game.openpatrician.model.ship.IShip;
import ch.sahits.game.openpatrician.utilities.annotation.ClassCategory;
import ch.sahits.game.openpatrician.utilities.annotation.EClassCategory;
import ch.sahits.game.openpatrician.utilities.annotation.LazySingleton;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.springframework.beans.factory.annotation.Autowired;

import java.util.List;

/**
 * Upgrade strategy for hte ship based on the danger.
 * @author Andi Hotz, (c) Sahits GmbH, 2017
 * Created on Nov 04, 2017
 */
@ClassCategory(EClassCategory.SINGLETON_BEAN)
@LazySingleton
public class DangerBasedShipUpgradeStrategy implements IAIShipUpgradeStrategy {
    private final Logger logger = LogManager.getLogger(getClass());
    @Autowired
    private ShipService shipService;
    @Autowired
    private DangerService dangerService;
    @Autowired
    private PirateActivity pirateActivity;
    @Autowired
    private ShipyardEngine shipyardEngine;

    @Override
    public boolean shouldUpgrade(IShip ship) {
        if (ship.getShipUpgradeLevel() == EShipUpgrade.LEVEL2) {
            return false;
        }
        int weaponsStrength = shipService.calculateShipsWeaponsStrength(ship);
        List<PirateActivityEntry> attacks = pirateActivity.getPirateActivity();
        int observationDuration = pirateActivity.getObservationPeriodInDays();
        int nbAttacks = dangerService.getNumberOfPirateAttacks(attacks);
        double twoWeeks = observationDuration / 14.0;
        double attacksInTwoWeeks = twoWeeks / nbAttacks;
        if (attacksInTwoWeeks > 1) { // There should be one attack in two weeks
            if (weaponsStrength < 4) {
                return true;
            }
        }
        if (attacksInTwoWeeks > 0) { // there should be one attack in four weeks / or half an attack in two
            if (weaponsStrength < 6) {
                return true;
            }
        }
        return false;
    }

    @Override
    public void upgrade(IShip ship, ICity city) {
        IShipyard shipyard = city.getCityState().getShipyardState();
        shipyardEngine.refitAIShip(shipyard, ship);
        logger.debug("Refit vessel {} in {}", ship.getName(), city.getName());

    }
}
