package ch.sahits.game.openpatrician.engine.event;

import ch.sahits.game.openpatrician.model.ship.INavigableVessel;
import ch.sahits.game.openpatrician.utilities.annotation.ClassCategory;
import ch.sahits.game.openpatrician.utilities.annotation.EClassCategory;
import ch.sahits.game.openpatrician.utilities.annotation.LazySingleton;
import ch.sahits.game.openpatrician.utilities.annotation.OptionalType;
import com.google.common.base.Preconditions;

import java.util.Optional;

/**
 * State of the AIPlayerEngine that holds the serializable part for the engine.
 * Created by andi on 01.04.17.
 */
@LazySingleton
@ClassCategory({EClassCategory.SINGLETON_BEAN, EClassCategory.SERIALIZABLE_BEAN})
public class AIPlayerEngineState {
    @OptionalType(INavigableVessel.class)
    private Optional<INavigableVessel> currentVesselExecutingTrade = Optional.empty();

    /**
     * Activate a vessel for engaging in a trade step. Before a vessel is activated, no other vessel may be active.
     * @param vessel to be activated.
     */
    public synchronized void activateVesselForTrade(INavigableVessel vessel) {
        Preconditions.checkArgument(!currentVesselExecutingTrade.isPresent(), "There must not be a previous vessel set.");
        currentVesselExecutingTrade = Optional.of(vessel);
    }

    /**
     * Deactive a vessel for trading. The vessel must currently be active
     * @param vessel to be deactivated.
     */
    public synchronized void deactivateVesselForTrade(INavigableVessel vessel) {
        Preconditions.checkArgument(currentVesselExecutingTrade.isPresent(), "There is currently no active vessel.");
        Preconditions.checkArgument(currentVesselExecutingTrade.get().equals(vessel), "The vessel to deactivate is currently not active");
        currentVesselExecutingTrade = Optional.empty();
    }

    /**
     * Check if a vessel is currently the active vessel doing trading.
     * @param vessel to be checked.
     * @return true if the <code>vessel</code> is the one actively trading.
     */
    public synchronized boolean isActive(INavigableVessel vessel) {
        return currentVesselExecutingTrade.isPresent() && currentVesselExecutingTrade.get().equals(vessel);
    }
}
