package ch.sahits.game.openpatrician.engine.event;

import ch.sahits.game.event.data.ClockTickDayChange;
import ch.sahits.game.event.data.PeriodicalTimeMonthEndUpdate;
import ch.sahits.game.event.data.PeriodicalTimeWeekEndUpdate;
import ch.sahits.game.openpatrician.clientserverinterface.event.PostponedDisplayDialogMessage;
import ch.sahits.game.openpatrician.clientserverinterface.model.event.CelebrationState;
import ch.sahits.game.openpatrician.clientserverinterface.model.event.ChildBirthState;
import ch.sahits.game.openpatrician.clientserverinterface.model.event.ChildDeathState;
import ch.sahits.game.openpatrician.clientserverinterface.model.event.DowryState;
import ch.sahits.game.openpatrician.clientserverinterface.model.event.FireState;
import ch.sahits.game.openpatrician.clientserverinterface.model.event.MarriageBrokerAnnouncementState;
import ch.sahits.game.openpatrician.clientserverinterface.model.event.MarriageCelebrationState;
import ch.sahits.game.openpatrician.clientserverinterface.model.event.PlagueState;
import ch.sahits.game.openpatrician.clientserverinterface.model.event.SpouseDeathState;
import ch.sahits.game.openpatrician.clientserverinterface.model.factory.ShipFactory;
import ch.sahits.game.openpatrician.clientserverinterface.service.DialogTemplateFactory;
import ch.sahits.game.openpatrician.clientserverinterface.service.DialogTemplateParameterSupplier;
import ch.sahits.game.openpatrician.clientserverinterface.service.EDialogTemplateType;
import ch.sahits.game.openpatrician.engine.AbstractEngine;
import ch.sahits.game.openpatrician.engine.event.task.ServerSideTaskFactory;
import ch.sahits.game.openpatrician.engine.player.IAIEventHandler;
import ch.sahits.game.openpatrician.engine.player.SocialRankChecker;
import ch.sahits.game.openpatrician.event.data.DisplayEventVideo;
import ch.sahits.game.openpatrician.event.data.ShipEntersPortEvent;
import ch.sahits.game.openpatrician.model.Date;
import ch.sahits.game.openpatrician.model.DateService;
import ch.sahits.game.openpatrician.model.IAIPlayer;
import ch.sahits.game.openpatrician.model.IHumanPlayer;
import ch.sahits.game.openpatrician.model.IPlayer;
import ch.sahits.game.openpatrician.model.PlayerList;
import ch.sahits.game.openpatrician.model.building.IBuilding;
import ch.sahits.game.openpatrician.model.building.IHospital;
import ch.sahits.game.openpatrician.model.building.IWell;
import ch.sahits.game.openpatrician.model.city.EPopulationClass;
import ch.sahits.game.openpatrician.model.city.ICity;
import ch.sahits.game.openpatrician.model.city.impl.ECityState;
import ch.sahits.game.openpatrician.model.event.EEventMediaType;
import ch.sahits.game.openpatrician.model.event.TargetedEvent;
import ch.sahits.game.openpatrician.model.event.TimedUpdatableTaskList;
import ch.sahits.game.openpatrician.model.initialisation.StartNewGameBean;
import ch.sahits.game.openpatrician.model.map.IMap;
import ch.sahits.game.openpatrician.model.personal.IChild;
import ch.sahits.game.openpatrician.model.personal.IReputation;
import ch.sahits.game.openpatrician.model.personal.ISpouseData;
import ch.sahits.game.openpatrician.model.personal.impl.Child;
import ch.sahits.game.openpatrician.model.product.EWare;
import ch.sahits.game.openpatrician.model.ship.EShipType;
import ch.sahits.game.openpatrician.model.ship.INavigableVessel;
import ch.sahits.game.openpatrician.model.ship.IShip;
import ch.sahits.game.openpatrician.model.ui.DialogTemplate;
import ch.sahits.game.openpatrician.model.ui.TargetedDialogStateWrapper;
import ch.sahits.game.openpatrician.utilities.annotation.ClassCategory;
import ch.sahits.game.openpatrician.utilities.annotation.DependentInitialisation;
import ch.sahits.game.openpatrician.utilities.annotation.EClassCategory;
import ch.sahits.game.openpatrician.utilities.service.DisableProperties;
import ch.sahits.game.openpatrician.utilities.service.RandomNameLoader;
import com.google.common.annotations.VisibleForTesting;
import com.google.common.eventbus.AsyncEventBus;
import com.google.common.eventbus.Subscribe;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.context.annotation.Lazy;
import org.springframework.stereotype.Service;

import javax.annotation.PostConstruct;
import javax.annotation.PreDestroy;
import java.time.LocalDateTime;
import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import java.util.Optional;
import java.util.Random;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.TimeUnit;
import java.util.stream.Collectors;

/**
 * This event engine triggers the various random events as well as handling the
 * messages from the event bus that result from such an event.
 * @author Andi Hotz, (c) Sahits GmbH, 2016
 *         Created on Dec 09, 2016
 */
@Service
@Lazy
@DependentInitialisation(StartNewGameBean.class)
@ClassCategory(EClassCategory.SINGLETON_BEAN)
public class EventEngine extends AbstractEngine {
    private final Logger logger = LogManager.getLogger(getClass());
    @Autowired
    private Random rnd;
    @Autowired
    @Qualifier("serverClientEventBus")
    private AsyncEventBus clientServerEventBus;
    @Autowired
    @Qualifier("timerEventBus")
    private AsyncEventBus timerEventBus;
    @Autowired
    private Date date;
    @Autowired
    @Qualifier("serverTimer")
    private ScheduledExecutorService executor;
    @Autowired
    private ShipFactory shipFactory;
    @Autowired
    private IAIEventHandler aiEventHandler;
    @Autowired
    private SocialRankChecker rankChecker;
    @Autowired
    private PlayerList players;
    @Autowired
    private IMap map;
    @Autowired
    private TimedUpdatableTaskList taskList;
    @Autowired
    private DateService dateService;
    @Autowired
    private DisableProperties disableService;
    @Autowired
    private ServerSideTaskFactory taskFactory;
    @Autowired
    private EventService eventService;
    @Autowired
    private EventEngineState eventState;
    @Autowired
    private DialogTemplateFactory dialogTemplateFactory;


    private static RandomNameLoader shipLoader = new RandomNameLoader("shipnames.properties");
    private static RandomNameLoader maleFirstNameLoader = new RandomNameLoader("firstnames.properties");
    private static RandomNameLoader femaleFirstNameLoader = new RandomNameLoader("female_firstnames.properties");
    private static RandomNameLoader lastNameLoader = new RandomNameLoader("lastnames.properties");


    @PostConstruct
    private void init() {
        clientServerEventBus.register(this);
        timerEventBus.register(this);
    }
    @PreDestroy
    private void unregister() {
        clientServerEventBus.unregister(this);
        timerEventBus.unregister(this);
    }
    @Override
    public List<AbstractEngine> getChildren() {
        return new ArrayList<>();
    }

    /**
     * Handle events that are targeted at a player and contain a dialog state.
     * @param dialogState wrapped dialog state that is to be handled
     */
    @Subscribe
    public void handleWrappedStateEvents(TargetedDialogStateWrapper dialogState) {
        if (disableService.randomEventsEnabled()) {
            if (dialogState.getState() instanceof MarriageCelebrationState) {
                handleMarriageState(dialogState);
            } else if (dialogState.getState() instanceof CelebrationState) {
                postToHumanPlayer(dialogState, "ch.sahits.game.openpatrician.engine.event.EventEngine.message.celebration.title");
            } else {
                throw new IllegalStateException("No handling implemented for " + dialogState.getState().getClass().getName());
            }
        }
    }
    @VisibleForTesting
    void handleMarriageState(TargetedDialogStateWrapper wrappedDialogState) {
        IPlayer player = wrappedDialogState.getPlayer();
        MarriageCelebrationState marriageState = (MarriageCelebrationState) wrappedDialogState.getState();
        ISpouseData spouse = marriageState.getSpouseData();
        player.marry(spouse);
        eventState.resetMarriedState(player);
        // Dowry
        long value = player.getCompany().getCompanyValue();
        IShip ship;
        ICity birthPlace = spouse.getBirthPlace();
        double x = birthPlace.getCoordinates().getX();
        if (value < 200000) {
            // Snaikka
            int capacity = shipFactory.calculateInitialCapacity(EShipType.SNAIKKA, x);
            ship = shipFactory.createSnaikka(shipLoader.getRandomName(), capacity);
        } else if (value < 1000000) {
            // Crayer
            int capacity = shipFactory.calculateInitialCapacity(EShipType.CRAYER, x);
            ship = shipFactory.createCrayer(shipLoader.getRandomName(), capacity);
        } else {
            // Cog
            int capacity = shipFactory.calculateInitialCapacity(EShipType.COG, x);
            ship = shipFactory.createCog(shipLoader.getRandomName(), capacity);
        }
        ship.damage(rnd.nextInt(50) + 10, false);
        ship.setLocation(player.getHometown().getCoordinates());
        ship.setOwner(player);
        player.addShip(ship);
        // Decide on ware
        if (rnd.nextBoolean()) {
            EWare ware = EWare.values()[rnd.nextInt(EWare.values().length)];
            int amount;
            if (ware.isBarrelSizedWare()) {
                amount = rnd.nextInt(ship.getCapacity()/2);
            } else {
                amount = rnd.nextInt(ship.getCapacity()/20);
            }
            ship.load(ware, amount, 0);
        }
        IReputation rep = birthPlace.getReputation(player);
        int repConnectionCounsilman = 1500;
        int updateReputation = (int)(0.5 * spouse.getPopularityInHerHomeTown());
        updateReputation += spouse.getConnectionsInHerHomeTown() * repConnectionCounsilman;
        updateReputation += spouse.getConnectionsInHerHomeTown() * 2 * repConnectionCounsilman;
        rep.update(updateReputation);
        rep = player.getHometown().getReputation(player);
        updateReputation = (int)(0.5 * spouse.getPopularityInYourHomeTown());
        updateReputation += spouse.getConnectionsInYourHomeTown() * repConnectionCounsilman;
        updateReputation += spouse.getConnectionsInYourHomeTown() * 2 * repConnectionCounsilman;
        rep.update(updateReputation);
        if (player instanceof IHumanPlayer) {
            String descriptionKey = "ch.sahits.game.openpatrician.display.dialog.event.CelebrationDialog."+marriageState.getSuccess().name();
            DisplayEventVideo event = DisplayEventVideo.builder()
                    .mediaType(EEventMediaType.CELEBRATION)
                    .durationInSeconds(20)
                    .titleKey("ch.sahits.game.openpatrician.engine.event.EventEngine.message.wedding.title")
                    .titleParams(new Object[0])
                    .descriptionKey(descriptionKey)
                    .descriptionParams(new Object[]{marriageState.getAmountGuests()})
                    .build();
                    clientServerEventBus.post(new TargetedEvent((IHumanPlayer) player, event));
            // trigger dowry message
            DowryState state = DowryState.builder()
                    .location(player.getHometown().getName())
                    .date(date.getCurrentDate())
                    .ship(ship)
                    .build();
            executor.schedule(() -> clientServerEventBus.post(new TargetedEvent((IHumanPlayer) player, wrappedDialogState)), 5, TimeUnit.SECONDS);
            executor.schedule(() -> {
                if (player instanceof IHumanPlayer) {
                    clientServerEventBus.post(new TargetedDialogStateWrapper(player, state));
                    eventService.postToHumanPlayer(state, player, "ch.sahits.game.openpatrician.engine.event.EventEngine.message.dowry.title");
                }
            }, 10, TimeUnit.SECONDS);
        } else {
            // Trigger ship arriving in city
            logger.debug("Dowry ship {} of {} {} enters port", ship.getName(), player.getName(), player.getLastName());
            aiEventHandler.initializeNewShip(ship, (IAIPlayer) player);
        }
    }

    private void postToHumanPlayer(TargetedDialogStateWrapper dialogState, String messageKey, Object... messageParams) {
        IPlayer player = dialogState.getPlayer();
        eventService.postToHumanPlayer(dialogState.getState(), player, messageKey, messageParams);
    }



    /**
     * Trigger the event for a marriage broker.
     * @param event ship entered port
     */
    @Subscribe
    public void handleShipEntersPort(ShipEntersPortEvent event) {
        INavigableVessel ship = event.getShip();
        if (ship.getOwner() instanceof IPlayer && disableService.randomEventsEnabled()) {
            IPlayer player = (IPlayer) ship.getOwner();
            // Marriage broker to initialize marriage
            // unmarried and not already in negotiation.
            Optional<EEventState> marriageState = eventState.getMarriageState(player);
            boolean meetBroker = rnd.nextInt(30) < 2; // 5% chance to meet a broker.
            if (!player.getSpouseData().isPresent() && marriageState != null && meetBroker && !marriageState.isPresent()) {
                if (player instanceof IAIPlayer) {
                    MarriageBrokerAnnouncementState state = createMarriageBrokerAnnouncementState(event, ship);
                    aiEventHandler.handleMarriageEvent((IAIPlayer) player, state);
                } else if (player instanceof IHumanPlayer) {
                    MarriageBrokerAnnouncementState state = createMarriageBrokerAnnouncementState(event, ship);
                    eventService.postToHumanPlayer(state, player, "ch.sahits.game.openpatrician.engine.event.EventEngine.message.marriageOffer.title");
                    clientServerEventBus.post(new TargetedEvent((IHumanPlayer) player, state));
                    eventState.setMarriageState(player, EEventState.MARRIAGE_UNDER_CONSIDERATION);
                }
            }
        }
    }

    private MarriageBrokerAnnouncementState createMarriageBrokerAnnouncementState(ShipEntersPortEvent event, INavigableVessel ship) {
        return MarriageBrokerAnnouncementState.builder()
                .date(date.getCurrentDate())
                .location(event.getCity().getName())
                .fromFirstName(maleFirstNameLoader.getRandomName())
                .fromLastName(lastNameLoader.getRandomName())
                .toLastName(ship.getOwner().getLastName())
                .genderMale(((IPlayer) ship.getOwner()).getPersonalData().isMale())
                .build();
    }

    /**
     * Trigger events to be checked at the end of week like sending the bill statement.
     * @param event week ended
     */
    @Subscribe
    public void handleEndOfWeekUpdates(PeriodicalTimeWeekEndUpdate event) {

    }

    /**
     * Trigger events at the end of month like rank update.
     * @param event month end
     */
    @Subscribe
    public void handleEndOfMonthUpdates(PeriodicalTimeMonthEndUpdate event) {
        rankChecker.doMonthlyCheck();
    }

    /**
     * Trigger events that can happen any day like plague, fire, child birth
     * @param event daily updates
     */
    @Subscribe
    public void handleDailyUpdates(ClockTickDayChange event) {
        if (disableService.randomEventsEnabled()) {
            boolean catastopheEvent = handleCatastropheEvents();
            for (IPlayer player : players) {
                boolean childBirth = false;
                boolean spouseDeath = false;
                if (player.getSpouseData().isPresent() && !catastopheEvent) {
                    childBirth = rnd.nextInt(400 * (player.getChildren().size() + 1)) == 42;
                    int spouseDeathPropLimit = 5000;
                    boolean spuseDeathSingle = rnd.nextInt(spouseDeathPropLimit) == 42;
                    if (!player.getPersonalData().isMale()) {
                        spuseDeathSingle = false;
                    }
                    spouseDeath = !childBirth && spuseDeathSingle;
                    if (childBirth) {
                        boolean male = rnd.nextBoolean();
                        String name;
                        if (male) {
                            name = maleFirstNameLoader.getRandomName();
                        } else {
                            name = femaleFirstNameLoader.getRandomName();
                        }
                        ChildBirthState state = ChildBirthState.builder()
                                .location(player.getHometown().getName())
                                .date(date.getCurrentDate())
                                .male(male)
                                .name(name)
                                .wifeDeath(spuseDeathSingle)
                                .build();
                        Child child = Child.builder()
                                .birthDate(date.getCurrentDate())
                                .name(name)
                                .build();
                        player.getChildren().add(child);
                        if (spuseDeathSingle) {
                            player.spouseDies();
                        }
                        eventService.postToHumanPlayer(state, player, "ch.sahits.game.openpatrician.engine.event.EventEngine.message.childBirth.title");
                    }
                    if (spouseDeath) {
                        SpouseDeathState state = SpouseDeathState.builder()
                                .location(player.getHometown().getName())
                                .date(date.getCurrentDate())
                                .build();
                        player.spouseDies();
                        eventService.postToHumanPlayer(state, player, "ch.sahits.game.openpatrician.engine.event.EventEngine.message.spouseDeath.title");
                    }

                }
                handleChildDeathEvent(catastopheEvent, player, childBirth, spouseDeath);
            }
        }
    }
    @VisibleForTesting
    void handleChildDeathEvent(boolean catastopheEvent, IPlayer player, boolean childBirth, boolean spouseDeath) {
        if (!catastopheEvent && !spouseDeath && !childBirth) {
            int childDeathLimit = calculateChildDeathLimit(player);
            boolean childDeath = !spouseDeath && childDeathLimit > 0 && rnd.nextInt(childDeathLimit) == 0;
            if (childDeath) {
                List<IChild> nonAdults = filterNonAdultChildren(player);
                IChild child = nonAdults.get(rnd.nextInt(nonAdults.size()));
                ChildDeathState state = ChildDeathState.builder()
                        .location(player.getHometown().getName())
                        .date(date.getCurrentDate())
                        .child(child)
                        .build();
                List<IChild> children = player.getChildren();
                children.remove(child);
                eventService.postToHumanPlayer(state, player, "ch.sahits.game.openpatrician.engine.event.EventEngine.message.childDeath.titl");
            }
        }
    }

    @VisibleForTesting
    boolean handleCatastropheEvents() {
        boolean catastopheEvent = false;
        int fireRnd = rnd.nextInt(365 * 3);
        boolean fireProbability = fireRnd < 10; // average of 10 fires per year
        int plagueRnd = rnd.nextInt(365 * 3);
        boolean plageProbability = plagueRnd < 1; // average one plague per years
        logger.info("Random events fireRnd={}, plagueRnd={}, fireProbability={}, plageProbability={}, random event={}", fireRnd, plagueRnd, fireProbability, plageProbability, (fireProbability || plageProbability));
        if (fireProbability || plageProbability) {
            List<ICity> cities = map.getCities().stream()
                    .filter(city -> city.getCityState().cityEventProperty().get() == null).collect(Collectors.toList());
            Collections.shuffle(cities);
            for (ICity city : cities) {
                final int popultation = city.getPopulationBinding().get();
                // fire probability is reduced by one well per 500 inhabitants
                StringBuilder sb = new StringBuilder()
                        .append("Refined probability for ")
                        .append(city.getName())
                        .append(" ");
                if (fireProbability) {
                    int nbWells = city.findBuilding(IWell.class, Optional.empty()).size(); // retrieve correct count
                    double reduction = popultation / (500 * (nbWells + 1)); // will be 1 with max wells
                    fireProbability = rnd.nextDouble() * reduction * 1.5 > 1;
                    sb.append("reduction=").append(reduction).append(", fireProbability=").append(fireProbability).append(", ");
                }
                if (plageProbability && !fireProbability) {
                    double sanitatyInstitutions = city.findBuilding(IHospital.class, Optional.empty()).size() + city.getPercentageRoad();
                    // plague probability is reduced by one sanity institution per 1500 inhabitants
                    double reduction = popultation / (1500 * (sanitatyInstitutions));
                    plageProbability = rnd.nextDouble() * reduction * 1.5 > 1;
                    sb.append("reduction=").append(reduction).append(", plageProbability=").append(plageProbability).append(", ");
                }
                logger.debug(sb.toString());
                if (fireProbability) {
                    city.getCityState().cityEventProperty().setValue(ECityState.FIRE);
                    eventState.addFire(city, date.getCurrentDate());
                    int deathtoll = rnd.nextInt(popultation / 20);
                    String titleKey = "ch.sahits.game.openpatrician.engine.event.EventEngine.video.fire.title";
                    Object[] titleParams = {city.getName()};
                    int buildingTotal = 0;
                    int duration = rnd.nextInt(9) + 2; // range [2,10]
                    LocalDateTime executionDate = date.getCurrentDate().plusDays(duration);
                    for (IPlayer player : players) {
                        Optional<ISpouseData> killedSpouse = eventService.calculateSpouseDeath(player, city);
                        int destroyBuildingUpperBound = Math.max(city.findBuilding(IBuilding.class, Optional.empty()).size(), 1);
                        FireState state = FireState.builder()
                                .location(city.getName())
                                .date(date.getCurrentDate())
                                .deathtoll(deathtoll)
                                .destroyedBuildings(rnd.nextInt(destroyBuildingUpperBound))
                                .killedSpouse(killedSpouse)
                                .build();
                        buildingTotal += state.getDestroyedBuildings();
                        taskList.add(taskFactory.getPostStateDialogMessageTask(executionDate, state, player, titleKey, titleParams));
                    }
                    DisplayEventVideo event = DisplayEventVideo.builder()
                            .mediaType(EEventMediaType.FIRE)
                            .durationInSeconds(20)
                            .titleKey(titleKey)
                            .titleParams(titleParams)
                            .descriptionKey("ch.sahits.game.openpatrician.engine.event.EventEngine.video.fire.description")
                            .descriptionParams(new Object[]{city.getName()})
                            .build();
                    for (IPlayer player : players) {
                        if (player instanceof IHumanPlayer) {
                            clientServerEventBus.post(new TargetedEvent((IHumanPlayer) player, event));
                        }
                    }
                    DialogTemplateParameterSupplier supplier = new DialogTemplateParameterSupplier(new Object[]{city.getName(), buildingTotal, deathtoll, duration});
                    DialogTemplate dialogTemplate = dialogTemplateFactory.createDialogTemplate(EDialogTemplateType.FIRE_FINISHED, supplier);
                    PostponedDisplayDialogMessage dialogMessage = new PostponedDisplayDialogMessage(executionDate, dialogTemplate);
                    for (IPlayer player : players) {
                        if (player instanceof IHumanPlayer) {
                            clientServerEventBus.post(new TargetedEvent((IHumanPlayer) player, dialogMessage));
                        }
                    }
                    taskList.add(taskFactory.getUpdatePopulationTask(executionDate, EPopulationClass.POOR, -deathtoll, city));
                    taskList.add(taskFactory.getClearCityEventTask(executionDate, city));
                }
                if (plageProbability && !fireProbability) {
                    city.getCityState().cityEventProperty().setValue(ECityState.PLAGUE);
                    int duration = rnd.nextInt(180) + 30;
                    LocalDateTime plagueStart = date.getCurrentDate();
                    LocalDateTime plagueEnd = plagueStart.plusDays(duration);
                    eventState.addPlague(city, plagueStart);
                    int deathtoll = rnd.nextInt(popultation / 10);
                    String titleKey = "ch.sahits.game.openpatrician.engine.event.EventEngine.video.plague.title";
                    Object[] titleParams = {city.getName()};
                    for (IPlayer player : players) {
                        Optional<ISpouseData> killedSpouse = eventService.calculateSpouseDeath(player, city);
                        PlagueState state = PlagueState.builder()
                                .location(city.getName())
                                .date(plagueEnd)
                                .deathtoll(deathtoll)
                                .killedSpouse(killedSpouse)
                                .since(plagueStart)
                                .build();
                        taskList.add(taskFactory.getPostStateDialogMessageTask(plagueEnd, state, player, titleKey, titleParams));
                    }
                    DisplayEventVideo event = DisplayEventVideo.builder()
                            .mediaType(EEventMediaType.PLAGUE)
                            .durationInSeconds(30)
                            .titleKey(titleKey)
                            .titleParams(titleParams)
                            .descriptionKey("ch.sahits.game.openpatrician.engine.event.EventEngine.video.plague.description")
                            .descriptionParams(new Object[]{city.getName()})
                            .build();
                    for (IPlayer player : players) {
                        if (player instanceof IHumanPlayer) {
                            clientServerEventBus.post(new TargetedEvent((IHumanPlayer) player, event));
                        }
                    }
                    int durationInMonth = duration / 30;
                    DialogTemplateParameterSupplier supplier = new DialogTemplateParameterSupplier(new Object[]{city.getName(), durationInMonth, deathtoll});
                    DialogTemplate dialogTemplate = dialogTemplateFactory.createDialogTemplate(EDialogTemplateType.PLAGUE_END, supplier);
                    PostponedDisplayDialogMessage dialogMessage = new PostponedDisplayDialogMessage(plagueEnd, dialogTemplate);
                    for (IPlayer player : players) {
                        if (player instanceof IHumanPlayer) {
                            clientServerEventBus.post(new TargetedEvent((IHumanPlayer) player, dialogMessage));
                        }
                    }
                    taskList.add(taskFactory.getUpdatePopulationTask(plagueEnd, EPopulationClass.POOR, -deathtoll, city));
                    taskList.add(taskFactory.getClearCityEventTask(plagueEnd, city));
                }
                catastopheEvent = fireProbability || plageProbability;
                if (catastopheEvent) {
                    break;
                }
            } // end for loop cities
        } // end if there are catasrophes at all
        return catastopheEvent;
    }


    private int calculateChildDeathLimit(IPlayer player) {
        if (player.getChildren().isEmpty()) {
            return 0;
        }
        int noAdultChildren = filterNonAdultChildren(player).size();
        return 9000 / noAdultChildren;
    }

    private List<IChild> filterNonAdultChildren(IPlayer player) {
        return player.getChildren().stream()
                .filter(child -> dateService.getAge(child.getBirthDate()) < 18)
                .collect(Collectors.toList());
    }

}
