package ch.sahits.game.openpatrician.engine.event;

import ch.sahits.game.openpatrician.model.IPlayer;
import ch.sahits.game.openpatrician.model.city.ICity;
import ch.sahits.game.openpatrician.utilities.annotation.ClassCategory;
import ch.sahits.game.openpatrician.utilities.annotation.EClassCategory;
import ch.sahits.game.openpatrician.utilities.annotation.LazySingleton;
import ch.sahits.game.openpatrician.utilities.annotation.MapType;

import java.time.LocalDateTime;
import java.util.HashMap;
import java.util.Map;
import java.util.Optional;

/**
 * State for the event engine holing serializable information used by the EventEngine.
 * Created by andi on 29.03.17.
 */
@LazySingleton
@ClassCategory({EClassCategory.SINGLETON_BEAN, EClassCategory.SERIALIZABLE_BEAN})
public class EventEngineState {
    @MapType(key = ICity.class, value = LocalDateTime.class)
    private Map<ICity, LocalDateTime> lastFire = new HashMap<>();
    @MapType(key = ICity.class, value = LocalDateTime.class)
    private Map<ICity, LocalDateTime> lastPlague = new HashMap<>();
    @MapType(key = String.class, value = EEventState.class)
    private Map<String,EEventState> marriageState = new HashMap<>();

    /**
     * Add or override the date of the last fire in the city
     * @param city where the fire happened
     * @param date when the fire happened
     */
    public void addFire(ICity city, LocalDateTime date) {
        lastFire.put(city, date);
    }

    /**
     * Retrieve the Date of the last fire. There might not have been a fire yet.
     * @param city which should be checked
     * @return date of the last fire or empty
     */
    public Optional<LocalDateTime> getLastFire(ICity city) {
        return Optional.ofNullable(lastFire.get(city));
    }

    /**
     * Add or override the date of the last plague in the city
     * @param city where the plague happened
     * @param date when the plague happened
     */
    public void addPlague(ICity city, LocalDateTime date) {
        lastPlague.put(city, date);
    }

    /**
     * Retrieve the Date of the last plague. There might not have been a plague yet.
     * @param city which should be checked
     * @return date of the last plague or empty
     */
    public Optional<LocalDateTime> getLastPlague(ICity city) {
        return Optional.ofNullable(lastPlague.get(city));
    }

    /**
     * Set or update the marriage state of a player.
     * @param player to be updated
     * @param state marriage state
     */
    public void setMarriageState(IPlayer player, EEventState state) {
        marriageState.put(player.getUuid(), state);
    }

    /**
     * Retrieve the current marriage state of the player.
     * @param player to be checked.
     * @return marriage state or empty
     */
    public Optional<EEventState> getMarriageState(IPlayer player) {
        return Optional.ofNullable(marriageState.get(player.getUuid()));
    }

    /**
     * Remove the player form the marriage state map.
     * @param player that gets married.
     */
    public void resetMarriedState(IPlayer player) {
        marriageState.remove(player.getUuid());
    }
}
