package ch.sahits.game.openpatrician.engine.event.task;

import ch.sahits.game.openpatrician.clientserverinterface.client.ICityPlayerProxyJFX;
import ch.sahits.game.openpatrician.clientserverinterface.service.ShipService;
import ch.sahits.game.openpatrician.model.DisplayMessage;
import ch.sahits.game.openpatrician.model.IHumanPlayer;
import ch.sahits.game.openpatrician.model.city.IShipyard;
import ch.sahits.game.openpatrician.model.event.TargetedEvent;
import ch.sahits.game.openpatrician.model.event.TimedTask;
import ch.sahits.game.openpatrician.model.ship.EShipUpgrade;
import ch.sahits.game.openpatrician.model.ship.IShip;
import ch.sahits.game.openpatrician.model.ship.IWeaponSlot;
import ch.sahits.game.openpatrician.model.ship.SecondaryLargeWeaponSlot;
import ch.sahits.game.openpatrician.model.ship.impl.ShipWeaponsLocationFactory;
import ch.sahits.game.openpatrician.model.weapon.EWeapon;
import ch.sahits.game.openpatrician.utilities.annotation.ClassCategory;
import ch.sahits.game.openpatrician.utilities.annotation.EClassCategory;
import ch.sahits.game.openpatrician.utilities.annotation.Prototype;
import com.google.common.eventbus.AsyncEventBus;
import com.thoughtworks.xstream.annotations.XStreamOmitField;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;

import java.time.LocalDateTime;
import java.util.List;

/**
 * @author Andi Hotz, (c) Sahits GmbH, 2014
 *         Created on Feb 23, 2014
 */
@Prototype
@ClassCategory({EClassCategory.SERIALIZABLE_BEAN, EClassCategory.PROTOTYPE_BEAN})
class RefitShipTask extends TimedTask {

    private final IShip ship;
    private final EShipUpgrade level;
    protected final ICityPlayerProxyJFX city;
    private final IShipyard shipyard;
    @Autowired
    @XStreamOmitField
    private ShipWeaponsLocationFactory shipWeaponLocationFactory;
    @Autowired
    @XStreamOmitField
    @Qualifier("serverClientEventBus")
    private AsyncEventBus clientServerEventBus;
    @Autowired
    @XStreamOmitField
    private ShipService shipService;

    public RefitShipTask(LocalDateTime executionTime, IShip ship, EShipUpgrade upgrade, ICityPlayerProxyJFX city, IShipyard shipyard) {
        super();
        setExecutionTime(executionTime);
        this.ship = ship;
        level = upgrade;
        this.city = city;
        this.shipyard = shipyard;
    }

    @Override
    public void run() {
        EShipUpgrade currentLevel = ship.getShipUpgradeLevel();
        while (currentLevel != level) {
            ship.upgrade();
            List<IWeaponSlot> oldWeaponSlots = ship.getWeaponSlots();
            List<IWeaponSlot> newWeaponSlots = shipWeaponLocationFactory.getShipWeaponsLocation(ship.getShipType(), level);
            ship.setWeaponSlots(newWeaponSlots);
            for (IWeaponSlot slot : oldWeaponSlots) {
                if (slot.getWeapon().isPresent()) {
                    EWeapon weapon = (EWeapon) slot.getWeapon().get();
                    if (slot instanceof SecondaryLargeWeaponSlot) {
                        if (!shipService.isLargeWeapon(weapon)) { // ignore large weapons in secondary slots
                            shipService.placeWeapon(weapon, ship);
                        }
                    } else {
                        // Not secondary slot
                        shipService.placeWeapon(weapon, ship);
                    }
                }
            }
            currentLevel = ship.getShipUpgradeLevel();
        }
        ship.setAvailable(true);
        city.getPlayersShips().add(ship);
        shipyard.removeCompletedUpgrade(ship);
        if (city.getPlayersShips().size() == 1) {
            city.activateShip(ship);
        }
        int currentCapacity = ship.getCapacity();

        if (ship.getOwner() instanceof IHumanPlayer) {
            DisplayMessage msg = new DisplayMessage("ch.sahits.game.openpatrician.engine.event.task.RefitShipTask.message", new Object[]{ship.getShipType(), level, currentCapacity});
            TargetedEvent displayMessage = new TargetedEvent((IHumanPlayer) ship.getOwner(), msg);
            clientServerEventBus.post(displayMessage);
        }
    }
}
