package ch.sahits.game.openpatrician.engine.event.task;

import ch.sahits.game.openpatrician.clientserverinterface.model.factory.PlayerInteractionFactory;
import ch.sahits.game.openpatrician.clientserverinterface.service.CityProductionAndConsumptionService;
import ch.sahits.game.openpatrician.model.Date;
import ch.sahits.game.openpatrician.model.IBalanceSheet;
import ch.sahits.game.openpatrician.model.building.ITradingOffice;
import ch.sahits.game.openpatrician.model.city.ICity;
import ch.sahits.game.openpatrician.model.city.impl.CitiesState;
import ch.sahits.game.openpatrician.model.city.impl.CityState;
import ch.sahits.game.openpatrician.model.event.TimedTask;
import ch.sahits.game.openpatrician.model.event.TimedUpdatableTaskList;
import ch.sahits.game.openpatrician.model.impl.BalanceSheet;
import ch.sahits.game.openpatrician.utilities.annotation.ClassCategory;
import ch.sahits.game.openpatrician.utilities.annotation.EClassCategory;
import ch.sahits.game.openpatrician.utilities.annotation.Prototype;
import ch.sahits.game.openpatrician.utilities.spring.DependentAnnotationConfigApplicationContext;
import com.thoughtworks.xstream.annotations.XStreamOmitField;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.context.ApplicationContext;

import javax.annotation.PostConstruct;
import java.time.LocalDateTime;
import java.util.List;
import java.util.Optional;

/**
 * Timed task to for weekly checks.
 * Created by andi on 20.03.17.
 */
@Prototype
@ClassCategory({EClassCategory.SERIALIZABLE_BEAN, EClassCategory.PROTOTYPE_BEAN})
public class WeeklyCityCheckTask extends TimedTask {
    @Autowired
    private Date date;
    @Autowired
    private TimedUpdatableTaskList taskList;
    @Autowired
    @XStreamOmitField
    private ServerSideTaskFactory taskFactory;
    @Autowired
    private CitiesState citiesState;

    @Autowired
    @XStreamOmitField
    private CityProductionAndConsumptionService productionAndConsumtionService;
    @Autowired
    @XStreamOmitField
    private PlayerInteractionFactory interactionFactory;
    @Autowired
    @XStreamOmitField
    private ApplicationContext context;

    @PostConstruct
    private void initialize() {
        setExecutionTime(getScheduledDateWeeklyUpdate());
    }

    private LocalDateTime getScheduledDateWeeklyUpdate() {
        return date.getCurrentDate().plusWeeks(1);
    }

    @Override
    public void run() {
        handleWeeklyUpdates();
    }

    private void handleWeeklyUpdates() {
        try {
            for (CityState cityState : citiesState.getCityEngineStates()) {
                productionAndConsumtionService.produceWares(cityState);
                ICity city = cityState.getCity();
                List<ITradingOffice> offices = city.findBuilding(ITradingOffice.class, Optional.empty());
                for (ITradingOffice office : offices) {
                    IBalanceSheet discardableSheet = office.getCurrentWeek();
                    IBalanceSheet newWeek = interactionFactory.createBalanceSheetFromPreviousWeek((BalanceSheet) discardableSheet);
                    office.replaceBalanceSheet(newWeek);
                    ((DependentAnnotationConfigApplicationContext)context).removePrototypeBean(discardableSheet);
                }
            }
        } finally {
            scheduleNextWeekUpdate();
        }
    }

    private void scheduleNextWeekUpdate() {
        taskList.add(taskFactory.getWeeklyCityCheck());
    }
}
