package ch.sahits.game.openpatrician.engine.player;

import ch.sahits.game.openpatrician.clientserverinterface.service.TradeService;
import ch.sahits.game.openpatrician.engine.player.strategy.CentralStorageStrategyHint;
import ch.sahits.game.openpatrician.model.IAIPlayer;
import ch.sahits.game.openpatrician.model.building.ETradeType;
import ch.sahits.game.openpatrician.model.building.IAutomatedTrading;
import ch.sahits.game.openpatrician.model.building.ITradingOffice;
import ch.sahits.game.openpatrician.model.building.IWorkShop;
import ch.sahits.game.openpatrician.model.building.impl.Steward;
import ch.sahits.game.openpatrician.model.city.EPopulationClass;
import ch.sahits.game.openpatrician.model.city.ICity;
import ch.sahits.game.openpatrician.model.city.PopulationConsume;
import ch.sahits.game.openpatrician.model.product.EWare;
import ch.sahits.game.openpatrician.model.product.IWare;
import ch.sahits.game.openpatrician.model.product.ProductionChain;
import ch.sahits.game.openpatrician.utilities.annotation.ClassCategory;
import ch.sahits.game.openpatrician.utilities.annotation.EClassCategory;
import ch.sahits.game.openpatrician.utilities.annotation.LazySingleton;
import com.google.common.annotations.VisibleForTesting;
import com.google.common.base.Preconditions;
import com.google.common.collect.ArrayListMultimap;
import com.google.common.collect.Multimap;
import org.springframework.beans.factory.annotation.Autowired;

import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.function.Predicate;
import java.util.stream.Collectors;

/**
 * Service to handle the setup of autmated trading for AI players.
 * @author Andi Hotz, (c) Sahits GmbH, 2017
 * Created on Oct 05, 2017
 */
@LazySingleton
@ClassCategory(EClassCategory.SINGLETON_BEAN)
public class TradeManagerService {

    @Autowired
    private ProductionChain productionChain;
    @Autowired
    private PopulationConsume consume;
    @Autowired
    private TradeService tradeService;
    /**
     * Hire a steward for the trading office.
     * @param tradingOffice for which the tradeing manager is to be hired
     */
    public void hireSteward(ITradingOffice tradingOffice) {
        Preconditions.checkArgument(!tradingOffice.getSteward().isPresent(), "No steward has to be present");
          tradingOffice.setSteward(new Steward());
    }

    /**
     * Ensure that the automated trading values are up to date.
      * @param office for which automatic trading is to be setup.
     */
    public void setupOrUpdateAutomaticTrading(ITradingOffice office) {
        Preconditions.checkArgument(office.getSteward().isPresent(), "Steward has to be present");
        IAutomatedTrading autoTrading = office.getOfficeTrading();
        IAIPlayer player = (IAIPlayer) office.getOwner();
        ICity city = office.getCity();
        List<IWorkShop> workShops = player.findBuildings(city, IWorkShop.class);

        // collect trade reasons
        Multimap<IWare, ETradeReason> reasons = collectTradeReasons(workShops, city);
        // compact to decide when to buy/sell
        Map<IWare, ETradeReason> definiteReasons = condense(reasons);
        // calculate amount and price
        List<Object[]> dertmineLimits = determineLimits(definiteReasons, city, player, workShops);
        for (EWare ware : EWare.values()) {
            autoTrading.setTradingType(ware, ETradeType.NONE);
        }

        for (Object[] entry : dertmineLimits) {
            IWare ware = (IWare) entry[0];
            if (definiteReasons.get(ware).isBuyReason()) {
                autoTrading.setTradingType(ware, ETradeType.CITY_OFFICE);
                if (definiteReasons.get(ware) == ETradeReason.PRODUCED_WARE_IN_CITY) {
                    autoTrading.setMovableToShip(ware, true);
                } else {
                    // buy ware for production
                    autoTrading.setMovableToShip(ware, false);
                }
            } else {
                autoTrading.setTradingType(ware, ETradeType.OFFICE_CITY);
                autoTrading.setMovableToShip(ware, true);

            }
            autoTrading.amountProperty(ware).setValue((int)entry[1]);
            autoTrading.priceProperty(ware).setValue((int)entry[2]);
        }

    }

    private List<Object[]> determineLimits(Map<IWare, ETradeReason> definiteReasons, ICity city, IAIPlayer player, List<IWorkShop> workShops) {
        List<Object[]> list = new ArrayList<>();
        for (IWare ware : definiteReasons.keySet()) {
            Object[] entry = new Object[3];
            entry[0] = ware;
            ETradeReason tradeReason = definiteReasons.get(ware);
            if (tradeReason.isBuyReason()) {
                entry[1] = calculateBuyAmount(ware, city, player, tradeReason, workShops);
                entry[2] = ware.getMaxBuyPriceModerate();
            } else {
                entry[1] = calculateSellAmount(ware, city, player, tradeReason, workShops);
                entry[2] = ware.getMinSellPriceModerate();
            }
            list.add(entry);
        }
        return list;
    }

    /**
     * Calculate the amount of wares that should be retained, when selling.
     */
    @VisibleForTesting
    int calculateSellAmount(IWare ware, ICity city, IAIPlayer player, ETradeReason tradeReason, List<IWorkShop> workShops) {
        switch (tradeReason) {
            case PRODUCED_WARE_IN_WORKSHOP:
                if (hasCentralStorageStrategy(city, player)) {
                    if (tradeService.getBasicNeeds().contains(ware)) {
                        return calculateCityNeeds(ware, city);
                    } else {
                        return (int)(workShops.stream()
                                .filter(workshop -> productionChain.getRequiredAmount(workshop.getProducableWare(), ware) > 0)
                                .mapToInt(workshop -> (int)Math.ceil(productionChain.getRequiredAmount(workshop.getProducableWare(), ware)))
                                .sum() * 0.5);
                    }
                } else {
                    return 0;
                }
            case REQUIRE_FOR_BASIC_NEEDS:
                if (hasCentralStorageStrategy(city, player)) {
                    return (int)(calculateCityNeeds(ware, city) * 0.5);
                } else {
                    return 0;
                }
            case REQUIRE_FOR_CITY_PRODUCTION:
                return workShops.stream()
                        .filter(workshop -> productionChain.getRequiredAmount(workshop.getProducableWare(), ware) > 0)
                        .mapToInt(workshop -> (int)Math.ceil(productionChain.getRequiredAmount(workshop.getProducableWare(), ware)))
                        .sum();
            default:
                throw new IllegalStateException("Unhandled sell trade reason: "+tradeReason);
        }
    }

    private int calculateCityNeeds(IWare ware, ICity city) {
        int sum = 0;
        for (EPopulationClass popClass : EPopulationClass.values()) {
            sum += consume.getNeed(ware, popClass, city.getPopulation(popClass));
        }
        return sum;
    }

    @VisibleForTesting
    int calculateBuyAmount(IWare ware, ICity city, IAIPlayer player, ETradeReason tradeReason, List<IWorkShop> workShops) {
        switch (tradeReason) {
            case REQUIRE_FOR_PRODUCTION:
                // one week supply
                return workShops.stream()
                        .filter(workshop -> productionChain.getRequiredWares(workshop.getProducableWare()).contains(ware))
                        .mapToInt(workshop -> (int)Math.ceil(productionChain.getRequiredAmount(workshop.getProducableWare(), ware)))
                        .sum();
            case PRODUCED_WARE_IN_CITY:
                // if needed in city or city is central storage, buy higher amount
                int baseAmount;
                if (ware.isBarrelSizedWare()) {
                    baseAmount = 20;
                } else {
                    baseAmount = 5;
                }
                if (tradeService.getBasicNeeds().contains(ware) || hasCentralStorageStrategy(city, player)) {
                    baseAmount *= 2;
                }
                return baseAmount;
            default:
                throw new IllegalStateException("Unhandled buy trade reason: "+tradeReason);
        }
    }

    private boolean hasCentralStorageStrategy(ICity city, IAIPlayer player) {
        return player.getPlayerContext().getHints(city).stream()
                .filter(hint -> hint instanceof CentralStorageStrategyHint).findFirst().isPresent();
    }

    @VisibleForTesting
    Map<IWare,ETradeReason> condense(Multimap<IWare, ETradeReason> reasons) {
        Map<IWare, ETradeReason> condensation = new HashMap<>();
        for (IWare ware : reasons.keySet()) {
            Collection<ETradeReason> reasonCol = reasons.get(ware);
            if (reasonCol.size() == 1) {
                condensation.put(ware, reasonCol.iterator().next());
            } else {
                ETradeReason mainReason = null;
                // buy
                List<ETradeReason> buyReasons = reasonCol.stream().filter(ETradeReason::isBuyReason)
                        .collect(Collectors.toList());
                // sort and pick first
                if (!buyReasons.isEmpty()) {
                    buyReasons.sort(ETradeReason.comparator());
                    mainReason = buyReasons.get(0);
                }

                // sell
                if (mainReason == null) { // only inspect sell reasons if there is no buy reason
                    List<ETradeReason> sellReasons = reasonCol.stream()
                            .filter(((Predicate<ETradeReason>)ETradeReason::isBuyReason).negate())
                            .collect(Collectors.toList());
                    // sort and pick first
                    if (!sellReasons.isEmpty()) {
                        sellReasons.sort(ETradeReason.comparator());
                        mainReason = sellReasons.get(0);
                    }

                }
                if (mainReason != null) {
                    condensation.put(ware, mainReason);
                }
            }
        }
        return condensation;
    }

    @VisibleForTesting
    Multimap<IWare, ETradeReason> collectTradeReasons(List<IWorkShop> workShops, ICity city) {
        Multimap<IWare, ETradeReason> map = ArrayListMultimap.create();
        for (IWorkShop workShop : workShops) {
            map.put(workShop.getProducableWare(), ETradeReason.PRODUCED_WARE_IN_WORKSHOP);
            List<IWare> requiredWares = productionChain.getRequiredWares(workShop.getProducableWare());
            requiredWares.stream().forEach(ware -> map.put(ware, ETradeReason.REQUIRE_FOR_PRODUCTION));
        }
        for (IWare ware : city.getEffectiveProduction()) {
            map.put(ware, ETradeReason.PRODUCED_WARE_IN_CITY);
            List<IWare> requiredWares = productionChain.getRequiredWares(ware);
            requiredWares.stream().forEach(reqWare -> map.put(reqWare, ETradeReason.REQUIRE_FOR_CITY_PRODUCTION));
        }
        for (IWare ware : city.getIneffectiveProduction()) {
            map.put(ware, ETradeReason.PRODUCED_WARE_IN_CITY);
            List<IWare> requiredWares = productionChain.getRequiredWares(ware);
            requiredWares.stream().forEach(reqWare -> map.put(reqWare, ETradeReason.REQUIRE_FOR_CITY_PRODUCTION));
        }
        tradeService.getBasicNeeds().stream().forEach(ware -> map.put(ware, ETradeReason.REQUIRE_FOR_BASIC_NEEDS));
        return map;
    }

}
