package ch.sahits.game.openpatrician.engine.player.strategy;

import ch.sahits.game.openpatrician.clientserverinterface.service.ShipService;
import ch.sahits.game.openpatrician.model.IAIPlayer;
import ch.sahits.game.openpatrician.model.IHumanPlayer;
import ch.sahits.game.openpatrician.model.city.ICity;
import ch.sahits.game.openpatrician.model.player.BuyWeapons;
import ch.sahits.game.openpatrician.model.player.IAIBuyWeaponStrategy;
import ch.sahits.game.openpatrician.model.product.ComputablePriceV2;
import ch.sahits.game.openpatrician.model.ship.IShip;
import ch.sahits.game.openpatrician.model.ship.IWeaponSlot;
import ch.sahits.game.openpatrician.model.ship.PrimaryLargeWeaponSlot;
import ch.sahits.game.openpatrician.model.ship.SecondaryLargeWeaponSlot;
import ch.sahits.game.openpatrician.model.weapon.ArmoryRegistry;
import ch.sahits.game.openpatrician.model.weapon.EWeapon;
import ch.sahits.game.openpatrician.model.weapon.IArmory;
import ch.sahits.game.openpatrician.utilities.javafx.bindings.ConstantIntegerBinding;
import com.thoughtworks.xstream.annotations.XStreamOmitField;
import javafx.beans.property.ReadOnlyIntegerProperty;
import javafx.beans.property.SimpleIntegerProperty;
import org.springframework.beans.factory.annotation.Autowired;

import java.util.ArrayList;
import java.util.List;
import java.util.Random;

/**
 * Base class for buying weapons strategy implementing common services.
 * @author Andi Hotz, (c) Sahits GmbH, 2017
 * Created on Oct 18, 2017
 */
public abstract class BaseBuyWeaponStrategy implements IAIBuyWeaponStrategy {
    @Autowired
    private Random rnd;
    @Autowired
    @XStreamOmitField
    private ArmoryRegistry registry;
    @Autowired
    @XStreamOmitField
    private ComputablePriceV2 computablePrice;
    @Autowired
    @XStreamOmitField
    private ShipService shipService;
    /**
     * Retrieve the number of weapon slots on the ship.
     */
    protected int getNumberOfWeaponSlots(IShip ship) {
        return ship.getWeaponSlots().size();
    }

    /**
     * Count the empty weapon slots.
     */
    protected int getNumberOfEmptyWeaponSlots(IShip ship) {
        return (int) ship.getWeaponSlots()
                .stream()
                .filter(slot -> !slot.getWeapon().isPresent())
                .count();
    }

    /**
     * Count the empty large weapon slots, where a large weapon can be placed.
     * @param ship for which to retrieve the empty slots
     * @return number of empty slots for large weapons
     */
    protected int getNumberOfEmptyLargeWeaponSlots(IShip ship) {
        List<IWeaponSlot> weaponSlots = ship.getWeaponSlots();
        int countFreeLarge = 0;
        for (int i = 0; i < weaponSlots.size(); i++) {
            IWeaponSlot slot = weaponSlots.get(i);
            if (!slot.getWeapon().isPresent()) {
                if (slot instanceof PrimaryLargeWeaponSlot) {
                    IWeaponSlot next = weaponSlots.get(i + 1);
                    if (!next.getWeapon().isPresent() && next instanceof SecondaryLargeWeaponSlot) {
                        countFreeLarge++;
                    }
                }
            }
        }
        return countFreeLarge;
    }

    /**
     * Retrieve the amount of weapons available in the armory.
     */
    protected int getWeaponAmount(IArmory armory, EWeapon weapon) {
        switch (weapon) {
            case HAND_WEAPON:
                return armory.cutlassAmountProperty().get();
            case BALLISTA_BIG:
                return armory.ballistaBigAmountProperty().get();
            case BALLISTA_SMALL:
                return armory.ballistaSmallAmountProperty().get();
            case TREBUCHET_BIG:
                return armory.trebuchetBigAmountProperty().get();
            case TREBUCHET_SMALL:
                return armory.trebuchetSmallAmountProperty().get();
            case BOMBARD:
                return armory.bombardAmountProperty().get();
            case CANNON:
                return armory.canonAmountProperty().get();
            default:
                throw new IllegalArgumentException("Unhandled weapon type: "+weapon);
        }
    }

    /**
     * Randomly select a large weapon available in the armory and buy it. No cash transfer.
     * @param armory from which to buy weapons
     * @return bought weapon or null, if there is no weapon to be bought.
     */
    protected EWeapon buyLargeWeapon(IArmory armory, int capacity) {
        List<EWeapon> weapons = new ArrayList<>();
        if (getWeaponAmount(armory, EWeapon.BALLISTA_BIG) > 0 && capacity > EWeapon.BALLISTA_BIG.getSize()) {
            weapons.add(EWeapon.BALLISTA_BIG);
        }
        if (getWeaponAmount(armory, EWeapon.TREBUCHET_BIG) > 0 && capacity > EWeapon.TREBUCHET_BIG.getSize()) {
            weapons.add(EWeapon.TREBUCHET_BIG);
        }
        if (getWeaponAmount(armory, EWeapon.BOMBARD) > 0 && capacity > EWeapon.BOMBARD.getSize()) {
            weapons.add(EWeapon.BOMBARD);
        }
        EWeapon chosenWeapon;
        if (weapons.isEmpty()) {
            return null;
        }
        chosenWeapon = weapons.get(rnd.nextInt(weapons.size()));
        switch (chosenWeapon) {
            case TREBUCHET_BIG:
                armory.updateTrebuchetBigAmount(-1);
                break;
            case BALLISTA_BIG:
                armory.updateBallistaBigAmount(-1);
                break;
            case BOMBARD:
                armory.updateBombardAmount(-1);
                break;
            default:
                throw new IllegalStateException("Cannot buy big weapon of type "+chosenWeapon);
        }
        return chosenWeapon;
    }

    /**
     * Randomly select a small weapon available in the armory and buy it. No cash transfer.
     * @param armory from which to buy weapons
     * @return bought weapon or null, if there is no weapon to be bought.
     */
    protected EWeapon buySmallWeapon(IArmory armory, int capacity) {
        List<EWeapon> weapons = new ArrayList<>();
        if (getWeaponAmount(armory, EWeapon.BALLISTA_SMALL) > 0 && capacity > EWeapon.BALLISTA_SMALL.getSize()) {
            weapons.add(EWeapon.BALLISTA_SMALL);
        }
        if (getWeaponAmount(armory, EWeapon.TREBUCHET_SMALL) > 0 && capacity > EWeapon.TREBUCHET_SMALL.getSize()) {
            weapons.add(EWeapon.TREBUCHET_SMALL);
        }
        if (getWeaponAmount(armory, EWeapon.CANNON) > 0 && capacity > EWeapon.CANNON.getSize()) {
            weapons.add(EWeapon.CANNON);
        }
        EWeapon chosenWeapon;
        if (weapons.isEmpty()) {
            return null;
        }
        chosenWeapon = weapons.get(rnd.nextInt(weapons.size()));
        switch (chosenWeapon) {
            case TREBUCHET_SMALL:
                armory.updateTrebuchetSmallAmount(-1);
                break;
            case BALLISTA_SMALL:
                armory.updateBallistaSmallAmount(-1);
                break;
            case CANNON:
                armory.updateCanonAmount(-1);
                break;
            default:
                throw new IllegalStateException("Cannot buy small weapon of type "+chosenWeapon);
        }
        return chosenWeapon;
    }

    /**
     * First the hand weapons are bought. Then the large weapons are purchased one at a time to ensure, that the weapon
     * can be bought (there might not be enough available in the armory. The buying of the large weapons is limited by
     * the number that should be bought and the overall strength. As a last step the small weapons are bought in the
     * same manner with the same limitations. In the end the blacksmith is paid.
     * @param target defining the target that should be met by buying the weapons.
     * @param ship for which the weapons should be bought
     * @param player who is purchasing the weapons
     * @param city in which to buy weapons
     */
    @Override
    public void buyWeapons(BuyWeapons target, IShip ship, IAIPlayer player, ICity city) {
        IArmory armory = registry.getArmory(city);
        int costs;
        int availableHandWeapons = getWeaponAmount(armory, EWeapon.HAND_WEAPON);
        int buyHandWeapons = Math.min(availableHandWeapons, target.getHandWeapons());
        armory.updateCutlassAmount(-buyHandWeapons);
        int price = computablePrice.buyPrice(EWeapon.HAND_WEAPON, new SimpleIntegerProperty(availableHandWeapons), new ConstantIntegerBinding(buyHandWeapons));
        costs = price * buyHandWeapons;
        for (int i = 0; i < buyHandWeapons; i++) {
            shipService.placeWeapon(EWeapon.HAND_WEAPON, ship);
        }
        int boughtLargeWeapons = 0;
        while (boughtLargeWeapons < target.getLargeWeapons() && shipService.calculateShipsWeaponsStrength(ship) < target.getTotalStrength()) {
            EWeapon boughtWeapon = buyLargeWeapon(armory, ship.getCapacity());
            if (boughtWeapon != null) {
                ReadOnlyIntegerProperty available = new SimpleIntegerProperty(getWeaponAmount(armory, boughtWeapon));
                price = computablePrice.buyPrice(boughtWeapon, available, new ConstantIntegerBinding(1));
                costs += price;
                boughtLargeWeapons++;
                shipService.placeWeapon(boughtWeapon, ship);
            } else {
                break;
            }
        }
        int boughtSmallWeapons = 0;
        while (boughtSmallWeapons < target.getSmallWeapons() && shipService.calculateShipsWeaponsStrength(ship) < target.getTotalStrength()) {
            EWeapon boughtWeapon = buySmallWeapon(armory, ship.getCapacity());
            if (boughtWeapon != null) {
                ReadOnlyIntegerProperty available = new SimpleIntegerProperty(getWeaponAmount(armory, boughtWeapon));
                price = computablePrice.buyPrice(boughtWeapon, available, new ConstantIntegerBinding(1));
                costs += price;
                boughtSmallWeapons++;
                shipService.placeWeapon(boughtWeapon, ship);
            } else {
                break;
            }
        }
        if (player instanceof IHumanPlayer) {
            player.getCompany().updateCash(-costs);
        } else {
            player.getCompany().updateCashDirectly(-costs);
        }
    }
}
