package ch.sahits.game.openpatrician.engine.player.strategy;

import ch.sahits.game.openpatrician.clientserverinterface.model.factory.ShipFactory;
import ch.sahits.game.openpatrician.engine.player.CollectWaresMissionData;
import ch.sahits.game.openpatrician.model.IAIPlayer;
import ch.sahits.game.openpatrician.model.ICompany;
import ch.sahits.game.openpatrician.model.city.ICity;
import ch.sahits.game.openpatrician.model.city.IShipDueDate;
import ch.sahits.game.openpatrician.model.city.IShipyard;
import ch.sahits.game.openpatrician.model.city.Shipyards;
import ch.sahits.game.openpatrician.model.player.IAIConstructionSelectionStrategy;
import ch.sahits.game.openpatrician.model.product.EWare;
import ch.sahits.game.openpatrician.model.product.ISpecialMission;
import ch.sahits.game.openpatrician.model.ship.EShipType;
import ch.sahits.game.openpatrician.model.ship.INavigableVessel;
import ch.sahits.game.openpatrician.model.ship.IShip;
import lombok.AccessLevel;
import lombok.Getter;
import lombok.Setter;
import org.springframework.beans.factory.annotation.Autowired;

import java.util.List;
import java.util.Map.Entry;
import java.util.Optional;

/**
 * Base implementation for a ship construction selection strategy that can be easily configured through
 * the non default constructor.
 * @author Andi Hotz, (c) Sahits GmbH, 2016
 *         Created on Jul 30, 2016
 */
public abstract class BaseShipConstructionSelectionStrategy implements IAIConstructionSelectionStrategy {
    /** Maximum ships being ordered */
    @Setter(AccessLevel.PROTECTED)
    private int constructMaxShips = 1;
    @Autowired
    private ShipFactory shipFactory;
    @Autowired
    private Shipyards shipyards;
    /**
     * The minimum of cash that is required before considering building a ship.
     */
    @Getter(AccessLevel.PROTECTED)
    private final int minCash;
    /**
     * The maximal fleet size up to which ships are ordered.
     */
    @Getter(AccessLevel.PROTECTED)
    private final int maxFeetSize;
    /**
     * The macimal cargo capacity of the fleet above which no ships are ordered.
     */
    @Getter(AccessLevel.PROTECTED)
    private final int maxTotalCargoCapacity;

    protected BaseShipConstructionSelectionStrategy(int minCash, int maxFeetSize, int maxTotalCargoCapacity) {
        this.minCash = minCash;
        this.maxFeetSize = maxFeetSize;
        this.maxTotalCargoCapacity = maxTotalCargoCapacity;
    }

    /**
     * {@inheritDoc}
     * Ship is ordered if enough cash is available and the fleet size and fleet capacity is below certain thresholds, as
     * defined at construction time.
     * @param player for which the check should be done.
     * @return
     */
    @Override
    public boolean shouldOrderNewConstruction(IAIPlayer player) {
        ICompany company = player.getCompany();
        if (company.getCash() < minCash) {
            return false;
        }
        List<IShip> fleet = player.getFleet();
        if (fleet.size() >= maxFeetSize) {
            return false;
        } else {
            int totalSize = getFleetCapacity(fleet);
            if (totalSize >= maxTotalCargoCapacity) {
                return false;
            }
        }
        int nbConstuctionOrders = 0;
        for (Entry<ICity, IShipyard> entry : shipyards) {
            IShipyard shipyard = entry.getValue();
            List<IShipDueDate> orders = shipyard.getShipBuildingList();
            for (IShipDueDate order : orders) {
                if (order.getShip().getOwner().equals(player)) {
                    nbConstuctionOrders++;
                }
                if (nbConstuctionOrders >= constructMaxShips) {
                    return false;
                }
            }
        }

        return true;
    }

    /**
     * Calculate the fleets total capacity.
     * @param fleet list of ships
     * @return capacity of all ships in <code>fleet</code>
     */
    protected int getFleetCapacity(List<IShip> fleet) {
        int totalSize = 0;
        for (IShip ship : fleet) {
            totalSize += ship.getCapacity();
        }
        return totalSize;
    }

    @Override
    public Optional<INavigableVessel> selectCollectingVessel(IAIPlayer player, EShipType shipType) {
        int totalAmountInBarrel = 0;
        for (EWare ware : EWare.values()) {
            int amount = shipFactory.getConstructionAmount(shipType, ware);
            totalAmountInBarrel += amount * ware.getSizeAsBarrels();
        }
        for (INavigableVessel vessel : player.getSelectableVessels()) {
            if (vessel.getCapacity() > totalAmountInBarrel) {
                if (isVesselEligibleForCollecting(player, vessel)) {
                    return Optional.of(vessel);
                }
            }
        }
        if (isVesselEligibleForCollecting(player, player.getSelectableVessels().get(0))) {
            return Optional.of(player.getSelectableVessels().get(0));
        } else {
            return Optional.empty();
        }
    }

    private boolean isVesselEligibleForCollecting(IAIPlayer player, INavigableVessel vessel) {
        return !(player.getTradeMission(vessel) instanceof CollectWaresMissionData || player.getTradeMission(vessel) instanceof ISpecialMission);
    }

}
