package ch.sahits.game.openpatrician.engine.player.strategy;

import ch.sahits.game.openpatrician.clientserverinterface.service.MapProxy;
import ch.sahits.game.openpatrician.clientserverinterface.service.MapService;
import ch.sahits.game.openpatrician.clientserverinterface.service.ShipService;
import ch.sahits.game.openpatrician.engine.player.SupplyCityMissionData;
import ch.sahits.game.openpatrician.engine.player.tradesteps.TransferToOfficeTradeStep;
import ch.sahits.game.openpatrician.engine.player.tradesteps.TravelToTradeStep;
import ch.sahits.game.openpatrician.model.IAIPlayer;
import ch.sahits.game.openpatrician.model.city.ICity;
import ch.sahits.game.openpatrician.model.event.IShipEntersPortEvent;
import ch.sahits.game.openpatrician.model.player.ICityProductionConsumptionKnowledge;
import ch.sahits.game.openpatrician.model.player.IProductionConsumptionKnowledge;
import ch.sahits.game.openpatrician.model.player.ITradeStategyHint;
import ch.sahits.game.openpatrician.model.product.EWare;
import ch.sahits.game.openpatrician.model.product.IWare;
import ch.sahits.game.openpatrician.model.ship.INavigableVessel;
import ch.sahits.game.openpatrician.utilities.annotation.ClassCategory;
import ch.sahits.game.openpatrician.utilities.annotation.EClassCategory;
import ch.sahits.game.openpatrician.utilities.annotation.LazySingleton;
import javafx.util.Pair;
import org.springframework.beans.factory.annotation.Autowired;

import java.util.ArrayList;
import java.util.Collections;
import java.util.HashSet;
import java.util.List;
import java.util.Optional;
import java.util.Set;
import java.util.stream.Collectors;

/**
 * Strategy to supply a central station with required wares. Required are all wares that are needed bor the basic requirement.
 * This strategy is only chosable if there are multiple vessels available and there is a trading office in the target city.
 * @author Andi Hotz, (c) Sahits GmbH, 2017
 * Created on Oct 21, 2017
 */
@LazySingleton
@ClassCategory(EClassCategory.SINGLETON_BEAN)
public class SupplyCentralTradingStationAIStrategy extends BasePlayerTradeStrategy {
    @Autowired
    private MapProxy map;
    @Autowired
    private MapService mapService;
    @Autowired
    private ShipService shipService;

    public SupplyCentralTradingStationAIStrategy() {
        tradeStrategyType = EAITradeStrategyType.COLLECT_WARES_FOR_STORAGE_LOCATION;
    }

    @Override
    public void initializeTradeCycle(IAIPlayer player, INavigableVessel vessel) {
        Pair<ICity, CentralStorageStrategyHint> setup = setupContextHint(player, vessel);
        // find needs of all supplied cities and add in the basic needs at a low level
        IProductionConsumptionKnowledge knowledge = player.getProductionAndConsumptionKnowledge();
        CentralStorageHintContext context = (CentralStorageHintContext) setup.getValue().getContext();
        SupplyCityMissionData missionData = new SupplyCityMissionData();
        missionData.setRequiredWares(new ArrayList<>(context.getWares()));
        missionData.setTargetCity(setup.getKey());
        player.setTradeMission(vessel, missionData);
        // Follow the same setup as supply hometown, avoid pickup cities that are being suplied.
        Optional<ICity> locatedInCity = shipService.findCity(vessel);
        if (!locatedInCity.isPresent() || !locatedInCity.get().equals(setup.getKey())) {
            // not in target city
            if (!locatedInCity.isPresent()) { // at sea
                // travel to nearest city that can provide wares
                ICity city = findCitySupplingWares(player, vessel, context);
                TravelToTradeStep travelTo = createTravelToStep(vessel, city);
                append(player, vessel, travelTo);
            } else { // in a different city buy wares and travel to target city
                addDefaultTradeSteps(vessel, player, locatedInCity.get(), setup.getKey(), context.getWares(), context.getWares(), false);
            }
        } else { // in target city
            ICity city = findCitySupplingWares(player, vessel, context);
            // Buy wares that can be sold there
            List<Pair<IWare, Number>> neededWares = getMostNeededWares(knowledge.getKnowlege(city), vessel);
            Set<IWare> wares = new HashSet<>();
            neededWares.stream().forEach(pair -> wares.add(pair.getKey()));

            addDefaultTradeSteps(vessel, player, locatedInCity.get(), city, context.getWares(), wares, false);

        }
        // If in target city move to city that can supply
        // else gather wares until cargo full
        // try transferring from storage before buying
        // exclude cities that are supplied
        // use cities closest to target
    }

    private ICity findCitySupplingWares(IAIPlayer player, INavigableVessel vessel, CentralStorageHintContext context) {
        HashSet<ICity> excludeCity = new HashSet<>(context.getSuppliedCities());
        Optional<ICity> optCity = findCityWithAllRequiredWares(new ArrayList<>(context.getWares()), player, vessel, excludeCity);
        ICity city;
        city = optCity.orElseGet(() -> findCitySupplyingWare(player, context.getWares().iterator().next(), vessel, excludeCity));
        return city;
    }

    Pair<ICity, CentralStorageStrategyHint> setupContextHint(IAIPlayer player, INavigableVessel vessel) {
        // chose city for supply depot
        List<ICity> citiesWithTradingOffice = map.getAllReachableCities(vessel).stream()
                .filter(city -> player.findTradingOffice(city).isPresent())
                .collect(Collectors.toList());
        Collections.shuffle(citiesWithTradingOffice);
        // This city should not yet be supplied
        ICity targetCity = null;
        // setup or update the context
        CentralStorageStrategyHint hint = null;
        for (ICity city : citiesWithTradingOffice) {
            if (targetCity != null) {
                break;
            }
            List<ITradeStategyHint> hints = player.getPlayerContext().getHints(city);
            for (ITradeStategyHint h : hints) {
                if (targetCity != null) {
                    break;
                }
                if (h instanceof CentralStorageStrategyHint) {
                    hint = (CentralStorageStrategyHint) h;
                    CentralStorageHintContext hintContext = (CentralStorageHintContext) h.getContext();
                    if (!hintContext.getSupplyingVessels().isEmpty()) {
                        break; // next city
                    }
                    // update
                    hintContext.addSupplyingVessel(vessel);
                    targetCity = city;
                    break;
                }
            }
            if (targetCity != null) {  // if the context was only updated
                break;
            }
            // no matching hint create new one
            hint = new CentralStorageStrategyHint();
            CentralStorageHintContext context = new CentralStorageHintContext();
            context.addSupplyingVessel(vessel);
            hint.setContext(context);
            player.getPlayerContext().add(city, hint);
            targetCity = city;
            // find up to 3 cities in the vincinity of city that are supplied.
            List<ICity> suppliedCities = new ArrayList<>();
            suppliedCities.add(city);
            ICity nextCity;
            List<ICity> excludeCities = map.getAllUnreachableCities(vessel);
            excludeCities.add(city);
            for (int i = 0; i < 3; i++) {
                nextCity = mapService.findNearbyCityRepeated(city, 1, 500, excludeCities);
                if (nextCity != null) {
                    suppliedCities.add(nextCity);
                    excludeCities.add(nextCity);
                } else {
                    break;
                }
            }
            suppliedCities.stream().forEach(context::addSuppliedCity);
            // List of wares required in the supplied cities, that is not produced in any of those cities.
            // determine wares
            List<IWare> consumed = new ArrayList<>();
            List<IWare> produced = new ArrayList<>();
            IProductionConsumptionKnowledge knowledge = player.getProductionAndConsumptionKnowledge();
            for (ICity suppliedCity : suppliedCities) {
                ICityProductionConsumptionKnowledge cityKnowledge = knowledge.getKnowlege(suppliedCity);
                for (IWare ware : EWare.values()) {
                    int producedAmount = cityKnowledge.getProductionAmount(ware);
                    int consumedAmount = cityKnowledge.getConsumptionAmount(ware);
                    if (producedAmount > 0 && !produced.contains(ware)) {
                        produced.add(ware);
                    }
                    if (consumedAmount > 0 && !consumed.contains(ware)) {
                        consumed.add(ware);
                    }
                }
            }
            List<IWare> requiredWares = new ArrayList<>(consumed);
            requiredWares.removeAll(produced);
            requiredWares.stream().forEach(context::addWare);
            player.getPlayerContext().add(city, hint);
        }
        return new Pair<>(targetCity, hint);
    }

    @Override
    public void handleShipArrivesInPort(IShipEntersPortEvent event) {
        INavigableVessel vessel = event.getShip();
        if (isMatchingTradeStrategy(vessel)) {
            IAIPlayer player = (IAIPlayer) vessel.getOwner();
            ICity city = event.getCity();
            SupplyCityMissionData missionData = (SupplyCityMissionData) player.getTradeMission(vessel);
            ICity targetCity = missionData.getTargetCity();
            CentralStorageStrategyHint hint = (CentralStorageStrategyHint) player.getPlayerContext().getHints(targetCity).stream().filter(h -> h instanceof CentralStorageStrategyHint).findFirst().get();
            CentralStorageHintContext context = (CentralStorageHintContext) hint.getContext();
            if (city.equals(targetCity)) {

                // - unload oal to storage
                TransferToOfficeTradeStep transferToOfficeStep = createTransferToOfficeTradeStep(vessel, city);
                append(player, vessel, transferToOfficeStep);
                // - find best city to supply wares
                ICity nextStop = findProvidingWares(city, context.getWares(), context.getSuppliedCities(), player, vessel);
                // buy wares needed there
                ICityProductionConsumptionKnowledge cityKnowledge = player.getProductionAndConsumptionKnowledge().getKnowlege(nextStop);
                List<Pair<IWare, Number>> needs = getMostNeededWares(cityKnowledge, vessel);
                HashSet<IWare> buyWares = new HashSet<>();
                needs.stream().forEach(pair -> buyWares.add(pair.getKey()));
                buyWares.removeAll(context.getWares());
                addDefaultTradeSteps(vessel, player, city, nextStop, new HashSet<>(context.getWares()), buyWares, false);
            } else {
                addDefaultTradeSteps(vessel, player, city, targetCity, new HashSet<>(context.getWares()), new HashSet<>(context.getWares()), false);
            }
            executeTradeSteps(player, vessel);
        }
    }


    @Override
    public boolean isSelectable(IAIPlayer player, INavigableVessel vessel) {
        // There are at least 4 ships
        if (player.getFleet().size() < 4) {
            return false;
        }
        List<ICity> citiesWithTradingOffice = map.getAllReachableCities(vessel).stream()
                .filter(city -> player.findTradingOffice(city).isPresent())
                .collect(Collectors.toList());
        // This city should not yet be supplied
        for (ICity city : citiesWithTradingOffice) {
            List<ITradeStategyHint> hints = player.getPlayerContext().getHints(city);
            for (ITradeStategyHint hint : hints) {
                if (hint instanceof CentralStorageStrategyHint) {
                    CentralStorageHintContext hintContext = (CentralStorageHintContext) hint.getContext();
                    return hintContext.getSupplyingVessels().isEmpty();
                }
            }
            return true;
        }
        return false;
    }
}
