package ch.sahits.game.openpatrician.engine.sea;

import ch.sahits.game.openpatrician.event.EGameStatusChange;
import ch.sahits.game.openpatrician.event.GameStateChange;
import ch.sahits.game.openpatrician.event.data.HeuristicGraphInitialisationComplete;
import ch.sahits.game.openpatrician.model.city.ICity;
import ch.sahits.game.openpatrician.utilities.annotation.ClassCategory;
import ch.sahits.game.openpatrician.utilities.annotation.EClassCategory;
import ch.sahits.game.openpatrician.utilities.annotation.MapType;
import com.carrotsearch.hppc.ObjectDoubleMap;
import com.carrotsearch.hppc.ObjectDoubleScatterMap;
import com.google.common.annotations.VisibleForTesting;
import com.google.common.base.Preconditions;
import com.google.common.eventbus.AsyncEventBus;
import com.google.common.eventbus.Subscribe;
import javafx.geometry.Point2D;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.context.annotation.Lazy;
import org.springframework.stereotype.Component;

import javax.annotation.PostConstruct;
import java.io.IOException;
import java.util.List;
import java.util.Map;
import java.util.Map.Entry;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ExecutorService;

/**
 * This singleton provides the heuristic for a specific map for the A* algorithm.
 * Initially the heuristic from all nodes to all cities is calculated, based on:
 * <ul>
 *     <li>The Eucledian distance</li>
 * </ul>
 * The heuristic can also be queried if a source has it's heuristic calculated and calculate
 * the heuristic values for that node.
 *
 * The heuristic may not be admissable, meaning that the value given in the heuristic for the
 * distance between A and B may be larger than the actual distance, however the result should still
 * be good enough.
 *
 * @author Andi Hotz, (c) Sahits GmbH, 2015
 *         Created on Dec 31, 2015
 */
@ClassCategory(EClassCategory.SINGLETON_BEAN)
@Component
@Lazy
public class AStarHeuristicProvider extends BaseGraphCalulationService {
    private final Logger logger = LogManager.getLogger(getClass());
    @Autowired
    @Qualifier("serverThreadPool")
    private ExecutorService serverThreadPool;
    @Autowired
    @Qualifier("serverClientEventBus")
    private AsyncEventBus clientServerEventBus;
    private final Object lock = new Object();
    /**
     * A map of heuristic from a node to each other node in the graph.
     * Key of the map is the source and the value is a map of the various destinations and their
     * weight from the source (outer key) to the destination (inner key)
     */
    @MapType(key = Point2D.class, value = Map.class)
    private final Map<Point2D, ObjectDoubleMap<Point2D>> heuristicMap = new ConcurrentHashMap<>();
    private int width;
    private int height;


    private boolean initializing = false;


    @PostConstruct
    private void initialize() {
        initializing = true;
        try {
            // as well, which means it needs to be reset first
            initImage();
            serverThreadPool.execute(this::createGraph);
        } catch (IOException e) {
            logger.error("Failed to create black and white image from the map", e);
            initializing = false;
        }
        clientServerEventBus.register(this);
    }
    @Subscribe
    public void registerGameLoad(GameStateChange gameStateChange) {
       if (!initializing && gameStateChange.getStatusChange() == EGameStatusChange.GAME_LOADED) {
           logger.info("Initialize heuristic after game load.");
           initialize();
       }
    }
    @VisibleForTesting
    void initImage() throws IOException {
        width = imageService.getWidth();
        height = imageService.getHeight();
    }

    @VisibleForTesting
    void createGraph() {
        long timestamp = System.currentTimeMillis();
        synchronized (lock) {
            List<ICity> cities = map.getCities();
            for (ICity city : cities) {
                ObjectDoubleMap<Point2D> hueristicForTarget = calculateHeuristicForSource(city.getCoordinates(), cities);
                heuristicMap.put(city.getCoordinates(), hueristicForTarget);
            }
            for (int x = 0; x < width; x += CHECK_DISTANCE) { // TODO: andi 1/3/16 split to different threads
                for (int y = 0; y < height; y += CHECK_DISTANCE) {
                    Point2D source = getPoint(x, y);
                    ObjectDoubleMap<Point2D> hueristicForTarget = calculateHeuristicForSource(source, cities);
                    heuristicMap.put(source, hueristicForTarget);
                }
            }

        }
        long took = System.currentTimeMillis() - timestamp;
        logger.debug("Created heuristic for "+heuristicMap.size()+" sources connecting to "+heuristicMap.values().iterator().next().size()+" nodes taking "+took+"ms");
        clientServerEventBus.post(new HeuristicGraphInitialisationComplete());
        initializing = false;
    }

    private ObjectDoubleMap<Point2D> calculateHeuristicForSource(Point2D source, List<ICity> cities) {
        ObjectDoubleMap<Point2D> distance = new ObjectDoubleScatterMap<>();
        for (ICity city : cities) {
            final Point2D coordinates = city.getCoordinates();
            double dist = calculateWeight(source, coordinates);
            distance.put(coordinates, dist);
        }

        return distance;
    }
    @VisibleForTesting
    protected double calculateWeight(Point2D from, Point2D to) {
        if (from.equals(to)) {
            return 0;
        }

        // Should calculate the amount of land between the two points

        return from.distance(to);
    }


    /**
     * Check if the heuristic for a target location is calculated.
     * @param source location
     * @return true if the heuristic is present.
     */
    public boolean heuristicForSourceAvailable(Point2D source) {
        return heuristicMap.containsKey(source);
    }

    /**
     * Retrieve the heuristic map. This method is blocking while the heuristic is
     * beeing calculated to ensure there are no out of date data contained.
     * @return Heuristic map for target locations to any location in the grid.
     */
    public Map<Point2D, ObjectDoubleMap<Point2D>> getHeuristic() {
        synchronized (lock) {
            return heuristicMap;
        }
    }

    /**
     * Add a new target node to the heuristic.
     * @param target location to be added.
     */
    public void addTargetNodeToHeuristic(Point2D target) {
        Preconditions.checkNotNull(target, "The target node may not be null");
         synchronized (lock) {
                 for (Entry<Point2D, ObjectDoubleMap<Point2D>> entry : heuristicMap.entrySet()) {
                     if (!entry.getValue().containsKey(target)) {
                         Point2D source = entry.getKey();
                         double dist = calculateWeight(source, target);
                         entry.getValue().put(target, dist);
                     }
                 }
         }
    }

    /**
     * Add a new target node to the heuristic.
     * @param source location to be added.
     */
    public void addSourceNodeToHeuristic(Point2D source) {
        Preconditions.checkNotNull(source, "The source node may not be null");
        if (!heuristicForSourceAvailable(source)) {
            synchronized (lock) {
                ObjectDoubleMap<Point2D> hueristicForTarget = calculateHeuristicForSource(source, map.getCities());
                 heuristicMap.put(source, hueristicForTarget);
            }
        } else {
            logger.debug("Heuristic for source point {} is available (point is contained as key)", source);
        }
    }
    /**
     * Find the closest point in the graph for a defined point.
     * @param point for which the closest defined point should be looked up.
     * @return closest point to <code>point</code>
     */
    public Point2D findClosest(Point2D point) {
        double distance = Double.MAX_VALUE;
        Point2D p = null;
        for (Point2D tempPoint  : getHeuristic().keySet()) {
            if (point.distance(tempPoint) < distance) {
                distance = point.distance(tempPoint);
                p = tempPoint;
            }
        }
        return p;
    }

}