package ch.sahits.game.openpatrician.engine.player;

import ch.sahits.game.openpatrician.clientserverinterface.service.PlayerProductionService;
import ch.sahits.game.openpatrician.engine.event.task.ServerSideTaskFactory;
import ch.sahits.game.openpatrician.model.Date;
import ch.sahits.game.openpatrician.model.IAIPlayer;
import ch.sahits.game.openpatrician.model.IPlayer;
import ch.sahits.game.openpatrician.model.PlayerList;
import ch.sahits.game.openpatrician.model.building.ITradingOffice;
import ch.sahits.game.openpatrician.model.event.TimedTask;
import ch.sahits.game.openpatrician.model.event.TimedUpdatableTaskList;
import ch.sahits.game.openpatrician.model.map.IMap;
import ch.sahits.game.openpatrician.utilities.annotation.ClassCategory;
import ch.sahits.game.openpatrician.utilities.annotation.EClassCategory;
import ch.sahits.game.openpatrician.utilities.annotation.Prototype;
import com.thoughtworks.xstream.annotations.XStreamOmitField;
import org.springframework.beans.factory.annotation.Autowired;

import javax.annotation.PostConstruct;
import java.time.LocalDateTime;
import java.util.List;
import java.util.Optional;

import static java.util.stream.Collectors.toList;

/**
 * Timed task for daily checks of player tasks.
 * @author Andi Hotz, (c) Sahits GmbH, 2017
 * Created on Oct 14, 2017
 */
@Prototype
@ClassCategory({EClassCategory.SERIALIZABLE_BEAN, EClassCategory.PROTOTYPE_BEAN})
public class DailyPlayerUpdater extends TimedTask {
    @Autowired
    private Date date;
    @Autowired
    private TimedUpdatableTaskList taskList;
    @Autowired
    private PlayerList players;
    @Autowired
    private IMap map;
    @Autowired
    @XStreamOmitField
    private ServerSideTaskFactory taskFactory;
    @Autowired
    @XStreamOmitField
    private TradeManagerService tradeManagerService;
    @Autowired
    @XStreamOmitField
    private PlayerProductionService playerProductionService;

    @PostConstruct
    private void initialize() {
        setExecutionTime(getScheduledDateDailyUpdate());
    }

    private LocalDateTime getScheduledDateDailyUpdate() {
        return date.getCurrentDate().plusDays(1);
    }

    @Override
    public void run() {
        handleDailyUpdates();
    }

    private void handleDailyUpdates() {
         try {
             for (IPlayer player : players) {
                 List<ITradingOffice> offices = findTradingOddices(player);
                 if (player instanceof IAIPlayer) {
                     handleTradingOfficeUpdates(offices);
                 }
                 handleWorkshopConsumptionAndProduction(offices);
             }
         } finally {
             scheduleNextDayUpdate();
         }
    }

    private void handleWorkshopConsumptionAndProduction(List<ITradingOffice> offices) {
        offices.parallelStream().forEach(office -> {
            playerProductionService.produceWares(office);
            playerProductionService.buyWareForProductionOneDay(office);
        });
    }

    private List<ITradingOffice> findTradingOddices(IPlayer player) {
        return map.getCities().stream()
                .map(player::findTradingOffice)
                .filter(Optional::isPresent)
                .map(Optional::get)
                .collect(toList());
    }

    private void handleTradingOfficeUpdates(List<ITradingOffice> offices) {
        offices.parallelStream().filter(office -> office.getSteward().isPresent()).forEach(office -> tradeManagerService.setupOrUpdateAutomaticTrading(office));
    }

    private void scheduleNextDayUpdate() {
        taskList.add(taskFactory.getDailyAIPlayerCheck());
    }
}
