package ch.sahits.game.openpatrician.engine.player.strategy;

import ch.sahits.game.openpatrician.model.IAIPlayer;
import ch.sahits.game.openpatrician.model.city.ICity;
import ch.sahits.game.openpatrician.model.ship.EShipType;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;

/**
 * Base strategy to build the bigest possible ships at a given location.
 * @author Andi Hotz, (c) Sahits GmbH, 2017
 * Created on Oct 27, 2017
 */
public abstract class BiggestShipConstructionSelectionStrategy extends BaseShipConstructionSelectionStrategy {
    public BiggestShipConstructionSelectionStrategy(int minCash, int maxFeetSize, int maxTotalCargoCapacity) {
        super(minCash, maxFeetSize, maxTotalCargoCapacity);
    }

    @Override
    public EShipType shouldBuildShipType(IAIPlayer player) {
        // Which ships can be built in hometown
        ICity where = getBuildLocation(player);
        List<EShipType> shipTypes = new ArrayList<>(Arrays.asList(where.getCityState().getShipyardState().getBuildableShipTypes()));
        int fleetCapacity = getFleetCapacity(player.getFleet());
        int maxTotalCargoCapacity = getMaxTotalCargoCapacity();
        double capacityOfMaxPercent = fleetCapacity *1.0 / maxTotalCargoCapacity;
        // if total capacity is below 70% of the max capacity => Holk
        if (capacityOfMaxPercent >= 0.7) {
            return biggestShipType(shipTypes);
        }
        // if total capacity is below 30% of the max capacity => Cog
        if (capacityOfMaxPercent <= 0.3) {
            shipTypes.remove(EShipType.HOLK);
        }
        // if total capacity is below 10% of the max capacity => Crayer
        if (capacityOfMaxPercent <= 0.1) {
            shipTypes.remove(EShipType.COG);
        }
        // if total capacity is below 5% of the max capacity => Snaikka
        if (capacityOfMaxPercent <= 0.05) {
            shipTypes.remove(EShipType.CRAYER);
        }
        return biggestShipType(shipTypes);
    }

    private EShipType biggestShipType(List<EShipType> shipTypes) {
        EShipType[] sizedTypes = {EShipType.HOLK, EShipType.COG, EShipType.CRAYER, EShipType.SNAIKKA};
        for (EShipType type : sizedTypes) {
            if (shipTypes.contains(type)) {
                return type;
            }
        }
        return null;
    }
}
