package ch.sahits.game.openpatrician.engine.player.strategy;

import ch.sahits.game.openpatrician.clientserverinterface.service.ShipService;
import ch.sahits.game.openpatrician.engine.sea.DangerService;
import ch.sahits.game.openpatrician.engine.sea.model.PirateActivity;
import ch.sahits.game.openpatrician.engine.sea.model.PirateActivityEntry;
import ch.sahits.game.openpatrician.model.IAIPlayer;
import ch.sahits.game.openpatrician.model.city.ICity;
import ch.sahits.game.openpatrician.model.player.BuyWeapons;
import ch.sahits.game.openpatrician.model.ship.IShip;
import ch.sahits.game.openpatrician.model.weapon.EWeapon;
import ch.sahits.game.openpatrician.utilities.annotation.ClassCategory;
import ch.sahits.game.openpatrician.utilities.annotation.EClassCategory;
import ch.sahits.game.openpatrician.utilities.annotation.LazySingleton;
import com.thoughtworks.xstream.annotations.XStreamOmitField;
import org.springframework.beans.factory.annotation.Autowired;

import java.util.List;
import java.util.Optional;

/**
 * Strategy to buy weapons when the successful pirate activity is high to fill half the ships slot with weapons.
 * @author Andi Hotz, (c) Sahits GmbH, 2017
 * Created on Oct 18, 2017
 */
@ClassCategory(EClassCategory.SINGLETON_BEAN)
@LazySingleton
public class BuyWeaponHighPirateSuccessHalfSlotsFilledStrategy extends BaseBuyWeaponStrategy {
    @Autowired
    private PirateActivity pirateActivity;
    @Autowired
    @XStreamOmitField
    private DangerService dangerService;
    @Autowired
    @XStreamOmitField
    private ShipService shipService;


    private final static long CASH_LIMIT = 10000L;

    /**
     * Weapons should be bought when:
     * <ul>
     *     <li>There was a successful pirate attack every 2 weeks</li>
     *     <li>There are free weapon slots</li>
     *     <li>at least 10'000 cash is available</li>
     * </ul>
     * Then the amounts are calculated:
     * <ul>
     *     <li>one hand weapon per sailor</li>
     *     <li>Half the amount of large weapons then there are slots</li>
     *     <li>Half the amount of remaining open slots for small weapons</li>
     *     <li>~1/3 times the amount of free weapon slots as strength</li>
     * </ul>
     */
    @Override
    public Optional<BuyWeapons> shouldBuyWeapons(IShip ship, IAIPlayer player, ICity city) {
        long cash = player.getCompany().getCash();
        int numberOfFreeSlots = getNumberOfEmptyWeaponSlots(ship); // also number of free small slots
        int numberOfFreeLargeSlots = getNumberOfEmptyLargeWeaponSlots(ship);
        List<PirateActivityEntry> attacks = pirateActivity.getPirateActivity();
        int numberOfAttacks = dangerService.getNumberOfSuccessfulPirateAttacks(attacks);
        int limitNumberOfFewAttacks = pirateActivity.getObservationPeriodInDays() / 14; // one attack every 2 weeks
        if (numberOfAttacks > limitNumberOfFewAttacks && numberOfFreeSlots > 0 && cash > CASH_LIMIT && ship.getCapacity() > 0) {
            int nbHandWeapons = Math.max(0, ship.getNumberOfSailors() - ship.getWeaponAmount(EWeapon.HAND_WEAPON));
            int largeWeapons = (int) Math.min(Math.round(numberOfFreeLargeSlots / 2.0), ship.getCapacity() / 2); // assume average size of large weapon to be 2
            int smallWeapons = Math.min(Math.max(0, numberOfFreeSlots - largeWeapons *2) / 2, ship.getCapacity());
            int existingStrength = shipService.calculateShipsWeaponsStrength(ship);
            int targetStrength = (int) (numberOfFreeSlots * 0.65) + existingStrength;
            if (largeWeapons == 0 && smallWeapons == 0) {
                targetStrength = existingStrength;
            }
            if (nbHandWeapons > 0 || largeWeapons > 0 || smallWeapons > 0) {
                return Optional.of(BuyWeapons.builder()
                        .handWeapons(nbHandWeapons)
                        .smallWeapons(smallWeapons)
                        .largeWeapons(largeWeapons)
                        .totalStrength(targetStrength)
                        .build());
            } else {
                return Optional.empty();
            }
        } else {
            return Optional.empty();
        }
    }


}
