package ch.sahits.game.openpatrician.engine.player.tradesteps;

import ch.sahits.game.openpatrician.clientserverinterface.service.ShipService;
import ch.sahits.game.openpatrician.event.data.ai.HireSailorEvent;
import ch.sahits.game.openpatrician.model.Date;
import ch.sahits.game.openpatrician.model.IAIPlayer;
import ch.sahits.game.openpatrician.model.city.ICity;
import ch.sahits.game.openpatrician.model.event.TimedTask;
import ch.sahits.game.openpatrician.model.event.TimedUpdatableTaskList;
import ch.sahits.game.openpatrician.model.people.ISailorState;
import ch.sahits.game.openpatrician.model.player.IAIHireSailorStrategy;
import ch.sahits.game.openpatrician.model.product.ITradeStep;
import ch.sahits.game.openpatrician.model.ship.IGroupableVessel;
import ch.sahits.game.openpatrician.model.ship.INavigableVessel;
import ch.sahits.game.openpatrician.model.ship.IShip;
import ch.sahits.game.openpatrician.utilities.annotation.ClassCategory;
import ch.sahits.game.openpatrician.utilities.annotation.EClassCategory;
import ch.sahits.game.openpatrician.utilities.annotation.Prototype;
import com.google.common.base.Preconditions;
import com.google.common.eventbus.AsyncEventBus;
import com.thoughtworks.xstream.annotations.XStreamOmitField;
import lombok.Setter;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;

import java.time.LocalDateTime;

/**
 * Trade step for hiring sailors on a ship.
 */
@ClassCategory({EClassCategory.SERIALIZABLE_BEAN, EClassCategory.PROTOTYPE_BEAN})
@Prototype
public class HireSailorsStep implements ITradeStep {
    @Setter
    private INavigableVessel ship;
    @Setter
    private ICity city;
    @Autowired
    private TimedUpdatableTaskList taskList;
    @Autowired
    private Date date;
    @Autowired
    @Qualifier("serverClientEventBus")
    @XStreamOmitField
    private AsyncEventBus clientServerEventBus;
    @Autowired
    @XStreamOmitField
    private ShipService shipService;

    @Override
    public boolean execute() {
        Preconditions.checkArgument(city.getCoordinates().equals(ship.getLocation()), "Ship "+ship.getName()+" is not in the city {} but at {}", city.getName(), ship.getLocation());
        ISailorState sailorState = city.getCityState().getTavernState().getSailors();
        int onShip = ship.getNumberOfSailors();
        int neededSailors = shipService.getMinNumberOfSailors(ship);
        if (ship instanceof IShip) {
            IAIPlayer player = (IAIPlayer) ship.getOwner();
            IAIHireSailorStrategy strategy = player.getHireSailorStrategyType().getStrategy();
            neededSailors = strategy.calculateRequiredSailorAmount((IShip) ship, player);
        }
        // Ignore the case of convoy
        if (onShip >= neededSailors) {
            return true;
        } else {
            int nbToHire = neededSailors - onShip;
            int inTavern = sailorState.getNumberOfSailors();
            if (inTavern >= nbToHire) {
                if (ship instanceof IShip) {
                    for (int i = 0; i < nbToHire; i++) {
                        sailorState.hire();
                    }
                    city.getCityState().getPopUpdateStatistic().hireSailors(nbToHire);
                    ((IShip)ship).setNumberOfSailors(onShip + nbToHire);
                    return true;
                } else {
                    IGroupableVessel group = (IGroupableVessel) ship;
                    for (IShip vessel : group.getShips()) {
                        onShip = vessel.getNumberOfSailors();
                        neededSailors = vessel.getMinNumberOfSailors();
                        nbToHire = neededSailors - onShip;
                        for (int i = 0; i < nbToHire; i++) {
                            sailorState.hire();
                        }
                        city.getCityState().getPopUpdateStatistic().hireSailors(nbToHire);
                        vessel.setNumberOfSailors(onShip + nbToHire);
                    }
                    return true;
                }
            } else {
                if (ship instanceof IShip) {
                    // hire max
                    for (int i = 0; i < inTavern; i++) {
                        sailorState.hire();
                    }
                    city.getCityState().getPopUpdateStatistic().hireSailors(inTavern);
                    ((IShip)ship).setNumberOfSailors(onShip + inTavern);
                } else {
                    IGroupableVessel group = (IGroupableVessel) ship;
                    for (IShip vessel : group.getShips()) {
                        onShip = vessel.getNumberOfSailors();
                        neededSailors = vessel.getMinNumberOfSailors();
                        nbToHire = Math.min(neededSailors - onShip, sailorState.getNumberOfSailors());
                        for (int i = 0; i < nbToHire; i++) {
                            sailorState.hire();
                        }
                        city.getCityState().getPopUpdateStatistic().hireSailors(nbToHire);
                        vessel.setNumberOfSailors(onShip + nbToHire);
                    }
                }
                // postpone task for tomorrow
                LocalDateTime currentDate = date.getCurrentDate();
                LocalDateTime tomorrow = currentDate.plusDays(1);
                // TimedTask definition here is ok, as HireSailorStep is serialized.
                TimedTask task = new TimedTask() {
                    {
                        setExecutionTime(tomorrow);
                    }
                    @Override
                    public void run() {
                        clientServerEventBus.post(new HireSailorEvent(ship, city));
                    }
                };
                taskList.add(task);
                return false;
            }
        }
    }

    @Override
    public String toString() {
        return "HireSailorsStep{" +
                ship.getName() + " in "
                + city.getName() + "needs "
                + shipService.getMinNumberOfSailors(ship) + " and has currently "
                + ship.getNumberOfSailors() +
                '}';
    }
}
