package ch.sahits.game.openpatrician.engine.sea;

import ch.sahits.game.openpatrician.clientserverinterface.model.PathInterpolatorMap;
import ch.sahits.game.openpatrician.clientserverinterface.model.VesselPositionUpdateData;
import ch.sahits.game.openpatrician.clientserverinterface.service.DialogTemplateFactory;
import ch.sahits.game.openpatrician.clientserverinterface.service.DialogTemplateParameterSupplier;
import ch.sahits.game.openpatrician.clientserverinterface.service.EDialogTemplateType;
import ch.sahits.game.openpatrician.clientserverinterface.service.MapService;
import ch.sahits.game.openpatrician.clientserverinterface.service.ShipService;
import ch.sahits.game.openpatrician.engine.EngineConfiguration;
import ch.sahits.game.openpatrician.engine.sea.model.ShipPositionUpdateTask;
import ch.sahits.game.openpatrician.model.DisplayTemplateMessage;
import ch.sahits.game.openpatrician.model.IHumanPlayer;
import ch.sahits.game.openpatrician.model.city.ICity;
import ch.sahits.game.openpatrician.model.event.TargetedEvent;
import ch.sahits.game.openpatrician.model.initialisation.StartNewGameBean;
import ch.sahits.game.openpatrician.model.map.IMap;
import ch.sahits.game.openpatrician.model.people.ICaptain;
import ch.sahits.game.openpatrician.model.ship.EShipType;
import ch.sahits.game.openpatrician.model.ship.INavigableVessel;
import ch.sahits.game.openpatrician.model.ui.DialogTemplate;
import ch.sahits.game.openpatrician.utilities.annotation.ClassCategory;
import ch.sahits.game.openpatrician.utilities.annotation.DependentInitialisation;
import ch.sahits.game.openpatrician.utilities.annotation.EClassCategory;
import com.google.common.base.Preconditions;
import com.google.common.eventbus.AsyncEventBus;
import javafx.geometry.Point2D;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.context.annotation.Lazy;
import org.springframework.stereotype.Component;

import javax.annotation.PostConstruct;
import java.util.List;
import java.util.Optional;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Future;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.TimeUnit;

/**
 * Service keeping track of all the ships that are traveling.
 * @author Andi Hotz, (c) Sahits GmbH, 2016
 *         Created on Jan 04, 2016
 */
@ClassCategory(EClassCategory.SINGLETON_BEAN)
@Component
@Lazy
@DependentInitialisation(StartNewGameBean.class)
public class SeafaringService {
    private final Logger logger = LogManager.getLogger(getClass());
    @Autowired
    private AStar aStar;
    @Autowired
    @Qualifier("clientThreadPool")
    private ExecutorService clientThreadPool;
    @Autowired
    private ShipPositionUpdateTask shipUpdateTask;
    @Autowired
    private ShipService shipService;
    @Autowired
    private IPathConverter pathConverter;
    @Autowired
    @Qualifier("schedulableServerThreadPool")
    private ScheduledExecutorService schedulableServerThreadPool;
    @Autowired
    private PathInterpolatorMap interpolators;
    @Autowired
    private IMap map;
    @Autowired
    private MapService mapService;
    @Autowired
    private DialogTemplateFactory dialogTemplateFactory;
    @Autowired
    @Qualifier("serverClientEventBus")
    protected AsyncEventBus clientServerEventBus;
    @PostConstruct
    private void initializeTimer() {
        schedulableServerThreadPool.scheduleAtFixedRate(shipUpdateTask, 0, EngineConfiguration.CLOCK_TICK_INTERVALL_MS, TimeUnit.MILLISECONDS);
    }
    /**
     * Send a ship or convoy on its way to a destination.
     * @param vessel that is sent
     * @param destination of the travel
     * @return list of points defining the path.
     */
    public List<Point2D> travelTo(INavigableVessel vessel, Point2D destination) {
        Preconditions.checkArgument(shipService.checkNumberOfSailors(vessel), "The vessel "+vessel.getName()+" has not enough sailors");
        EShipType type = shipService.getShipType(vessel);
        if (type == EShipType.HOLK || type == EShipType.COG) {
            Optional<ICity> optCity = map.findCity(destination);
            optCity.ifPresent(iCity -> {
                ICity city = iCity;
                Preconditions.checkArgument(!city.isRiverCity(), "City on a river is not reachable by ship type " + type);
            });
        }
        Point2D source = vessel.getLocation();
        Preconditions.checkArgument(!source.equals(destination), "Destination and source may not be the same");
        Future<List<Point2D>> future = clientThreadPool.submit(() -> aStar.findPath(source, destination));
        try {
            if (vessel.getCaptian().isPresent()) {
                ICaptain captain = vessel.getCaptian().get();
                double distance = source.distance(destination);
                int km = (int) Math.rint(mapService.convertToDistenceInKm(distance));
                captain.updatedSailedDistance(km);
                if (captain.upgradeToNextNavigationLevel() && vessel.getOwner() instanceof IHumanPlayer) {
                    DialogTemplateParameterSupplier parameterSupplier = new DialogTemplateParameterSupplier(new Object[]{captain.getName(), vessel.getName(), captain.getSalary()});
                    DialogTemplate template = dialogTemplateFactory.createDialogTemplate(EDialogTemplateType.CAPTAIN_UPGRADE, parameterSupplier);
                    DisplayTemplateMessage message = new DisplayTemplateMessage("ch.sahits.game.openpatrician.model.people.impl.CaptainState.captainUpgrade.title", template);
                    TargetedEvent tagetDisplayMsg = new TargetedEvent((IHumanPlayer)vessel.getOwner(), message);
                    clientServerEventBus.post(tagetDisplayMsg);
                }
            }
            return future.get();
        } catch (InterruptedException|ExecutionException e) {
            logger.warn("Failed to execute path finding in separate thread from "+ source+" to "+destination+" of "+vessel.getName()+" by "+vessel.getOwner().getName()+" "+vessel.getOwner().getLastName()+".", e);
            e.printStackTrace();
            return aStar.findPath(source, destination);
        }
    }


    /**
     * Travel from one city to another.
     * @param vessel which travels between cities
     * @param toCity destination city
     */
    public void travelBetweenCities(INavigableVessel vessel, ICity toCity) {
        List<Point2D> path = travelTo(vessel, toCity.getCoordinates());
        Preconditions.checkNotNull(path);
        pathConverter.createPath(vessel, path, 1);
        VesselPositionUpdateData vesselPositionUpdateData = interpolators.get(vessel);
        vesselPositionUpdateData.setDestinationCity(true);
        vesselPositionUpdateData.setSourceCity(true);
    }

    /**
     * Travel from a location that is not a city to a city.
     * @param vessel traveling to a city
     * @param toCity destination city
     */
    public void travelToCity(INavigableVessel vessel, ICity toCity) {
        List<Point2D> path = travelTo(vessel, toCity.getCoordinates());
        Preconditions.checkNotNull(path);
        pathConverter.createPath(vessel, path, 1);
        VesselPositionUpdateData vesselPositionUpdateData = interpolators.get(vessel);
        vesselPositionUpdateData.setDestinationCity(true);
        vesselPositionUpdateData.setSourceCity(false);
    }

    /**
     * Travel between two locations, where neither is a city.
     * @param vessel travelling between two locations at sea
     * @param location destination location
     */
    public void travelNotBetweenCities(INavigableVessel vessel, Point2D location) {
        List<Point2D> path = travelTo(vessel, location);
        Preconditions.checkNotNull(path);
        pathConverter.createPath(vessel, path, 1);
        VesselPositionUpdateData vesselPositionUpdateData = interpolators.get(vessel);
        vesselPositionUpdateData.setDestinationCity(false);
        vesselPositionUpdateData.setSourceCity(false);
    }

    /**
     * Travel from a city to a place that is not a city.
     * @param vessel traveling from a city to a location at sea
     * @param location destination location
     */
    public void travelFromCity(INavigableVessel vessel, Point2D location) {
        List<Point2D> path = travelTo(vessel, location);
        Preconditions.checkNotNull(path);
        pathConverter.createPath(vessel, path, 1);
        VesselPositionUpdateData vesselPositionUpdateData = interpolators.get(vessel);
        vesselPositionUpdateData.setDestinationCity(false);
        vesselPositionUpdateData.setSourceCity(true);
    }
}
