package ch.sahits.game.openpatrician.engine.event.task;

import ch.sahits.game.openpatrician.clientserverinterface.model.event.MarriageBrokerAnnouncementState;
import ch.sahits.game.openpatrician.clientserverinterface.model.event.MarriageOfferState;
import ch.sahits.game.openpatrician.clientserverinterface.model.factory.StateFactory;
import ch.sahits.game.openpatrician.clientserverinterface.model.task.CelebrationTask;
import ch.sahits.game.openpatrician.engine.player.strategy.CollectCelebrationWaresStrategy;
import ch.sahits.game.openpatrician.engine.player.tradesteps.TradeStepFactory;
import ch.sahits.game.openpatrician.model.Date;
import ch.sahits.game.openpatrician.model.IAIPlayer;
import ch.sahits.game.openpatrician.model.building.ITradingOffice;
import ch.sahits.game.openpatrician.model.city.ICity;
import ch.sahits.game.openpatrician.model.event.TimedTask;
import ch.sahits.game.openpatrician.model.event.TimedUpdatableTaskList;
import ch.sahits.game.openpatrician.model.personal.ISpouseData;
import ch.sahits.game.openpatrician.model.product.EWare;
import ch.sahits.game.openpatrician.model.product.ITradeStep;
import ch.sahits.game.openpatrician.model.product.IWare;
import ch.sahits.game.openpatrician.model.ship.INavigableVessel;
import ch.sahits.game.openpatrician.utilities.annotation.ClassCategory;
import ch.sahits.game.openpatrician.utilities.annotation.EClassCategory;
import ch.sahits.game.openpatrician.utilities.annotation.Prototype;
import com.google.common.base.Preconditions;
import com.thoughtworks.xstream.annotations.XStreamOmitField;
import javafx.geometry.Point2D;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.context.ApplicationContext;

import java.time.LocalDateTime;
import java.util.List;
import java.util.Optional;

import static java.util.Arrays.asList;

/**
 * Created by andi on 25.03.17.
 */
@Prototype
@ClassCategory({EClassCategory.SERIALIZABLE_BEAN, EClassCategory.PROTOTYPE_BEAN})
public class MarriageOfferTask extends TimedTask {
    @Autowired
    @XStreamOmitField
    private StateFactory stateFactory;
    @Autowired
    private TimedUpdatableTaskList taskList;
    @Autowired
    @XStreamOmitField
    private TradeStepFactory tradeStepFactory;
    @Autowired
    private Date date;
    @Autowired
    @XStreamOmitField
    private ServerSideTaskFactory taskFactory;
    @Autowired
    @XStreamOmitField
    private ApplicationContext context;

    private IAIPlayer player;
    private MarriageBrokerAnnouncementState state;
    private LocalDateTime deadLine;

    public MarriageOfferTask(IAIPlayer player, MarriageBrokerAnnouncementState state, LocalDateTime deadLine) {
        this.player = player;
        this.state = state;
        this.deadLine = deadLine;
        setExecutionTime(deadLine);
    }

    @Override
    public void run() {
        ISpouseData spouse = stateFactory.createSpouse(state, player);
        MarriageOfferState marriageOffer = MarriageOfferState.builder()
                .date(deadLine)
                .fromFirstName(state.getFromFirstName())
                .fromLastName(state.getFromLastName())
                .location(state.getLocation())
                .player(player)
                .tillDate(deadLine.plusDays(30))
                .spouse(spouse)
                .build();
        handleMarriageOffer(player, marriageOffer);
    }

    /**
     * Handle the offer of a marriage.
     * @param player who get's a marriage offer.
     */
    private void handleMarriageOffer(IAIPlayer player, MarriageOfferState offer) {
        ICity hometown = player.getHometown();
        Optional<ITradingOffice> optOffice = player.findTradingOffice(hometown);
        Preconditions.checkArgument(optOffice.isPresent(), "In the hometown a trading office is required");
        ITradingOffice office = optOffice.get();
        List<IWare> wares = asList(EWare.BEER, EWare.WINE, EWare.GRAIN, EWare.MEAT, EWare.FISH, EWare.HONEY);
        ITradeStep buy = tradeStepFactory.createAggregatedBuyTradingOfficeTradeStep(office, hometown, wares);
        buy.execute();
        INavigableVessel nearestVessel = player.getFleet().get(0);
        double minDistance = Double.MAX_VALUE;
        Point2D coord = hometown.getCoordinates();
        for (INavigableVessel vessel : player.getSelectableVessels()) {
            double distance = vessel.getLocation().distance(coord);
            if (distance < minDistance) {
                minDistance = distance;
                nearestVessel = vessel;
            }
        }
        // retrieve CollectCelebrationWaresStrategy
        CollectCelebrationWaresStrategy strategy = context.getBean(CollectCelebrationWaresStrategy.class);
        strategy.setCelebrationLocation(hometown);
        strategy.initializeTradeCycle(player, nearestVessel);
        player.updateTradeWaitingStatus(nearestVessel, false);
        // Add CelebrationTask (move the factory method to client server interface)
        LocalDateTime weddingDate = date.getCurrentDate().plusDays(21);
        CelebrationTask task = taskFactory.getMarriageCelebrationEvent(weddingDate, player, hometown, offer.getSpouse());
        taskList.add(task);
    }

    @Override
    public String toString() {
        return "MarriageOfferTask{" +
                "UUID=" + getUuid() +
                ", execution at " + getExecutionTime() +
                ", player=" + player.getUuid() + " " + player.getName() + " " + player.getLastName() +
                ", state=" + state +
                ", deadLine=" + deadLine +
                '}';
    }
}
