package ch.sahits.game.openpatrician.engine.event.task;

import ch.sahits.game.openpatrician.event.data.ShipEntersPortEvent;
import ch.sahits.game.openpatrician.model.city.ICity;
import ch.sahits.game.openpatrician.model.product.ISpecialMission;
import ch.sahits.game.openpatrician.model.ship.INavigableVessel;
import ch.sahits.game.openpatrician.utilities.annotation.ClassCategory;
import ch.sahits.game.openpatrician.utilities.annotation.EClassCategory;
import ch.sahits.game.openpatrician.utilities.annotation.Prototype;
import com.google.common.eventbus.AsyncEventBus;
import com.google.common.eventbus.Subscribe;
import com.thoughtworks.xstream.annotations.XStreamOmitField;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;

import javax.annotation.PostConstruct;

/**
 * Special mission where ship A waits on arrival of ship B in a city.
 * @author Andi Hotz, (c) Sahits GmbH, 2018
 * Created on Sep 23, 2018
 */
@Prototype
@ClassCategory({EClassCategory.MODEL, EClassCategory.SERIALIZABLE_BEAN, EClassCategory.PROTOTYPE_BEAN})
public class WaitForShipArrival implements ISpecialMission {

    private final INavigableVessel waitingVessel;
    private final INavigableVessel waitingOnVessel;
    private final ICity waitingIn;
    private final Runnable actionUponArrival;

    private boolean waitingShipArrived = false;
    private boolean waitedShipArrived = false;


    @Autowired
    @XStreamOmitField
    @Qualifier("serverClientEventBus")
    private AsyncEventBus clientServerEventBus;

    public WaitForShipArrival(INavigableVessel waitingVessel, INavigableVessel waitingOnVessel,
                              ICity waitingIn, Runnable actionUponArrival, boolean waitingShipArrived) {
        this.waitingVessel = waitingVessel;
        this.waitingOnVessel = waitingOnVessel;
        this.waitingIn = waitingIn;
        this.actionUponArrival = actionUponArrival;
        this.waitingShipArrived = waitingShipArrived;
    }

    @PostConstruct
    private void init() {
        clientServerEventBus.register(this);
    }

    @Subscribe
    public void handleShipArrivesInCity(ShipEntersPortEvent event) {
        if (waitingIn.equals(event.getCity())) {
            if (waitingVessel.equals(event.getShip())) {
                waitingShipArrived = true;
            }
            if (waitingOnVessel.equals(event.getShip())) {
                waitedShipArrived = true;
            }
        }
        if (waitingShipArrived && waitedShipArrived) {
            actionUponArrival.run();
            clientServerEventBus.unregister(this);
        }
    }
}
