package ch.sahits.game.openpatrician.engine.player.strategy;

import ch.sahits.game.openpatrician.clientserverinterface.service.GuildService;
import ch.sahits.game.openpatrician.model.IAIPlayer;
import ch.sahits.game.openpatrician.model.ICompany;
import ch.sahits.game.openpatrician.model.city.ICity;
import ch.sahits.game.openpatrician.model.city.guild.GuildList;
import ch.sahits.game.openpatrician.model.city.guild.IGuild;
import ch.sahits.game.openpatrician.model.player.IAIGuildJoinStrategy;
import ch.sahits.game.openpatrician.model.ship.INavigableVessel;
import ch.sahits.game.openpatrician.utilities.annotation.ClassCategory;
import ch.sahits.game.openpatrician.utilities.annotation.EClassCategory;
import ch.sahits.game.openpatrician.utilities.annotation.LazySingleton;
import org.springframework.beans.factory.annotation.Autowired;

import java.util.Optional;

/**
 * Base implementation for an AI guild join strategy.
 * @author Andi Hotz, (c) Sahits GmbH, 2017
 * Created on Oct 04, 2017
 */
@ClassCategory(EClassCategory.SINGLETON_BEAN)
@LazySingleton
public abstract class BaseJoinGuildStrategy implements IAIGuildJoinStrategy {
    private static final int ENTRY_FEE_FACTOR = 10;
    @Autowired
    private GuildList guildList;
    @Autowired
    private GuildService guildService;

    /**
     * {@inheritDoc}
     * Base requirements are: there has to be a guild, not yet a member and enough money to join.
     * @param player who should decide on joining
     * @param city where to join
     * @param vessel possible vessel in the city
     * @return
     */
    @Override
    public boolean shouldJoin(IAIPlayer player, ICity city, Optional<INavigableVessel> vessel) {
        Optional<IGuild> maybeGuild = guildList.findGuild(city);
        if (maybeGuild.isPresent()) {
            IGuild guild = maybeGuild.get();
            if (guild.getMembers().contains(player)) {
                return false;
            } else {
                ICompany company = player.getCompany();
                int fee = guildService.getEntryFee(player);
                return fee * ENTRY_FEE_FACTOR < company.getCash() && additionalJoinRequirementsMet(player, city, vessel);
            }
        }
        return false;
    }

    /**
     * Additional requirients added by custom strategies.
     */
    protected abstract boolean additionalJoinRequirementsMet(IAIPlayer player, ICity city, Optional<INavigableVessel> vessel);

    @Override
    public void join(IAIPlayer player, ICity city) {
        ICompany company = player.getCompany();
        Optional<IGuild> maybeGuild = guildList.findGuild(city);
        IGuild guild = maybeGuild.get();
        int fee = guildService.getEntryFee(player);
        company.updateCashDirectly(-fee);
        guild.join(player);
    }
}
