package ch.sahits.game.openpatrician.engine;

import ch.sahits.game.event.data.PeriodicalTimeWeekEndUpdate;
import ch.sahits.game.openpatrician.clientserverinterface.model.event.ConvoyDisolveEvent;
import ch.sahits.game.openpatrician.clientserverinterface.model.event.DonationRequestState;
import ch.sahits.game.openpatrician.clientserverinterface.model.event.MarriageBrokerAnnouncementState;
import ch.sahits.game.openpatrician.clientserverinterface.model.event.MarriageFeastRequestState;
import ch.sahits.game.openpatrician.clientserverinterface.model.event.OrlegShipOutfittedEvent;
import ch.sahits.game.openpatrician.clientserverinterface.service.ConvoyService;
import ch.sahits.game.openpatrician.clientserverinterface.service.MapProxy;
import ch.sahits.game.openpatrician.clientserverinterface.service.MapService;
import ch.sahits.game.openpatrician.clientserverinterface.service.ShipService;
import ch.sahits.game.openpatrician.clientserverinterface.service.StrategyHolderService;
import ch.sahits.game.openpatrician.engine.event.AIPlayerEngineState;
import ch.sahits.game.openpatrician.engine.event.EEventState;
import ch.sahits.game.openpatrician.engine.event.EventEngineState;
import ch.sahits.game.openpatrician.engine.event.task.MarriageOfferTask;
import ch.sahits.game.openpatrician.engine.event.task.ServerSideTaskFactory;
import ch.sahits.game.openpatrician.engine.land.city.ShipyardEngine;
import ch.sahits.game.openpatrician.engine.player.AITradeChecker;
import ch.sahits.game.openpatrician.engine.player.CollectConstructionWaresMissionData;
import ch.sahits.game.openpatrician.engine.player.IAIEventHandler;
import ch.sahits.game.openpatrician.engine.player.SupplyCityMissionData;
import ch.sahits.game.openpatrician.engine.player.strategy.BlockadeStrategy;
import ch.sahits.game.openpatrician.engine.player.strategy.CollectCelebrationWaresStrategy;
import ch.sahits.game.openpatrician.engine.player.strategy.FormOrJoinConvoyStrategy;
import ch.sahits.game.openpatrician.engine.sea.SeafaringService;
import ch.sahits.game.openpatrician.engine.sea.model.ConvoyRepairMission;
import ch.sahits.game.openpatrician.event.EGameStatusChange;
import ch.sahits.game.openpatrician.event.GameStateChange;
import ch.sahits.game.openpatrician.event.data.BuildingAuctionFinished;
import ch.sahits.game.openpatrician.event.data.BuildingNotAuctioned;
import ch.sahits.game.openpatrician.event.data.GraphInitialisationComplete;
import ch.sahits.game.openpatrician.event.data.ShipAuctionFinished;
import ch.sahits.game.openpatrician.event.data.ShipEntersPortEvent;
import ch.sahits.game.openpatrician.event.data.ShipNotAuctioned;
import ch.sahits.game.openpatrician.event.data.ShipyardOrderBuild;
import ch.sahits.game.openpatrician.event.data.ai.BlockadeShipRequestEvent;
import ch.sahits.game.openpatrician.event.data.ai.SpecialMissionFinishedEvent;
import ch.sahits.game.openpatrician.model.AIPlayerList;
import ch.sahits.game.openpatrician.model.Date;
import ch.sahits.game.openpatrician.model.IAIPlayer;
import ch.sahits.game.openpatrician.model.IPlayer;
import ch.sahits.game.openpatrician.model.city.ICity;
import ch.sahits.game.openpatrician.model.city.IShipyard;
import ch.sahits.game.openpatrician.model.event.TimedUpdatableTaskList;
import ch.sahits.game.openpatrician.model.people.IShipOwner;
import ch.sahits.game.openpatrician.model.personal.ESocialRank;
import ch.sahits.game.openpatrician.model.player.AIPlayerContext;
import ch.sahits.game.openpatrician.model.player.ETradeStrategyPreference;
import ch.sahits.game.openpatrician.model.player.IAIConstructionSelectionStrategy;
import ch.sahits.game.openpatrician.model.player.IAIEventDecisionStrategy;
import ch.sahits.game.openpatrician.model.player.IAITradeStrategy;
import ch.sahits.game.openpatrician.model.player.IAITradeStrategyType;
import ch.sahits.game.openpatrician.model.player.IProductionConsumptionKnowledge;
import ch.sahits.game.openpatrician.model.product.IBlockadeMission;
import ch.sahits.game.openpatrician.model.product.ISpecialMission;
import ch.sahits.game.openpatrician.model.product.ITradeMissionData;
import ch.sahits.game.openpatrician.model.product.ITradeStep;
import ch.sahits.game.openpatrician.model.sea.BlockadeState;
import ch.sahits.game.openpatrician.model.sea.IBlockade;
import ch.sahits.game.openpatrician.model.sea.TravellingVessels;
import ch.sahits.game.openpatrician.model.ship.EShipType;
import ch.sahits.game.openpatrician.model.ship.ICog;
import ch.sahits.game.openpatrician.model.ship.IConvoy;
import ch.sahits.game.openpatrician.model.ship.ICrayer;
import ch.sahits.game.openpatrician.model.ship.IHolk;
import ch.sahits.game.openpatrician.model.ship.INavigableVessel;
import ch.sahits.game.openpatrician.model.ship.IShip;
import ch.sahits.game.openpatrician.model.ship.ISnaikka;
import ch.sahits.game.openpatrician.utilities.annotation.ClassCategory;
import ch.sahits.game.openpatrician.utilities.annotation.EClassCategory;
import ch.sahits.game.openpatrician.utilities.annotation.LazySingleton;
import ch.sahits.game.openpatrician.utilities.annotation.ListType;
import ch.sahits.game.openpatrician.utilities.service.DisableProperties;
import com.google.common.annotations.VisibleForTesting;
import com.google.common.eventbus.AsyncEventBus;
import com.google.common.eventbus.Subscribe;
import javafx.geometry.Point2D;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.context.ApplicationContext;

import javax.annotation.PostConstruct;
import java.time.LocalDateTime;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Map.Entry;
import java.util.Optional;
import java.util.Queue;
import java.util.Random;
import java.util.concurrent.ConcurrentLinkedQueue;
import java.util.stream.Collectors;

import static ch.sahits.game.openpatrician.engine.player.strategy.EAITradeStrategyType.COLLECT_WARES_FOR_STORAGE_LOCATION;
import static ch.sahits.game.openpatrician.engine.player.strategy.EAITradeStrategyType.DELIVER_WARES_FROM_CENTRAL_STORAGE;
import static ch.sahits.game.openpatrician.engine.player.strategy.EAITradeStrategyType.PRODUCTION_CHAIN;
import static ch.sahits.game.openpatrician.engine.player.strategy.EAITradeStrategyType.SUPPLY_HOMETOWN;
import static ch.sahits.game.openpatrician.engine.player.strategy.EAITradeStrategyType.TRADE_ROUTE;
import static java.util.Collections.emptyList;

/**
 * This implements the engine for an AI Player
 * @author Andi Hotz, (c) Sahits GmbH, 2014
 *         Created on Jan 23, 2014
 */
@Slf4j(topic = "EVENTS")
@LazySingleton
@ClassCategory(EClassCategory.SINGLETON_BEAN)
public class AIPlayerEngine extends AbstractEngine implements IAIEventHandler {
    @Autowired
    private AIPlayerList players;
    @Autowired
    @Qualifier("serverClientEventBus")
    private AsyncEventBus clientServerEventBus;
    @Autowired
    @Qualifier("timerEventBus")
    private AsyncEventBus timerEventBus;
    @Autowired
    private BlockadeState blockadeState;
    @Autowired
    private BlockadeStrategy blockadeStrategy;
    @Autowired
    private SeafaringService seafaringService;
    @Autowired
    private ShipService shipService;
    @Autowired
    private Date date;
    @Autowired
    private MapService mapService;
    @Autowired
    private TravellingVessels vessels;
    @Autowired
    private AITradeChecker tradeChecker;
    @Autowired
    private ShipyardEngine shipyardEngine;
    @Autowired
    private TimedUpdatableTaskList taskList;
    @Autowired
    private ServerSideTaskFactory taskFactory;
    @Autowired
    private Random rnd;
    @Autowired
    private StrategyHolderService strategyHolderService;
    @Autowired
    private DisableProperties disableService;
    @Autowired
    private EventEngineState eventState;
    @Autowired
    private AIPlayerEngineState engineState;
    @Autowired
    private MapProxy map;
    @Autowired
    private FormOrJoinConvoyStrategy convoyStrategy;
    @Autowired
    private ConvoyService convoyService;
    @Autowired
    private ApplicationContext context;

    private EGameStatusChange currentGameState = EGameStatusChange.NEW_GAME;

    /** Queue for ship enters port events. If the list is null, this means the events can be handled directly. */
    @ListType(ShipEntersPortEvent.class)
    private Queue<ShipEntersPortEvent> shipEntersPortEventQueue = new ConcurrentLinkedQueue<>();

    @Override
    public List<AbstractEngine> getChildren() {
        return emptyList();
    }
    @PostConstruct
    private void init() {
        clientServerEventBus.register(this);
        timerEventBus.register(this);
    }

    private void initializeStrategyForVessel(IAIPlayer player, INavigableVessel ship) {
        if (initializeConvoy(player, ship)) {
            initializeStrategyInternal(player, ship);
        }
    }

    private void initializeStrategyInternal(IAIPlayer player, INavigableVessel ship) {
        IAITradeStrategy tradeStrategy = player.getTradeStrategyType(ship).getStrategy();
        tradeStrategy.initializeTradeCycle(player, ship);
        player.updateTradeWaitingStatus(ship, false);
        initializeWeightedTradeStrategies(player);
    }

    private boolean initializeConvoy(IAIPlayer player, INavigableVessel ship) {
        if (!(ship instanceof IConvoy)) {
            if (convoyStrategy.shouldJoinOrFormConvoy(player)) {
                Optional<ICity> currentCity = shipService.findCity(ship);
                if (currentCity.isPresent()) {
                    if (convoyStrategy.mustBeFittableForOrleg(ship)) {
                        Optional<IShip> orleg = convoyService.outfitAsOrleg((IShip) ship, currentCity.get());
                        if (!orleg.isPresent()) {
                            // postpone any further steps.
                            log.info("Ship {} ({}) could not be fitted for orleg duties on the spot.", 
                                    ship.getName(), ship.getUuid());
                            return false;
                        }
                    }
                    Optional<IConvoy> convoy = convoyStrategy.joinOrFormConvoy((IShip) ship, currentCity.get());
                    if (convoy.isPresent() && player.getTradeStrategyType(convoy.get()) == null) {
                        initializeRandomTradeStrategy(convoy.get(), player);
                    }
                } else {
                    log.warn("The ship {} ({}) of {} {} is not in a city when initializing the strategy",
                            ship.getName(), ship.getUuid(), player.getName(), player.getLastName());
                }
            }
        }
        return true;
    }

    /**
     * Reset the weighted trade strategies in the context.
     */
    @VisibleForTesting
    void initializeWeightedTradeStrategies(IAIPlayer player) {
        AIPlayerContext context = player.getPlayerContext();
        context.getWeightedStrategies().clear();
        ESocialRank rank = player.getRank();
        if (rank == ESocialRank.COUNCILMAN || rank == ESocialRank.PATRICIAN) {
            context.addWeightedTradeStrategy(SUPPLY_HOMETOWN);
        }
        if (rank == ESocialRank.MAYOR) {
            context.addWeightedTradeStrategy(COLLECT_WARES_FOR_STORAGE_LOCATION);
            context.addWeightedTradeStrategy(DELIVER_WARES_FROM_CENTRAL_STORAGE);
        }
        if (player.getCompany().getCash() < 100000L) {
            context.addWeightedTradeStrategy(TRADE_ROUTE);
            context.addWeightedTradeStrategy(PRODUCTION_CHAIN);
        }
    }
    @VisibleForTesting
    List<IAITradeStrategyType> getSelectableTradeStrategy(INavigableVessel vessel, IAIPlayer player) {
        // Consider also currently running strategies
        Map<IAITradeStrategyType, Integer> weigthedStrategies = new HashMap<>();
        strategyHolderService.getTradeStrategies().stream().forEach(strategyType -> weigthedStrategies.put(strategyType, 1));

        for (Entry<IAITradeStrategyType, Integer> entry : player.getPlayerContext().getWeightedStrategies().entrySet()) {
            IAITradeStrategyType key = entry.getKey();
            if (key.equals(SUPPLY_HOMETOWN) && !hasSupplyHomeTownStrategy(player)) {
                for (int i = 0; i < entry.getValue(); i++) {
                    weigthedStrategies.put(key, weigthedStrategies.get(key) + 1);
                }
            }
            if (key.equals(COLLECT_WARES_FOR_STORAGE_LOCATION) && !hasFreeCentralStorageCollectionStrategy(player)) {
                for (int i = 0; i < entry.getValue(); i++) {
                    weigthedStrategies.put(key, weigthedStrategies.get(key) + 1);
                }
            }
            if (key.equals(DELIVER_WARES_FROM_CENTRAL_STORAGE) || key.equals(TRADE_ROUTE) || key.equals(PRODUCTION_CHAIN)) {
                for (int i = 0; i < entry.getValue(); i++) {
                    weigthedStrategies.put(key, weigthedStrategies.get(key) + 1);
                }
            }
        }

        for (IAITradeStrategyType type : strategyHolderService.getTradeStrategies()) {
            for (ETradeStrategyPreference preference : type.getStrategyPreferrance()) {
                if (isStrategyMatch(vessel, preference)) {
                    weigthedStrategies.put(type, weigthedStrategies.get(type) + 1);
                }
            }
        }

        List<IAITradeStrategyType> tradeStrategyTypes = new ArrayList<>();
        for (Entry<IAITradeStrategyType, Integer> entry : weigthedStrategies.entrySet()) {
            int number = entry.getValue();
            for (int i = 0; i < number; i++) {
                tradeStrategyTypes.add(entry.getKey());
            }
        }
        return tradeStrategyTypes.stream()
                .filter(type -> type.getStrategy().isSelectable(player, vessel))
                .collect(Collectors.toList());
    }

    private boolean isStrategyMatch(INavigableVessel vessel, ETradeStrategyPreference preference) {
        switch (preference) {
            case COG:
                return vessel instanceof ICog;
            case CONVOY:
                return vessel instanceof IConvoy;
            case CRAYER:
                return vessel instanceof ICrayer;
            case FAST_SHIP:
                return vessel.getTopSpeed() > 9;
            case HEAVY_ARMED:
                return shipService.calculateShipsWeaponsStrength(vessel) > 5;
            case HOLK:
                return vessel instanceof IHolk;
            case MANY_SAILORS:
                return vessel.getNumberOfSailors() > vessel.getMinNumberOfSailors() * 2;
            case RIVER_SHIP:
                return vessel instanceof ICrayer || vessel instanceof ISnaikka;
            case SNAIKKA:
                return vessel instanceof ISnaikka;
            default:
                throw new IllegalArgumentException("Unknown preferrence: "+preference);
        }
    }
    @VisibleForTesting
    boolean hasSupplyHomeTownStrategy(IAIPlayer player) {
        List<INavigableVessel> vessels = player.getSelectableVessels();
        for (INavigableVessel v : vessels) {
            if (SUPPLY_HOMETOWN.equals(player.getTradeStrategyType(v))) {
                  return true;
            }
        }
        return false;
    }

    @VisibleForTesting
    boolean hasFreeCentralStorageCollectionStrategy(IAIPlayer player) {
        List<INavigableVessel> vessels = player.getSelectableVessels();
        List<ICity> citiesWithTradingOffice = map.getAllCities().stream()
                .filter(city -> player.findTradingOffice(city).isPresent())
                .collect(Collectors.toList());
        for (INavigableVessel vessel : vessels) {
            if (COLLECT_WARES_FOR_STORAGE_LOCATION.equals(player.getTradeStrategyType(vessel))) {
                SupplyCityMissionData missionData = (SupplyCityMissionData) player.getTradeMission(vessel);
                citiesWithTradingOffice.remove(missionData.getTargetCity());
            }
        }
        return !citiesWithTradingOffice.isEmpty();
    }

    @Subscribe
    public void handleBlockadeStart(BlockadeShipRequestEvent event) {
        IAIPlayer player = event.getPlayer();
        int requiredShips = event.getNumberOfShips();
        for (int i = 0; i < requiredShips; i++) {
            Optional<IShip> optShip = blockadeStrategy.selectShip(player);
            if (optShip.isPresent()) {
                IShip vessel = optShip.get();
                player.setTradeMission(vessel, new IBlockadeMission() {});
                seafaringService.travelTo(vessel, event.getAssemblyCity().getCoordinates());
            } else {
                break;
            }
        }
    }
    @Subscribe
    public void handleShipArrivesInCity(ShipEntersPortEvent event) {
        if (shipEntersPortEventQueue != null && !shipEntersPortEventQueue.isEmpty()) {
            shipEntersPortEventQueue.add(event);
            log.info("Add ShipEntersPortEvent to queue.");
        } else {
            handleShipArrival(event);
        }
    }

    /**
     * Internal handling of the ship arrival.
     * @param event ship enters port
     */
    private void handleShipArrival(ShipEntersPortEvent event) {
        final INavigableVessel vessel = event.getShip();
        final ICity city = event.getCity();
        if (!vessel.getLocation().equals(city.getCoordinates())) {
            log.warn("The vessel is not in the city, but is moving, this can happen when a new destination is chosen after the ShipPositionUpdate task determined the ship entered the port, but the event was not yet handled.");
            return;
        }
        IShipOwner owner = vessel.getOwner();
        log.debug("Current position of vessel ({} ({})): {}", vessel.getName(), vessel.getUuid(), vessel.getLocation());
        vessels.remove(vessel);
        if (owner instanceof IAIPlayer) {
            IAIPlayer player = (IAIPlayer) owner;
            IProductionConsumptionKnowledge knowlege = player.getProductionAndConsumptionKnowledge();
            knowlege.updateKnowledge(event.getCity());
            ITradeMissionData missionData = player.getTradeMission(vessel);
            if (missionData instanceof ISpecialMission) {
                if (missionData instanceof IBlockadeMission) {
                    handleBlockadeMission(vessel, player, event.getCity());
                } else {
                    throw new IllegalStateException("The special mission "+missionData+" is not implemented");
                }
            } else {
                player.updateTradeWaitingStatus(vessel, false);
                try {
                    IAITradeStrategy strategy = player.getTradeStrategyType(vessel).getStrategy();
                    engineState.activateVesselForTrade(vessel);
                    strategy.handleShipArrivesInPort(event);
                } catch (RuntimeException e) {
                    log.error("Failed to handle ship arrival of "+vessel.getName()+" ("+vessel.getUuid()+") of "+player.getName()+" "+player.getLastName(), e);
                } finally {
                    engineState.deactivateVesselForTrade(vessel);
                }
            }
        }
    }

    /**
     * Listen to the start of the game loading to continue with the handling of ship arrival events.
     * @param gameStateChange event for game state change
     */
    @Subscribe
    public void handleGameLoad(GameStateChange gameStateChange) {
        currentGameState = gameStateChange.getStatusChange();
        log.info("Changed Game status to {}", currentGameState);
        if (gameStateChange.getStatusChange() == EGameStatusChange.LOAD_GAME_INIT) { // FIXME: 4/8/18 this is only reached if the same event is first handled by GameStateHandler creating this instance
            shipEntersPortEventQueue = new ConcurrentLinkedQueue<>();
            log.info("Loading of a game was initialized. Queuing further ShipEntersPortEvent.");
        }
    }
    @Subscribe
    public void handleTradeStrategyStart(GraphInitialisationComplete event) {
        if (shipEntersPortEventQueue != null) {
            log.info("Handle {} queued ShipEntersPortEvent after graph initialisation", shipEntersPortEventQueue.size());
            while (!shipEntersPortEventQueue.isEmpty()) {
                ShipEntersPortEvent shipEntersPortEvent = shipEntersPortEventQueue.poll();
                handleShipArrival(shipEntersPortEvent);
            }
            shipEntersPortEventQueue = null;
        }
        // When loading a game from in game flag must be set to false, if the map is a different one.
        if (disableService.aiTradingEnabled() && currentGameState != EGameStatusChange.GAME_LOADED) { // FIXME: 8/15/18 this is not required when the game was loaded.
            for (IAIPlayer player : players) {
                ICity hometown = player.getHometown();
                IProductionConsumptionKnowledge knowlege = player.getProductionAndConsumptionKnowledge();
                knowlege.updateKnowledge(hometown);
                List<IShip> shipList = player.getFleet();
                for (IShip ship : shipList) {
                    try {
                        // Only initialize if there are no trade steps (not after loading)
                        if (!player.isInitialized(ship)) {
                            initializeStrategyForVessel(player, ship);
                        }
                    } catch (RuntimeException e) {
                        log.error("Failed to initialize strategy for ship  " + ship.getName() + " ("+ship.getUuid()+") of " + player.getName() + " " + player.getLastName(), e);
                    }
                }

                List<INavigableVessel> vesselList = player.getSelectableVessels();
                for (INavigableVessel ship : vesselList) {
                    startTradeStrategySteps(player, ship);
                }
            }
        }
    }

    private void startTradeStrategySteps(IAIPlayer player, INavigableVessel ship) {
        IAITradeStrategy tradeStrategy = player.getTradeStrategyType(ship).getStrategy();
        // If the vessel is traveling do not execute
        if (!player.hasMoreTradeSteps(ship)) {
            if (shipService.findCity(ship).isPresent()) {
                tradeStrategy.initializeTradeCycle(player, ship);
                log.info("No more tradesteps for vessel {} ({}) of {} {} defined, but located in city: Reinitialize trade strategy", ship.getName(), ship.getUuid(), player.getName(), player.getLastName());
            } else {
                if (!vessels.isTravelling(ship)) { // already traveling to destination
                    ICity nearestCity = tradeStrategy.getCityToRestartTradeCycle(ship);
                    seafaringService.travelToCity(ship, nearestCity);
                    log.warn("No tradesteps defined for vessel {} ({}) of {} {}. Travel to nearest city {}", ship.getName(), ship.getUuid(), player.getName(), player.getLastName(), nearestCity.getName());
                }
            }
        }
        if (!player.waitingForTradeStepToFinish(ship)) {
            tradeStrategy.executeTradeSteps(player, ship);
        }  else {
            log.warn("Skip starting of trade execution as waitingForTradeStepToFinish for {} ({}) of {} {} is {}", ship.getName(), ship.getUuid(), player.getName(), player.getLastName(), player.waitingForTradeStepToFinish(ship));
        }
    }

    private void handleBlockadeMission(INavigableVessel vessel, IAIPlayer player, ICity city) {
        LocalDateTime now = date.getCurrentDate();
        ICity aldermanCity = mapService.getAldermanCity();
        if (city.equals(aldermanCity)) {
            if (blockadeState.entrySet().isEmpty()) {
                initializeStrategyForVessel(player, vessel);
            } else {
                for (Entry<ICity, IBlockade> entry : blockadeState.entrySet()) {
                    IBlockade blockade = entry.getValue();
                    if (blockade.getAssemblyDate().isBefore(now)) {
                        // Missed assemby date
                        initializeStrategyForVessel(player, vessel);
                    }
                }
            }
        } else {
            initializeStrategyForVessel(player, vessel);
        }
    }

    /**
     * Check weather the current vessel is engaged in executing trade steps.
     * @param vessel for which to check if it is active
     * @return true if the vessel is active for trading
     */
    public boolean executeCurrentTrade(INavigableVessel vessel) {
        return engineState.isActive(vessel);
    }

    @Subscribe
    public void handleWeeklyTasks(PeriodicalTimeWeekEndUpdate event) {
        if (disableService.aiTradingEnabled()) {
            for (IAIPlayer player : players) {
                List<INavigableVessel> shipList = player.getSelectableVessels();
                for (INavigableVessel vessel : shipList) {
                    if (!tradeChecker.isEngagedInTrading(vessel) && tradeChecker.shouldBeTrading(vessel)) {
                        IAIPlayer owner = (IAIPlayer) vessel.getOwner();
                        log.warn("The vessel {} ({}) of {} {} is currently not trading (position: {}), strategy={}: reinitialize", vessel.getName(), vessel.getUuid(), owner.getName(), owner.getLastName(), vessel.getLocation(), owner.getTradeStrategyType(vessel));
                        clearRemainingTradeSteps(vessel, player);
                        initializeStrategyForVessel(player, vessel);
                        // Execute
                        IAITradeStrategy tradeStrategy = player.getTradeStrategyType(vessel).getStrategy();
                        tradeStrategy.executeTradeSteps(player, vessel);
                    }
                }
                // Check Ship construction
                IAIConstructionSelectionStrategy constructionStategy = player.getConstructionSelectionType().getStrategy();
                if (constructionStategy.shouldOrderNewConstruction(player)) {
                    EShipType type = constructionStategy.shouldBuildShipType(player);
                    Optional<INavigableVessel> vessel = constructionStategy.selectCollectingVessel(player, type);
                    vessel.ifPresent(iNavigableVessel ->
                            constructionStategy.initShipConstruction(player, iNavigableVessel, type));
                }
                initializeWeightedTradeStrategies(player);
            }
        }
    }

    /**
     * Remove any remaining trade steps.
     * @param vessel for which the trade steps should be checked
     * @param player for which the trade steps should be checked.
     */
    private void clearRemainingTradeSteps(INavigableVessel vessel, IAIPlayer player) {
        while(player.hasMoreTradeSteps(vessel)) {
            player.getNextTradeStep(vessel);
        }
        player.setTradeMission(vessel, null);
    }

    /**
     * Handle the reinitialisation after finishing a special mission.
     * @param event special mission is finished
     */
    @Subscribe
    public void handleSpecialMissionFinished(SpecialMissionFinishedEvent event) {
        INavigableVessel vessel = event.getVessel();
        IAIPlayer player = (IAIPlayer) vessel.getOwner();
        ICity city = event.getCity();
        log.debug("Special mission for ship {} ({}) of {} {} ended in {}",vessel.getName(), vessel.getUuid(), player.getName(), player.getLastName(), city.getName());
        ITradeMissionData missionData = event.getMissionData();
        if (missionData instanceof CollectConstructionWaresMissionData) {
            IShipyard shipyard = city.getCityState().getShipyardState();
            ShipyardOrderBuild shipConstructionOrder = new ShipyardOrderBuild(shipyard, ((CollectConstructionWaresMissionData) missionData).getShipType(), player);
            shipyardEngine.handleOrderBuildEvent(shipConstructionOrder);
        }
        if (!player.hasMoreTradeSteps(vessel)) {
            initializeRandomTradeStrategy(vessel, player);
            initializeStrategyForVessel(player, vessel);
            // Execute
            IAITradeStrategy tradeStrategy = player.getTradeStrategyType(vessel).getStrategy();
            tradeStrategy.executeTradeSteps(player, vessel);
        }
    }

    private void initializeRandomTradeStrategy(INavigableVessel vessel, IAIPlayer player) {
        List<IAITradeStrategyType> tradeStrategies = getSelectableTradeStrategy(vessel, player);
        player.setTradeStrategyType(vessel, tradeStrategies.get(rnd.nextInt(tradeStrategies.size())));
    }

    @Subscribe
    public void handleShipAuctionFinished(ShipAuctionFinished event) {
        IPlayer owner = event.getNewOwner();
        if (owner instanceof IAIPlayer) {
            IShip ship = event.getShip();
            ship.setOwner(owner);
            owner.addShip(ship);
            initializeRandomTradeStrategy(ship, (IAIPlayer) owner);
            initializeStrategyForVessel((IAIPlayer) owner, ship);
            startTradeStrategySteps((IAIPlayer) owner, ship);
        }
    }
    @Subscribe
    public void handleShipNotAuction(ShipNotAuctioned event) {
        IShip ship = event.getShip();
        IPlayer owner = (IPlayer) ship.getOwner();
        if (owner instanceof IAIPlayer) {
            owner.addShip(ship);
            initializeRandomTradeStrategy(ship, (IAIPlayer) owner);
            initializeStrategyForVessel((IAIPlayer) owner, ship);
            startTradeStrategySteps((IAIPlayer) owner, ship);
        }
    }
    @Subscribe
    public void handleBuildingAuctionFinished(BuildingAuctionFinished event) {
        if (event.getNewOwner() instanceof IAIPlayer) {
            event.getBuilding().setOwner(event.getNewOwner());
        }
    }
    @Subscribe
    public void handleBuildingNotAuction(BuildingNotAuctioned event) {
        // no action
    }

    @Override
    public void handleMarriageEvent(IAIPlayer player, MarriageBrokerAnnouncementState state) {
        IAIEventDecisionStrategy strategy = player.getEventDecitionStrategyType().getStrategy();
        if (!player.getSpouseData().isPresent()  && strategy.acceptMarriagBrokereOffer(player)) {
            int delay = rnd.nextInt(50) + 24;
            LocalDateTime deadLine = state.getDate().plusDays(delay);
            MarriageOfferTask task = taskFactory.getMarriageOfferTask(player, state, deadLine);
            log.debug("Add marriage offer task for {} {}", player.getName(), player.getLastName());
            taskList.add(task);
            eventState.setMarriageState(player, EEventState.MARRIAGE_UNDER_CONSIDERATION);
        }
    }


    @Override
    public void handleMarriageWareDelivery(IAIPlayer player, MarriageFeastRequestState state) {
         boolean accept = rnd.nextBoolean() && player.getFleet().size() > 5;
         if (accept) {
             INavigableVessel nearestVessel = player.getFleet().get(0);
             double minDistance = Double.MAX_VALUE;
             Point2D coord = state.getLocation().getCoordinates();
             for (INavigableVessel vessel : player.getSelectableVessels()) {
                 double distance = vessel.getLocation().distance(coord);
                 if (distance < minDistance) {
                     minDistance = distance;
                     nearestVessel = vessel;
                 }
             }
             CollectCelebrationWaresStrategy strategy = context.getBean(CollectCelebrationWaresStrategy.class);
             strategy.setCelebrationLocation(state.getLocation());
             strategy.initializeTradeCycle(player, nearestVessel);
             player.updateTradeWaitingStatus(nearestVessel, false);
         }
    }

    @Override
    public void handleDonationRequest(IAIPlayer player, DonationRequestState state) {
        boolean accept = rnd.nextBoolean() && player.getCompany().getCash() > state.getAmount() * 30;
        if (accept) {
            player.getCompany().updateCashDirectly(-state.getAmount());
        }
    }

    /**
     * Method to handle the finishing of a ship construction.
     * @param ship that is newly constructed
     * @param player owner of the ship
     */
    public void handleShipConstructionFinished(IShip ship, IAIPlayer player) {
        player.addShip(ship);
        initializeNewShip(ship, player);
    }

    /**
     * Method to handle the initializing of a trade strategy for a new ship.
     * @param ship to be initialized
     * @param player owner of the ship
     */
    @Override
    public void initializeNewShip(IShip ship, IAIPlayer player) {
        initializeRandomTradeStrategy(ship, player);
        initializeStrategyForVessel(player, ship);
        startTradeStrategySteps(player, ship);
    }

    /**
     * Initialize the visible ships.
     * @param event game state change event
     */
    @Subscribe
    public void handelLoadedNewGame(GameStateChange event) {
        if (event.getStatusChange() == EGameStatusChange.GAME_LOADED) {
            for (IAIPlayer player : players) {
                for (IShip ship : player.getFleet()) {
                    StringBuilder sb = new StringBuilder();
                    sb.append("Trade steps for ship ").append(ship.getName())
                            .append(" (").append(ship.getUuid()).append(") of ")
                            .append(player.getName()).append(" ").append(player.getLastName()).append(" after loading:\n");
                    for (ITradeStep tradeStep : player.getTradeSteps(ship)) {
                        sb.append(tradeStep).append(" ");
                    }
                    log.info(sb.toString());
                }
            }
        }
    }
    @Subscribe
    public void handleConvoyDissolve(ConvoyDisolveEvent event) {
        IConvoy convoy = event.getConvoy();
        if (convoy.getOwner() instanceof IAIPlayer) {
            IAIPlayer player = (IAIPlayer) convoy.getOwner();
            IShip orleg = convoy.getOrlegShip();
            List<IShip> members = convoy.getShips().stream()
                    .filter(ship -> !ship.equals(orleg) && ship.getOwner().equals(player))
                    .collect(Collectors.toList());
            boolean publicConvoy = convoy.isPublicConvoy();
            ConvoyRepairMission tradeMission = new ConvoyRepairMission(orleg, members, publicConvoy); // register
            player.setTradeMission(orleg, tradeMission);
        }
    }

    /**
     * When a ship has finally been fitted for orlead duties, create the convoy.
     * @param event
     */
    @Subscribe
    public void handleShipFittedForOrleg(OrlegShipOutfittedEvent event) {
        IAIPlayer player = (IAIPlayer) event.getShip().getOwner();
        initializeStrategyForVessel(player, event.getShip());
    }

    // TODO: andi 5/22/16 When ship is captured or destroyed

}
